// (c) Microsoft Corporation 2005-2007.  

#light

namespace Microsoft.FSharp.Collections

open Microsoft.FSharp.Core
open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
open Microsoft.FSharp.Core.Operators
open Microsoft.FSharp.Collections
open Microsoft.FSharp.Primitives.Basics
open System.Collections

#if CLI_AT_MOST_1_1
open Microsoft.FSharp.Compatibility
#else
open System.Collections.Generic
#endif
type seq<'a> = IEnumerable<'a>
      

module IEnumerator = 

  let not_started() = raise (new System.InvalidOperationException("Enumeration has not started. Call MoveNext."))
  let already_finished() = raise (new System.InvalidOperationException("Enumeration already finished."))
  let check started = if not started then not_started()
  let dispose (r : #System.IDisposable) = r.Dispose()

  let untyped_iter (f : 'a -> unit) (e :> IEnumerator)  = 
    while e.MoveNext() do
      f (unbox e.Current);

  let untyped_fold f x (e :> IEnumerator) = 
    let mutable state = x 
    while e.MoveNext() do
      state <- f state (unbox e.Current);
    state

  let untyped_map f (e :> IEnumerator)  = 
    { new IEnumerator 
      with Reset() = e.Reset()
      and  get_Current() = box (f (unbox e.Current))
      and  MoveNext() = e.MoveNext() }
 
  let untyped_filter f (e :> IEnumerator) = 
    let started = ref false 
    let this = 
        { new IEnumerator
            with get_Current() = 
                check !started; 
                unbox e.Current
            and  MoveNext() = 
                let rec next() = 
                    if not !started then started := true;
                    e.MoveNext() && (f  (unbox e.Current) || next()) 
                in next()
            and  Reset() = 
                started := false;
                e.Reset();  } 
    this

  let untyped_empty() = 
    let started = ref false 
    { new IEnumerator
        with get_Current() = 
            check !started; 
            already_finished()
        and  MoveNext() = 
            if not !started then started := true;
            false
        and  Reset() = 
            started := false;
            ();  } 

  let to_typed (e :> IEnumerator) : IEnumerator<'a> = 
    { new IEnumerator<'a> with 
          member x.Current = unbox e.Current
      interface IEnumerator with 
          member x.Current = unbox  e.Current
          member x.MoveNext() = e.MoveNext()
          member x.Reset() = e.Reset();
      interface System.IDisposable with 
          member x.Dispose() = ()  }
  
//#if CLI_AT_LEAST_2_0
  
  type EmptyEnumerator<'a>() = 
    let mutable started = false 
    interface IEnumerator<'a> with 
          member x.Current = 
            check started; 
            (already_finished() : 'a)
            
    interface System.Collections.IEnumerator with 
        member x.Current = 
            check started; 
            (already_finished() : obj)
        member x.MoveNext() = 
            if not started then started <- true;
            false
        member x.Reset() = 
            started <- false;
            ()
    interface System.IDisposable with 
          member x.Dispose() = () 

  
  let empty () = (new EmptyEnumerator<'a>() :> IEnumerator<'a>)
  
  let iter f (e : #IEnumerator<'a>) = 
      while e.MoveNext() do
        f e.Current;
      done

  let iteri f (e : #IEnumerator<'a>) = 
      let mutable i = 0 
      while e.MoveNext() do
        f i e.Current;
        i <- i + 1;
      done

  let rec nth i (e : #IEnumerator<'a>) = 
      if not (e.MoveNext()) || i < 0 then  invalid_arg "Seq.nth"; 
      if i = 0 then e.Current
      else nth (i-1) e

  let iter2 f (e1 : #IEnumerator<'a>) (e2 : #IEnumerator<'b>) = 
      while (let n1 = e1.MoveNext()
             let n2 = e2.MoveNext()
             if n1 <> n2 then invalid_arg "Seq.iter2 the collections have different lengths";
             n1) do
          f e1.Current e2.Current;
      done

  let map f (e :> IEnumerator<'a>) = 
    { new IEnumerator<'b> with 
          member x.Current = f e.Current
      interface IEnumerator with 
          member x.Current = box (f e.Current)
          member x.MoveNext() = e.MoveNext()
          member x.Reset() = e.Reset();
      interface System.IDisposable with 
          member x.Dispose() = e.Dispose()  }

  let mapi f (e :> IEnumerator<'a>) = 
    let i = ref (-1) 
    { new IEnumerator<'b> with
          member x.Current= 
              if !i < 0 then not_started();
              f !i e.Current
      interface IEnumerator with
          member x.Current = 
              if !i < 0 then not_started();
              box (f !i e.Current)
          member x.MoveNext() = i := !i + 1; e.MoveNext()
          member x.Reset() = i := -1; e.Reset();
      interface System.IDisposable with
          member x.Dispose() = e.Dispose()  }

  let map2 f (e1 : #IEnumerator<'a>) (e2 : #IEnumerator<'b>) = 
    { new IEnumerator<'c> with 
          member x.Current = f e1.Current e2.Current
      interface IEnumerator with 
          member x.Current = box (f e1.Current e2.Current)
          member x.MoveNext() = 
              let n1 = e1.MoveNext()
              let n2 = e2.MoveNext()
              if n1 <> n2 then invalid_arg "Seq.map2: the collections have different lengths";
              n1     
          member x.Reset() = e1.Reset(); e2.Reset();
      interface System.IDisposable with 
          member x.Dispose() = e1.Dispose(); e2.Dispose()  }

  let map_to_typed f (e :> IEnumerator) = 
    { new IEnumerator<'b> with 
          member x.Current = f (unbox e.Current)
      interface IEnumerator with 
          member x.Current = box (f (unbox e.Current))
          member x.MoveNext() = e.MoveNext()
          member x.Reset() = e.Reset();
      interface System.IDisposable with 
          member x.Dispose() = ()  }

  let isNone = function None -> true | _ -> false
  
  let choose f (e :> IEnumerator<'a>) = 
    let started = ref false 
    let curr = ref None 
    let get() =  check !started; (match !curr with None -> already_finished() | Some x -> x) 
    { new IEnumerator<'b> with 
          member x.Current = get()
      interface IEnumerator with 
          member x.Current = box (get())
          member x.MoveNext() = 
              if not !started then started := true;
              curr := None; 
              while (isNone !curr && e.MoveNext()) do 
                (curr := f e.Current);
              done;
              Option.is_some !curr
          member x.Reset() = 
              started := false;
              curr := None;
              e.Reset();
      interface System.IDisposable with 
          member x.Dispose() = e.Dispose()  }

  let filter f (e :> IEnumerator<'a>) = 
    let started = ref false 
    let this = 
        { new IEnumerator<'b> with 
              member x.Current = check !started; e.Current
          interface IEnumerator with 
              member x.Current = check !started; box e.Current
              member x.MoveNext() = 
                  let rec next() =
                      if not !started then started := true;
                      e.MoveNext() && (f  e.Current || next()) 
                  next()
              member x.Reset() = 
                  started := false;
                  e.Reset();
          interface System.IDisposable with 
              member x.Dispose() = e.Dispose()  } 
    this

  let unfold f x = 
      let started = ref false 
      let curr = ref None 
      let state = ref x 
      let getCurr() = 
          check !started;
          match !curr with None -> already_finished() | Some x -> x 
      { new IEnumerator<'b> with 
            member x.Current = getCurr()
        interface IEnumerator with 
            member x.Current = box (getCurr())
            member x.MoveNext() = 
                if not !started then started := true;
                match f !state with 
                | None -> (curr := None; false) 
                | Some(r,s) -> curr := Some r; state := s; true
            member self.Reset() = 
                started := false;
                state := x; 
                curr := None
        interface System.IDisposable with 
            member x.Dispose() = () } 

  let readAndClear r = lock r (fun () -> match !r with None -> None | Some x as res -> r := None; res)

  let generate openf compute closef : IEnumerator<'b> = 
      let started = ref false 
      let curr = ref None
      let state = ref (Some(openf())) 
      let getCurr() = 
          check !started;
          match !curr with None -> already_finished() | Some x -> x 
      let start() = if not !started then (started := true) 

      let dispose() = readAndClear state |> Option.iter closef
      let finish() = (dispose(); curr := None) 
      {  new IEnumerator<'b> with 
             member x.Current = getCurr()
         interface IEnumerator with 
             member x.Current = box (getCurr())
             member x.MoveNext() = 
                 start();
                 match !state with 
                 | None -> false (* we started, then reached the end, then got another MoveNext *)
                 | Some s -> 
                     match (try compute s with e -> finish(); rethrow ()) with 
                     | None -> finish(); false
                     | Some(r) as x -> curr := x; true

             member x.Reset() = 
                 started := false;
                 finish();
                 state := Some(openf())
         interface System.IDisposable with 
             member x.Dispose() = dispose() } 

  let fold  f x (e :> IEnumerator<'a>)  = 
      let mutable state = x 
      while e.MoveNext() do
          state <- f state  e.Current;
      state

  let fold1  f (e :> IEnumerator<'a>)  = 
      if not (e.MoveNext()) then invalid_arg "Seq.fold1";
      let mutable state = e.Current 
      while e.MoveNext() do
          state <- f state  e.Current;
      state

  let length (e :> IEnumerator<'a>)  = 
      let mutable state = 0 
      while e.MoveNext() do
          state <-  state + 1;
      state

  let exists f (e :> IEnumerator<'a>)  = 
      let mutable state = false 
      while (e.MoveNext() && not state) do
          state <- f e.Current
      state

  let for_all f (e :> IEnumerator<'a>)  = 
      let mutable state = true 
      while (e.MoveNext() && state) do
          state <- f e.Current
      state

  let hd (e :> IEnumerator<'a>)  = 
      if (e.MoveNext()) then e.Current
      else invalid_arg "hd"

  let nonempty (e :> IEnumerator<'a>)  = 
      e.MoveNext()

  let take n (e :> IEnumerator<'a>)  = 
      let mutable state = [] 
      for i = 0 to n - 1 do 
          if not (e.MoveNext()) then invalid_arg "take";
          state <-  e.Current :: state;
      List.rev state

  let to_list (e :> IEnumerator<_>) = 
      let mutable res = [] 
      while e.MoveNext() do
          res <- e.Current :: res
      List.rev res
    
#if CLI_AT_MOST_1_1
#else
  let to_array (e :> IEnumerator<_>) = 
      let res = new List<_>()
      while e.MoveNext() do
          res.Add(e.Current)
      res.ToArray()
#endif

  type ArrayEnumerator<'a>(arr: 'a array) = 
      let mutable curr = -1
      let mutable len = arr.Length
      member x.Get() =
           if curr >= 0 then 
             if curr >= len then already_finished()
             else arr.[curr]
           else 
             not_started()
      interface IEnumerator<'a> with 
            member x.Current = x.Get()
      interface System.Collections.IEnumerator with 
            member x.MoveNext() = 
                   if curr >= len then false
                   else 
                     curr <- curr + 1;
                     (curr < len)
            member x.Current = box(x.Get())
            member x.Reset() = curr <- -1
      interface System.IDisposable with 
            member x.Dispose() = ()        

  let of_array arr = (new ArrayEnumerator<'a>(arr) :> IEnumerator<'a>)
  
//#endif

  let untyped_to_list (e :> IEnumerator)  = 
      let mutable state = [] 
      while e.MoveNext() do
          state <-  unbox e.Current :: state;
      List.rev state


  let do_finally f (e : #IEnumerator<'a>) = 
      { new IEnumerator<'b> with
            member x.Current = e.Current
        interface IEnumerator with
            member x.Current = (e :> IEnumerator).Current
            member x.MoveNext() = e.MoveNext()
            member x.Reset() = e.Reset();
        interface System.IDisposable with
            member x.Dispose() = f(); e.Dispose()  }


  type Singleton<'a>(v:'a) = 
      let mutable started = false
      interface IEnumerator<'a> with
            member x.Current = v
      interface IEnumerator with
          member x.Current = box v
          member x.MoveNext() = if started then false else (started <- true; true)
          member x.Reset() = ()
      interface System.IDisposable with
          member x.Dispose() = ()  

  let singleton x = (new Singleton<'a>(x) :> IEnumerator<'a>)

// Use generators for some implementations of IEnumerables.
//
module Generator = 

    #light
    open System.Collections
    #if CLI_AT_MOST_1_1
    open Microsoft.FSharp.Compatibility
    #else
    open System.Collections.Generic
    #endif
    
    type Step<'a> = 
        | Stop
        | Yield of 'a
        | Goto of Generator<'a>

    and Generator<'a> = 
        abstract Apply: (unit -> Step<'a>)
        abstract Disposer: (unit -> unit) option

    let dispose (g:Generator<'a>) = 
        match g.Disposer with 
        | None -> () 
        | Some f -> f()
    
    let chainDisposeG d1 (g:Generator<'a>) =
        let app = g.Apply
        let disp = match g.Disposer with Some f2 -> Some(fun () -> f2(); d1()) | None -> Some d1
        { new Generator<_> with 
              member x.Apply = app 
              member x.Disposer = disp }

    let appG (g:Generator<_>) =
        //System.Console.WriteLine("{0}.appG", box g)
        let res = g.Apply()
        match res with 
        | Goto(next) -> 
            Goto(next)
        | Yield _ ->  
            res
        | Stop -> 
            //System.Console.WriteLine("appG: Stop")
            dispose g; 
            res
    
    // Binding. 
    //
    // We use a type defintion to apply a local dynamic optimization. 
    // We automatically right-associate binding, i.e. push the continuations to the right.
    // That is, bindG (bindG G1 cont1) cont2 --> bindG G1 (cont1 o cont2)
    // This makes constructs such as the following linear rather than quadratic:
    //
    //  let rec rwalk n = { if n > 0 then 
    //                         ->> rwalk (n-1)
    //                         -> n }

    type GenerateThen<'a>(g:Generator<'a>, cont : unit -> Generator<'a>) =
        member self.Generator = g
        member self.Cont = cont
        interface Generator<'a> with 
             member x.Apply = fun () -> 
                  match appG g with 
                  | Stop -> 
                      // OK, move onto the generator given by the continuation
                      Goto(cont())

                  | Yield v as res -> 
                      res
                  
                  | Goto next -> 
                      Goto(GenerateThen.Bind(next,cont))
             member x.Disposer = 
                  g.Disposer


        static member Bind (g:Generator<'a>, cont) = 
            match g with
            | :? GenerateThen<_> as g -> GenerateThen.Bind(g.Generator,(fun () -> GenerateThen.Bind (g.Cont(), cont)))
            | g -> (new GenerateThen<'a>(g, cont) :> Generator<'a>)


    let bindG g cont = GenerateThen.Bind(g,cont)

    //let emptyG () = 
    //    { new Generator<_> with 
    //           member x.Apply = (fun () -> Stop)
    //           member x.Disposer = None }
    //
    //let delayG f  = 
    //    { new Generator<_> with 
    //           member x.Apply = fun () -> Goto(f())
    //           member x.Disposer = None }
    //
    //let useG (v: #System.IDisposable) f = 
    //    { new Generator<_> with 
    //           member x.Apply = (fun () -> 
    //               let g = f v in 
    //               // We're leaving this generator but want to maintain the disposal on the target.
    //               // Hence chain it into the disposer of the target
    //               Goto(chainDisposeG v.Dispose g))
    //           member x.Disposer = Some (fun () -> v.Dispose()) }
    //
    //let yieldG (v:'a) = 
    //    let yielded = ref false
    //    { new Generator<_> with 
    //           member x.Apply = fun () -> if !yielded then Stop else (yielded := true; Yield(v))
    //           member x.Disposer = None }
    //
    //let rec whileG gd b = if gd() then bindG (b()) (fun () -> whileG gd b) else emptyG()
    //
    //let yieldThenG x b = bindG (yieldG x) b
    //
    //let forG (v: #seq<'a>) f = 
    //    let e = v.GetEnumerator() in 
    //    whileG e.MoveNext (fun () -> f e.Current)

    // Internal type. Drive an underlying generator. Crucially when the generator returns
    // a new generator we simply update our current generator and continue. Thus the enumerator
    // effectively acts as a reference cell holding the current generator. This means that
    // infinite or large generation chains (e.g. caused by long sequences of append's, including 
    // possible delay loops) can be referenced via a single enumerator.
    //
    // A classic case where this arises in this sort of sequence expression:
    //    let rec data s = { yield s; 
    //                       yield! data (s + random()) }
    //
    // This translates to 
    //    let rec data s = Seq.delay (fun () -> Seq.append (Seq.singleton s) (Seq.delay (fun () -> data (s+random()))))
    //
    // When you unwind through all the Seq, IEnumerator and Generator objects created, 
    // you get (data s).GetEnumerator being an "generateFromEnumerator(EnumeratorWrappingLazyGenerator(...))" for the append.
    // After one element is yielded, we move on to the generator for the inner delay, which in turn
    // comes back to be a "generateFromEnumerator(EnumeratorWrappingLazyGenerator(...))".
    //
    // Defined as a type so we can optimize Enumerator/Generator chains in enumerateFromLazyGenerator
    // and generateFromEnumerator.


    type EnumeratorWrappingLazyGenerator<'a>(g:Generator<'a>) =
        let mutable g = g
        let mutable curr = None
        let mutable finished = false
        member e.Generator = g
        interface IEnumerator<'a> with
            member x.Current= match curr with Some(v) -> v | None -> failwith "MoveNext not called, or finished"
        interface System.Collections.IEnumerator with
            member x.Current = box (x :> IEnumerator<_>).Current
            member x.MoveNext() = 
                //System.Console.WriteLine "EnumeratorWrappingLazyGenerator, MoveNext"  
                not finished && 
                (match appG g with 
                 | Stop -> 
                    curr <- None; 
                    finished <- true; //System.Console.WriteLine "EnumeratorWrappingLazyGenerator, MoveNext, done"  
                    false
                 | Yield(v) ->
                    curr <- Some(v); 
                    true
                 | Goto(next) ->
                    (g <- next);
                    (x :> IEnumerator).MoveNext())
            member x.Reset() = failwith "cannot reset this enumerator"
        interface System.IDisposable with
            member x.Dispose() = if not finished then dispose g

    // Internal type, used to optimize Enumerator/Generator chains
    type LazyGeneratorWrappingEnumerator<'a>(e:IEnumerator<'a>) =
        member g.Enumerator = e
        interface Generator<'a> with
            member g.Apply =  fun () -> 
                if e.MoveNext() then 
                    Yield(e.Current) 
                else 
                    Stop
            member g.Disposer= Some(e.Dispose)

    let enumerateFromGenerator(g:Generator<'a>) = 
        match g with 
        | :? LazyGeneratorWrappingEnumerator<'a> as g -> g.Enumerator
        | _ -> (new EnumeratorWrappingLazyGenerator<_>(g) :> IEnumerator<_>)

    let generateFromEnumerator (e:IEnumerator<'a>) =
        match e with 
        | :? EnumeratorWrappingLazyGenerator<'a> as e ->  e.Generator
            //System.Console.WriteLine("generateFromEnumerator: elim")
        | _ -> (new LazyGeneratorWrappingEnumerator<'a>(e) :> Generator<'a>)
  
  

module IEnumerable = 

    let untyped_iter f (ie :> IEnumerable) = 
        let e = ie.GetEnumerator()
        IEnumerator.untyped_iter f e

    let untyped_fold f x (ie :> IEnumerable) = 
        let e = ie.GetEnumerator() 
        IEnumerator.untyped_fold f x e

    let untyped_map f (ie :> IEnumerable) = 
        { new IEnumerable with GetEnumerator() = IEnumerator.untyped_map f (ie.GetEnumerator()) }

    let untyped_filter f (ie :> IEnumerable) = 
        { new IEnumerable with GetEnumerator() = IEnumerator.untyped_filter f (ie.GetEnumerator()) }

    let mk_IEnumerable f = 
        { new IEnumerable<'b> with 
            member x.GetEnumerator() = f()
          interface IEnumerable with 
            member x.GetEnumerator() = (f() :> IEnumerator) }

    let delay (f: unit -> #IEnumerable<'a>) = mk_IEnumerable (fun () -> f().GetEnumerator())

    let unfold f x = mk_IEnumerable (fun () -> IEnumerator.unfold f x) 
    
    type EmptyEnumerable<'a> = 
        | EmptyEnumerable
        interface IEnumerable<'a> with 
            member x.GetEnumerator() = IEnumerator.empty<'a>()
        interface IEnumerable with 
            member x.GetEnumerator() = (IEnumerator.empty<'a>() :> IEnumerator) 
          
    let empty () = (EmptyEnumerable :> IEnumerable<'a>)
    
    let init_infinite f = unfold (fun s -> Some(f s, s + 1)) 0
    let init_finite n f = unfold (fun s -> if s < n then Some(f s, s+1) else None) 0

    let iter    f (ie :> IEnumerable<'a>) = using (ie.GetEnumerator()) (IEnumerator.iter    f)
    let nth     i (ie :> IEnumerable<'a>) = using (ie.GetEnumerator()) (IEnumerator.nth     i)
    let iteri   f (ie :> IEnumerable<'a>) = using (ie.GetEnumerator()) (IEnumerator.iteri   f)
    let exists  f (ie :> IEnumerable<'a>) = using (ie.GetEnumerator()) (IEnumerator.exists  f) 
    let for_all f (ie :> IEnumerable<'a>) = using (ie.GetEnumerator()) (IEnumerator.for_all f)
    let iter2   f (ie1 :> IEnumerable<'a>) (ie2 :> IEnumerable<'b>)    = 
        use ie1 = ie1.GetEnumerator()
        use ie2 = ie2.GetEnumerator()
        IEnumerator.iter2 f ie1 ie2


    // Build an IEnumerble by wrapping/transforming iterators as they get generated.
    let revamp f (ie :> IEnumerable<'a>) = mk_IEnumerable (fun () -> f (ie.GetEnumerator()))
    let revamp2 f (ie1 :> IEnumerable<'a>) (ie2 :> IEnumerable<'b>) = mk_IEnumerable (fun () -> f (ie1.GetEnumerator()) (ie2.GetEnumerator()))
    let proj_untyped_to_typed f (e :> IEnumerable)= mk_IEnumerable (fun () -> f (e.GetEnumerator()))
    let untyped_to_typed e = proj_untyped_to_typed IEnumerator.to_typed e
    let map_with_type f e  = proj_untyped_to_typed (IEnumerator.map_to_typed f) e
    let map_to_typed f e   = proj_untyped_to_typed (IEnumerator.map_to_typed f) e
    let filter f ie      = revamp  (IEnumerator.filter f) ie
    let map    f ie      = revamp  (IEnumerator.map    f) ie
    let mapi   f ie      = revamp  (IEnumerator.mapi   f) ie
    let map2   f ie1 ie2 = revamp2 (IEnumerator.map2    f) ie1 ie2
    let choose f ie      = revamp  (IEnumerator.choose f) ie
    let combine ie1 ie2  = map2    (fun x y -> x,y) ie1 ie2

    let generate openf compute closef = mk_IEnumerable (fun () -> IEnumerator.generate openf compute closef) 
    let generate_using (openf : unit -> ('b :> System.IDisposable)) compute = generate openf compute (fun s -> s.Dispose())

    let of_functions opener movenext current = 
        generate 
            opener 
            (fun x -> if movenext x then Some(current x) else None) 
            (fun x -> match box(x) with :? System.IDisposable as id -> id.Dispose() | _ -> ())


    let first f (ie :> IEnumerable<'a>)  = 
        use e = ie.GetEnumerator()
        let mutable res = None 
        while (Option.is_none res && e.MoveNext()) do
          res <-  f e.Current;
        done;
        res
      
    let tryfind f (ie :> IEnumerable<'a>)  = 
        use e = ie.GetEnumerator()
        let mutable res = None 
        while (Option.is_none res && e.MoveNext()) do
          let c = e.Current in if f c then res <- Some(c)
        done;
        res

    let find f ie = match tryfind f ie with None -> not_found() | Some x -> x

    let hd (ie : #IEnumerable<'a>)        = using (ie.GetEnumerator()) IEnumerator.hd 
    let take n (ie : #IEnumerable<'a>)    = 
        (* Note: don't create or dispose any IEnumerable if n = 0 *)
        if n = 0 then [] else  using (ie.GetEnumerator()) (IEnumerator.take n)
    let nonempty (ie : #IEnumerable<'a>)  = using (ie.GetEnumerator()) IEnumerator.nonempty 
    let length (ie : #IEnumerable<'a>)    = using (ie.GetEnumerator()) IEnumerator.length
    let fold f x (ie : #IEnumerable<'a>)  = using (ie.GetEnumerator()) (IEnumerator.fold f x)
    let fold1 f (ie : #IEnumerable<'a>)  = using (ie.GetEnumerator()) (IEnumerator.fold1 f)

    let get (ie :> IEnumerator<_>) = if ie.MoveNext() then Some(ie.Current) else None

    let fromGenerator f = mk_IEnumerable(fun () -> Generator.enumerateFromGenerator (f()))
    let toGenerator (ie : #IEnumerable<_>) = Generator.generateFromEnumerator (ie.GetEnumerator())


    /// An optimized object for map_concat
    type ConcatEnumerator<'a,'b when 'b :> IEnumerable<'a>>(ies: IEnumerable<'b>) = 
        let mutable outerEnum = ies.GetEnumerator()
        let mutable currInnerEnum = IEnumerator.empty()

        let mutable started = false 
        let mutable finished = false 
        
        [<DefaultValue(false)>] // false = unchecked
        val mutable currElement : 'a

        member x.Finish() = 
            finished <- true
            match box currInnerEnum with 
            | null -> ()
            | _ -> 
                currInnerEnum.Dispose()
                currInnerEnum <- Unchecked.defaultof<_>
            match box outerEnum with 
            | null -> ()
            | _ -> 
                outerEnum.Dispose()
                outerEnum <- Unchecked.defaultof<_>
            
        member x.GetCurrent() = 
              IEnumerator.check started;
              if finished then IEnumerator.already_finished() else x.currElement
        
        interface IEnumerator<'a> with 
           member x.Current = x.GetCurrent()

        interface IEnumerator with 
           member x.Current = box (x.GetCurrent())
               
           member x.MoveNext() = 
               if not started then (started <- true) 
               if finished then false
               else 
                  let rec takeInner () = 
                    // check the inner list
                    if currInnerEnum.MoveNext() then 
                        x.currElement <- currInnerEnum.Current; 
                        true
                    else
                        // check the outer list
                        let rec takeOuter() = 
                            if outerEnum.MoveNext() then 
                                let ie = outerEnum.Current 
                                // Optimization to detect the statically-allocated empty IEnumerables
                                match box ie with
                                | :? EmptyEnumerable<'a> -> 
                                     // This one is empty, just skip, don't call GetEnumerator, try again
                                     takeOuter()
                                | _ -> 
                                     // OK, this one may not be empty.
                                     // Don't forget to dispose of the enumerator for the inner list now we're done with it
                                     currInnerEnum.Dispose(); 
                                     currInnerEnum <- ie.GetEnumerator(); 
                                     takeInner ()
                            else 
                                // We're done
                                x.Finish()
                                false
                        takeOuter()

                  try takeInner ()
                  with e -> x.Finish(); rethrow () 

           member x.Reset() = 
               started <- false;
               x.Finish();
               outerEnum <- ies.GetEnumerator();
               currInnerEnum <- IEnumerator.empty()

        interface System.IDisposable with 
            member x.Dispose() = x.Finish() 

      
    let concat (ies: #IEnumerable<'b :> IEnumerable<'a>>) = 
        mk_IEnumerable (fun () -> new ConcatEnumerator<_,_>(ies) :> IEnumerator<'a>)
            
    let append (ie1: #IEnumerable<'a>) (ie2: #IEnumerable<'a>) = 
        fromGenerator(fun () -> Generator.bindG (toGenerator ie1) (fun () -> toGenerator ie2))

    let map_concat f ies = concat (map f ies)


    let compare (f:'a -> 'a -> int) (s1 : #IEnumerable<'a>) (s2: #IEnumerable<'a>) = 
        use e1 = s1.GetEnumerator()
        use e2 = s2.GetEnumerator()
        let rec go () = 
            let e1ok = e1.MoveNext() 
            let e2ok = e2.MoveNext() 
            let c = (if e1ok = e2ok then 0 else if e1ok then 1 else -1) 
            if c <> 0 then c else
            if not e1ok || not e2ok then 0 
            else
                let c = f e1.Current e2.Current 
                if c <> 0 then c else
                go () 
        go()

    let of_list (l : 'a list) = (l :> seq<'a>)
    let to_list (c :> IEnumerable<_>) = using (c.GetEnumerator()) IEnumerator.to_list 

    // Create a new object to ensure underlying array may not be mutated by a backdoor cast 
    let of_array (a : 'a array) = mk_IEnumerable (fun () -> IEnumerator.of_array a)        
#if CLI_AT_MOST_1_1
    let to_array ie = Array.of_list (to_list ie)
#else
    let to_array (c :> IEnumerable<_>)  = using (c.GetEnumerator()) IEnumerator.to_array
#endif

    let singleton x = mk_IEnumerable (fun () -> IEnumerator.singleton x)

    let untyped_to_list (ie :> IEnumerable) = IEnumerator.untyped_to_list (ie.GetEnumerator())

    let truncate n (x: #seq<'a>) =
        generate 
            ( fun () -> ref 0,x.GetEnumerator() ) 
            ( fun (i,ie) -> if !i >= n || not (ie.MoveNext()) then None else (i := !i + 1; Some(ie.Current)) )
            ( fun (_,ie) -> ie.Dispose () )

    let pairwise (x: #seq<'a>) =
        generate 
            ( fun () -> ref None,x.GetEnumerator() ) 
            ( fun (i,ie) -> 
                 let rec look() = 
                   if not (ie.MoveNext()) then None 
                   else 
                     let curr = ie.Current in 
                     match !i with 
                     | None -> i := Some(curr); look()
                     | Some prev -> i := Some(curr); Some(prev,curr) 
                 look())
            ( fun (_,ie) -> ie.Dispose () )


module Seq = 

    let untyped_iter f ie = IEnumerable.untyped_iter f ie
    let untyped_fold f x ie = IEnumerable.untyped_fold f x ie
    let untyped_map f ie = IEnumerable.untyped_map f ie
    let untyped_filter f ie = IEnumerable.untyped_filter f ie
    let delay f = IEnumerable.delay f
    let unfold f x = IEnumerable.unfold f x
    let empty<'a> : seq<'a> = IEnumerable.empty()
    let init_infinite f = IEnumerable.init_infinite f
    let init_finite n f = IEnumerable.init_finite n f
    let iter    f ie = IEnumerable.iter f ie
    let nth i ie = IEnumerable.nth i ie
    let iteri f ie = IEnumerable.iteri f ie
    let exists f ie = IEnumerable.exists f ie
    let for_all f ie = IEnumerable.for_all f ie
    let iter2 f ie1 ie2 = IEnumerable.iter2 f ie1 ie2
    let untyped_to_typed e = IEnumerable.untyped_to_typed e
    let map_with_type f e  = IEnumerable.map_with_type f e
    let map_to_typed f e   = IEnumerable.map_to_typed f e
    let filter f ie      = IEnumerable.filter f ie
    let map    f ie      = IEnumerable.map f ie
    let mapi   f ie      = IEnumerable.mapi f ie
    let map2   f ie1 ie2 = IEnumerable.map2 f ie1 ie2
    let choose f ie      = IEnumerable.choose f ie
    let zip     ie1 ie2  = IEnumerable.combine ie1 ie2
    let combine ie1 ie2  = IEnumerable.combine ie1 ie2
    let generate openf compute closef = IEnumerable.generate openf compute closef
    let generate_using openf compute = IEnumerable.generate_using openf compute
    let of_functions opener movenext current = IEnumerable.of_functions opener movenext current 
    let first f ie = IEnumerable.first f ie
    let tryfind f ie = IEnumerable.tryfind f ie
    let find f ie = IEnumerable.find f ie
    let hd ie = IEnumerable.hd ie
    let take n ie = IEnumerable.take n ie
    let nonempty ie = IEnumerable.nonempty ie
    let length ie = IEnumerable.length ie
    let fold f x ie      = IEnumerable.fold f x ie
    let fold1 f ie      = IEnumerable.fold1 f ie
    let get ie           = IEnumerable.get ie      
    let concat ies       = IEnumerable.concat ies
    let append ie1 ie2   = IEnumerable.append ie1 ie2
    let map_concat f ies = IEnumerable.map_concat f ies
    let compare f s1 s2  = IEnumerable.compare f s1 s2
    let of_list l   = IEnumerable.of_list l
    let to_list ie  = IEnumerable.to_list ie
    let of_array arr  = IEnumerable.of_array arr
    let to_array ie = IEnumerable.to_array ie
    let singleton x = IEnumerable.singleton x
    let untyped_to_list ie = IEnumerable.untyped_to_list ie
    let truncate n ie = IEnumerable.truncate n ie
    let pairwise ie = IEnumerable.pairwise ie


    // CONSIDER: consider rewriting this as a generator
    let cons (x:'a) (ie2: #IEnumerable<'a>) = 
        generate
          (fun () -> ref None)
          (fun curr -> 
              match !curr with 
              | None -> curr := Some(ie2.GetEnumerator()); Some(x)
              | Some(e2) -> get e2)
          (fun curr -> match !curr with None -> () | Some e2 -> e2.Dispose())

    // CONSIDER: consider rewriting this as a generator. A long sequence of tail_cons will
    // be O(n^2)
    let tail_cons (ie1: #IEnumerable<'a>) (x:'a) = 
        generate
          (fun () -> ref (Some(ie1.GetEnumerator())) )
          (fun curr -> 
              match !curr with 
              | None -> None
              | Some(e2) -> 
                  match get e2 with 
                  | None -> curr := None; Some(x)
                  | res -> res)                
          (fun curr -> match !curr with None -> () | Some e2 -> e2.Dispose())

    // CONSIDER: consider rewriting this as a generator
    let scan0 f z (seq : #seq<'a>) = 
        generate 
          (fun () -> seq.GetEnumerator(), ref z) 
          (fun (ie,zref) -> 
              if ie.MoveNext() then 
                  let z' = f !zref ie.Current in 
                  zref := z';
                  Some(z')
              else 
                  None) 
          (fun (ie,_) -> ie.Dispose())

    let scan f z seq = cons z (scan0 f z seq)

    let scan1 f (seq : #seq<'a>) = 
        generate 
          (fun () -> seq.GetEnumerator(), ref None) 
          (fun (ie,zref) -> 
              match !zref with 
                | None ->
                  if ie.MoveNext() then 
                    let z0 = ie.Current in 
                    zref := Some(z0);
                    Some(z0)
                  else 
                    invalid_arg "Seq.scan1"
                | Some z -> 
                  if ie.MoveNext() then 
                      let z' = f z ie.Current in 
                      zref := Some(z');
                      Some(z')
                  else 
                      None)
          (fun (ie,_) -> ie.Dispose())

                            
    let do_finally (rest : #seq<'a>) (compensation : unit -> unit)  =
        IEnumerable.mk_IEnumerable (fun () -> 
            try 
                IEnumerator.do_finally
                    compensation
                    (rest.GetEnumerator()) 
            with e -> 
                compensation(); 
                rethrow() )

    let using (resource : #System.IDisposable) rest = 
        IEnumerable.mk_IEnumerable (fun () -> 
            try 
                let rest2 = (rest resource :> seq<'a>) in
                IEnumerator.do_finally
                    (fun () -> resource.Dispose()) 
                    (rest2.GetEnumerator()) 
            with e -> 
                resource.Dispose(); 
                rethrow() )

    let generated (g : unit -> bool) (b: #seq<'a>) : seq<'a> = 
        let started = ref false 
        let curr = ref None
        let getCurr() = 
            IEnumerator.check !started;
            match !curr with None -> IEnumerator.already_finished() | Some x -> x 
        let start() = if not !started then (started := true) 

        let finish() = (curr := None) 
        concat 
           (IEnumerable.mk_IEnumerable (fun () -> 
                { new IEnumerator<'b> with 
                      member x.Current = getCurr()
                   interface IEnumerator with 
                      member x.Current = box (getCurr())
                      member x.MoveNext() = 
                           start();
                           let keepGoing = (try g() with e -> finish(); rethrow ()) in
                           if keepGoing then 
                               curr := Some(b); true
                           else 
                               finish(); false
                      member x.Reset() = 
                           started := false;
                           finish()
                   interface System.IDisposable with 
                      member x.Dispose() = () }))

    let find_index p s = s |> find (snd >> p) |> fst
    let find_indexi p s = s |> find (fun (x,y) -> p x y) |> fst
    let tryfind_index p s = s |> tryfind (snd >> p) |> Option.map fst
    let tryfind_indexi p s = s |> tryfind (fun (x,y) -> p x y) |> Option.map fst
    
#if CLI_AT_MOST_1_1
#else
    let cache_all (seq : #seq<'a>) = 
        let t = new List<_>(3)
        let complete = ref false
        // Hold the underlying sequence in a reference so that it gets garbage collected
        // once we've done a complete traversal.
        let seq = ref (Some seq)
        generate
           // Generate the initial state
           (fun () -> (ref 0, lock t (fun () -> if !complete then None else Some(seq.Value.Value.GetEnumerator()))))
           // Compute an entry
           (fun (i,eopt) -> 
                if !i < t.Count then 
                    // OK, we have a lookaside entry
                    let r = lock t (fun () -> t.[!i])
                    i := !i + 1;
                    //if we're not complete and have an enumerator then we have to move it forward
                    // just in case we need it to fill in later results. If we're complete we
                    // can ignore the enumerator.
                    if not !complete then 
                        match eopt with 
                        | None -> ()
                        | Some e -> e.MoveNext() |> ignore; 
                    // ok, return the result from the table
                    Some(r) 
                else 
                    // OK, no lookaside entry
                     match eopt with 
                     | None -> None
                     | Some e -> 
                        if e.MoveNext() then 
                            let r = e.Current 
                            // Write the entry. A race to here doesn't matter.
                            lock t (fun () -> t.Add(r));
                            // Increment our private state
                            i := !i + 1; 
                            Some(r) 
                        else 
                            // Hey, reached the end! No more enumerators needed!
                            lock t (fun () -> complete := true; seq := None);
                            None)
           // Cleanup
           (fun (i,eopt) -> match eopt with None -> () | Some e -> e.Dispose())

    [<Sealed>]
    type CacheResult<'a>(cleanup,res:seq<'a>) =
        interface System.IDisposable with 
            member x.Dispose() = cleanup()
        interface IEnumerable<'a> with 
            member x.GetEnumerator() = res.GetEnumerator()
        interface IEnumerable with 
            member x.GetEnumerator() = (res :> System.Collections.IEnumerable).GetEnumerator()
        member obj.Clear() = cleanup()
           
    let cache (seq : #seq<'a>) = 
        // Wrap a seq to ensure that it is enumerated just once and only as far as is necessary.
        //
        // This code is required to be thread safe.
        // The necessary calls should be called at most once (include .MoveNext() = false).
        // The enumerator should be disposed (and dropped) when no longer required.
        //------
        // The state is (prefix,enumerator) with invariants:
        //   * the prefix followed by elts from the enumerator are the initial sequence.
        //   * the prefix contains only as many elements as the longest enumeration so far.
        let prefix      = List<_>()
        let enumeratorR = ref None : IEnumerator<'a> option option ref // nested options rather than new type...
                           // None          = Unstarted.
                           // Some(Some e)  = Started.
                           // Some None     = Finished.
        let oneStepTo i = 
          // If possible, step the enumeration to prefix length i (at most one step).
          // Be speculative, since this could have already happened via another thread.
          lock enumeratorR (fun () ->
                                if not (i < prefix.Count) then // is a step still required?
                                    // If not yet started, start it (create enumerator).
                                    if !enumeratorR = None then enumeratorR := Some (Some (seq.GetEnumerator()))
                                    match (!enumeratorR).Value with
                                    | Some enumerator -> if enumerator.MoveNext() then
                                                            prefix.Add(enumerator.Current)
                                                         else
                                                            enumerator.Dispose()     // Move failed, dispose enumerator,
                                                            enumeratorR := Some None // drop it and record finished.
                                    | None -> ())
        let result = 
            unfold (fun i -> // i being the next position to be returned
                          if i < prefix.Count then
                            Some (prefix.[i],i+1)
                          else
                            oneStepTo i
                            if i < prefix.Count then
                              Some (prefix.[i],i+1)
                            else
                              None) 0
        let cleanup() = 
           lock enumeratorR (fun () -> 
               prefix.Clear()
               begin match !enumeratorR with 
               | Some (Some e) -> IEnumerator.dispose e; 
               | _ -> ()
               end;
               enumeratorR := None)
        new CacheResult<_>(cleanup, result)

    let readonly (s:#seq<_>) = IEnumerable.mk_IEnumerable (fun x -> s.GetEnumerator())

    let groupBy keyf seq =

        delay (fun () -> 
            let dict = new Dictionary<'key,List<'a>>()

            // Build the groupings
            seq |> iter (fun v -> 
                let key = keyf v 
                let ok,prev = dict.TryGetValue(key)
                if ok then prev.Add(v)
                else let prev = new List<'a>(1)
                     dict.[key] <- prev
                     prev.Add(v))

            // Trim the size of each result group.
            dict |> iter (fun group -> group.Value.TrimExcess())
                     
            // Return the sequence-of-sequences. Don't reveal the 
            // internal collections: just reveal them as sequences
            dict |> map (fun group -> (group.Key, readonly group.Value)))

    let orderBy keyf seq =
        delay (fun () -> 
            let arr = to_array seq 
    #if CLI_AT_MOST_1_1
            System.Array.Sort((arr :> System.Array), 
                              { new System.Collections.IComparer
                                  with Compare(a,b) = Microsoft.FSharp.Core.Operators.compare (keyf a) (keyf b) } )
    #else
            System.Array.Sort(arr, { new IComparer<'a> with member self.Compare(x1,x2) = Microsoft.FSharp.Core.Operators.compare (keyf x1) (keyf x2) })
    #endif
            readonly arr)

    let countBy keyf seq =

        delay (fun () -> 
            let dict = new Dictionary<'key,int>()

            // Build the groupings
            seq |> iter (fun v -> 
                let key = keyf v 
                let ok,prev = dict.TryGetValue(key)
                if ok then dict.[key] <- dict.[key] + 1
                else dict.[key] <- 1)

            dict |> map (fun group -> (group.Key, group.Value)))

    let sumByInt     f seq = fold (fun acc v -> acc + f v) 0    seq
    let sumByFloat   f seq = fold (fun acc v -> acc + f v) 0.0  seq
    let sumByFloat32 f seq = fold (fun acc v -> acc + f v) 0.0f seq
    let sumByInt64   f seq = fold (fun acc v -> acc + f v) 0L   seq


#endif


