
//-----------------------------------------------------------------------------
// Some helper functions used by more or less every DirectX application
//
// Copyright (c) Microsoft Corporation 2005-2006.
// This sample code is provided "as is" without warranty of any kind. 
// We disclaim all warranties, either express or implied, including the 
// warranties of merchantability and fitness for a particular purpose. 
//-----------------------------------------------------------------------------

#light

module DirectXLib

open Compatibility
open System
open System.Drawing
open System.Windows.Forms
open Microsoft.DirectX
open Microsoft.DirectX.Direct3D

//-----------------------------------------------------------------------------
// Float32 arithmetic is standard
//-----------------------------------------------------------------------------

let PI = float32(Math.PI)

//-----------------------------------------------------------------------------
// Create a stream of Float32 time values
//-----------------------------------------------------------------------------

let time = 
   let startTime = Environment.TickCount in 
   (fun () -> float32(Environment.TickCount - startTime))
   
//-----------------------------------------------------------------------------
// A standard function to create a standard form
//-----------------------------------------------------------------------------
 
let NewStandardForm(title,width,height) = 
    new Form(Text = "F# Direct3D Tutorial 6 - Meshes",
             ClientSize = new Size(400, 300),
             Visible = true)

//-----------------------------------------------------------------------------
// A standard function to create a standard device
//-----------------------------------------------------------------------------

let NewStandardDevice(form : Form) = 
  let presentParams = new PresentParameters(Windowed = true,
                                            SwapEffect = SwapEffect.Discard,
                                            // Turn on a Depth stencil
                                            EnableAutoDepthStencil = true,
                                            // And the stencil format
                                            AutoDepthStencilFormat = DepthFormat.D16)
  new Device(0, DeviceType.Hardware, form,
             CreateFlags.SoftwareVertexProcessing, 
             [| presentParams |]) 


//-----------------------------------------------------------------------------
// Abstract away some of the gore from DirectX programming.  For these samples
// we simply want to render some stuff on a form.  
//-----------------------------------------------------------------------------

type Renderer = 
  { OnDeviceReset : Device -> RendererForDevice }
and RendererForDevice = 
  { Render: unit -> unit }

let SimpleRenderer (f : Device -> unit) = 
  { OnDeviceReset = (fun device -> { Render = fun () -> f device} ) }

let Renderer f = { OnDeviceReset = f }
let RendererForDevice f = { Render = f }

let RenderInLoop((form:Form), (device:Device), renderers) =
  // We reapply the renderers to the device upon each reset
  
  let actions = ref [] in 
  let OnDeviceReset () =
    // Turn on the zbuffer and turn on ambient lighting 
    device.RenderState.ZBufferEnable <- true;
    device.RenderState.Ambient <- System.Drawing.Color.White;
    actions :=  List.map (fun f -> f.OnDeviceReset(device)) renderers; in

  let invisible = ref false in 
  let CheckForMinimized() =
        invisible := (form.WindowState = FormWindowState.Minimized) 
                 || (form.Visible = false) in 

  let render() = 
    if not !invisible then begin
      device.BeginScene();
      List.iter (fun f -> f.Render()) !actions;
      device.EndScene();
      device.Present()
    end in
    
  form.add_Resize(new EventHandler(fun _ _ -> CheckForMinimized()));
  form.add_Activated(new EventHandler(fun _ _ -> CheckForMinimized()));
  form.add_Paint(new PaintEventHandler(fun _ _ -> render()));
  device.add_DeviceReset (new EventHandler(fun _ _ -> OnDeviceReset()));
  OnDeviceReset();
  while form.Created do
        render();
        Application.DoEvents();
  done

//-----------------------------------------------------------------------------
// A little library of standard rendering functions
//-----------------------------------------------------------------------------

let ChangingWorldTransform(worldf) = SimpleRenderer (fun device -> device.Transform.World <- worldf())
let Clear(color:Color) = SimpleRenderer (fun device -> device.Clear(ClearFlags.ZBuffer ||| ClearFlags.Target, color, 1.0f, 0))
let View(view) = SimpleRenderer (fun device -> device.Transform.View <- view)
let Projection(proj) = SimpleRenderer (fun device -> device.Transform.Projection <- proj)
  

//---------------------------------------------------------------------------
// Loader and Renderer for MeshData
//---------------------------------------------------------------------------

type MeshData= 
  { mesh : Mesh; // Our mesh object in sysmem
    meshMaterials : Material[]; // Materials for our mesh
    meshTextures : Texture[];  } // Textures for our mesh

let LoadMeshDataFromFile((device : Device), file) =         
   // Load the mesh from the specified file.  'materials' is an
   // 'out' parameter: for these F# currently requires that you pass
   // in a value of type 'ref'.
   let materialsRes = ref (null : ExtendedMaterial[]) in
   let mesh = Mesh.FromFile(file, MeshFlags.SystemMemory,device, materialsRes) in
   let meshExtendedMaterials = !materialsRes in

   // Set the ambient color for each material (D3DX does not do 
   // this).  F# uses copy-out/copy-in for this mutation of a .net struct
   let meshMaterials = 
     CompatArray.map 
       (fun (m : ExtendedMaterial) -> 
          let mutable m3D = m.Material3D in
          m3D.Ambient <- m3D.Diffuse;
          m3D) 
       meshExtendedMaterials in 
     
   let meshTextures =
     CompatArray.map
       (fun (m : ExtendedMaterial) -> TextureLoader.FromFile(device, m.TextureFilename)) 
       meshExtendedMaterials in
   { mesh = mesh;
     meshMaterials = meshMaterials;
     meshTextures = meshTextures }


  // The Render function for mesh data
let RenderMeshDataFromFile(file) =
  Renderer(fun device -> 
    let meshData = LoadMeshDataFromFile((device : Device), file) in 
    RendererForDevice(fun () ->
    // Meshes are divided into subsets, one for each material. Render them in
    // a loop
    for i = 0 to meshData.meshMaterials.Length - 1 do
       // Set the material and texture for this subset
       device.Material <- CompatArray.get meshData.meshMaterials i;
       device.SetTexture(0, CompatArray.get meshData.meshTextures i);
            
       // Draw the mesh subset
       meshData.mesh.DrawSubset(i)
    done))
    
    
  