(* (c) Microsoft Corporation 2005-2007.  *)


(*F# 
/// Binary reader.  Read a .NET binary and concert it to Abstract IL data
/// structures.
/// 
/// Notes:
///   - The metadata in the loaded modules will be relative to 
///     those modules, e.g. ScopeRef_local will mean "local to 
///     that module".  You must use [rescope_typ] etc. if you want to include 
///     (i.e. copy) the metadata into your own module. 
///
///   - PDB (debug info) reading/folding:
///     The PDB reader is invoked if you give a PDB path 
///     This indicates if you want to search for PDB files and have the 
///     reader fold them in.  You cannot currently name the pdb file 
///     directly - you can only name the path.  Giving "None" says 
///     "do not read the PDB file even if one exists" 
/// 
///     The debug info appears primarily as I_seqpoint annotations in 
///     the instruction streams.  Unfortunately the PDB information does
///     not, for example, tell you how to map back from a class definition
///     to a source code line number - you will need to explicitly search
///     for a sequence point in the code for one of the methods of the 
///     class.  That is not particularly satisfactory, and it may be
///     a good idea to build a small library which extracts the information
///     you need.  
module Microsoft.Research.AbstractIL.BinaryReader 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
module Il = Microsoft.Research.AbstractIL.IL 
F#*)  

val read_binary: 
    string -> (* Filename of .NET DLL or EXE *) 
    string option -> (* PDB path (note: not the PDB file) *) 
    Il.modul

type readerOptions =
 { pdbPath: string option;
   manager: Il.manager option;
   mscorlib: Il.mscorlib_refs;
   optimizeForMemory: bool  (* normally off, i.e. optimize for startup-path speed *) }

val defaults : readerOptions

(* The following are the same as read_binary, except the *)
(* non-memory resources (i.e. the file handle) associated with *)
(* the read can be recovered by calling close_reader.  Any reamining *)
(* lazily-computed items in the metadata graph returend by modul_of_reader *)
(* will no longer be valid. *)
type binary_reader
val open_binary_reader: string -> readerOptions -> binary_reader
val close_binary_reader: binary_reader -> unit
val modul_of_binary_reader: binary_reader -> Il.modul

(* Open a binary reader, except first copy the entire contents of the binary into *)
(* memory, close the file and ensure any subsequent reads happen from the in-memory store. *)
(* PDB files may not be read with this option. *)
val open_binary_reader_in_memory: string -> readerOptions -> binary_reader

(* report statistics from all reads *)
val report: out_channel -> unit

(* low level functions *)
val sigptr_get_bool :   Bytes.bytes -> int -> bool * int
val sigptr_get_i8 : Bytes.bytes -> int -> Nums.i8 * int
val sigptr_get_i32 : Bytes.bytes -> int -> int32 * int
val sigptr_get_z_i32 :Bytes.bytes -> int -> int32 * int
val sigptr_get_string : int -> Bytes.bytes -> int -> string * int
val sigptr_foldi : (Bytes.bytes -> int -> 'a * int) -> int -> Bytes.bytes -> int -> ('a list * int)

