(* -------------------------------------------------------------------- 
 * Abstract IL Numbers. 
 *
 * (c) Microsoft Corporation. All rights reserved 
 * -------------------------------------------------------------------- *)

(*F# 
module Microsoft.Research.AbstractIL.Internal.Nums 
F#*)

type u8  (*F# = System.Byte F#*)
type u16 (*F# = System.UInt16 F#*)
type i8  (*F# = System.SByte F#*)
type i16 (*F# = System.Int16 F#*)
type u32 (*F# = System.UInt32 F#*)
type i32      = int32
type i64      = int64
type u64 (*F# = System.UInt64 F#*)

(** A type representing the precise IEEE bits of a 32-bit floating point number *)
type ieee32 (*F# = System.Single F#*)

(** A type representing the precise IEEE bits of a 64-bit floating point number *)
type ieee64 (*F# = System.Double F#*)

type unichar (*F# = System.Char F#*)

val u16_to_unichar: u16     -> unichar
val unichar_to_u16: unichar -> u16

(* Widening conversions *)
val u8_to_int: u8 -> int
val i8_to_int: i8 -> int
val u16_to_int: u16 -> int
val i16_to_int: i16 -> int

val i8_to_i32: i8 -> i32
val u8_to_i32: u8 -> i32
val i16_to_i32: i16 -> i32
val u16_to_i32: u16 -> i32
val int_to_i32: int -> i32

val u32_to_i64: u32 -> i64
val int_to_i64: int -> i64
val i32_to_i64: i32 -> i64

(* Narrowing conversions: may be checked and raise exceptions *)
val int_to_i8: int -> i8
val i32_to_i8: i32 -> i8
val i64_to_i8: i64 -> i8

val int_to_u8: int -> u8
val i32_to_u8: i32 -> u8
val i64_to_u8: i64 -> u8

val int_to_i16: int -> i16
val i32_to_i16: i32 -> i16
val i64_to_i16: i64 -> i16

val int_to_u16: int -> u16
val i32_to_u16: i32 -> u16
val i64_to_u16: i64 -> u16

val i32_to_int: i32 -> int
val u64_to_int: u64 -> int
val u32_to_int: u32 -> int
val i64_to_int: i64 -> int

val i64_to_i32: i64 -> i32
val u64_to_u32: u64 -> u32

(* Sign-loss conversions: unchecked and do not raise exceptions.  *)
val i8_to_u8: i8 -> u8
val u8_to_i8: u8 -> i8
val i16_to_u16: i16 -> u16
val u16_to_i16: u16 -> i16
val i32_to_u32: i32 -> u32
val u32_to_i32: u32 -> i32
val i64_to_u64: i64 -> u64
val u64_to_i64: u64 -> i64

(* Floating point conversions *)
val float_to_ieee64: float -> ieee64
val float_to_ieee32: float -> ieee32
val ieee64_to_float: ieee64 -> float
val ieee32_to_float: ieee32 -> float

(*F# val ieee32_to_float32: ieee32 -> float32 F#*)

val int_to_ieee32: int -> ieee32
val int_to_ieee64: int -> ieee64

val i64_to_ieee64: i64 -> ieee64

(* Arithmetic *)
val i32_lt: i32 -> i32 -> bool
val i32_ge: i32 -> i32 -> bool
val i32_max: i32 -> i32 -> i32
val u16_ge: u16 -> u16 -> bool
val u16_sub: u16 -> u16 -> u16
val i32_zero: i32
val u16_zero: u16
val incr_i32: i32 -> i32

(* Raw view of F.P. *)
val ieee64_of_eight_bytes: int -> int -> int -> int -> int -> int -> int -> int -> ieee64
val ieee32_of_four_bytes: int -> int -> int -> int -> ieee32
val bits_to_ieee64: i64 -> ieee64
val bits_to_ieee32: i32 -> ieee32
val ieee64_to_bits: ieee64 -> i64
val ieee32_to_bits: ieee32 -> i32

(* Parsing *)
val string_to_i64: string -> i64
val string_to_i32: string -> i32
val string_to_ieee32 : string -> ieee32

(* to_string *)
val u16_to_string: u16 -> string

(* Output string format *)
val output_u8: out_channel -> u8 -> unit
val output_i8: out_channel -> i8 -> unit
val output_u16: out_channel -> u16 -> unit
val output_i16: out_channel -> i16 -> unit
val output_u32: out_channel -> u32 -> unit
val output_i32: out_channel -> i32 -> unit
val output_u64: out_channel -> u64 -> unit
val output_i64: out_channel -> i64 -> unit
val output_ieee32: out_channel -> ieee32 -> unit
val output_ieee64: out_channel -> ieee64 -> unit
