// (c) Microsoft Corporation 2005-2007.  

#light

module Microsoft.FSharp.Primitives.Basics 

open Microsoft.FSharp.Core
open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
open Microsoft.FSharp.Collections
open Microsoft.FSharp.Core.Operators
#if CLI_AT_MOST_1_1
open Microsoft.FSharp.Compatibility
#else
open System.Collections.Generic
#endif

let ( ** ) x y = System.Math.Pow(x,y)
let abs (x:int) = if x < 0 then -x else x

module PrimArray = 

    let inline length (arr: 'a array) =  (# "ldlen conv.i4" arr : int #)  
    let inline get (arr: 'a array) (n:int) =  arr.[n]
    let inline set (arr: 'a array) (n:int) (x:'a) =  arr.[n] <- x
    let inline zero_create (n:int) = (# "newarr.erasable !0" type ('a) n : 'a array #)


module List = 

    let nonempty x = match x with [] -> false | _ -> true

    let length (l : 'a list)  = l.Length
    let for_all p (l : 'a list) = l.ForAll(p)
    let exists p (l : 'a list) = l.Exists(p)
    let filter p (l : 'a list) = l.Filter(p)
    let rev (l : 'a list) = l.Reverse()
    let map f (l : 'a list) = l.Map(f)
    let mapi f (l : 'a list) = l.MapIndexed(f)
    let iter f (l : 'a list) = l.Iterate(f)
    let iteri f (l : 'a list) = l.IterateIndexed(f)

    let rec find f l = 
        match l with 
        | [] -> not_found()
        | h::t -> if f h then h else find f t

    let rec tryfind f l = 
        match l with 
        | [] -> None 
        | h::t -> if f h then Some h else tryfind f t

    let rec rev_append l1 l2 = 
        match l1 with 
        | [] -> l2
        | h::t -> rev_append t (h::l2)

    // let rec (@) x y = match x with [] -> y | (h::t) -> h :: (t @ y)
    //
    // optimized mutation-based implementation. This code is only valid in fslib, where mutation of private
    // tail cons cells is permitted in carefully written library code.
    let append_to_fresh_cons_tail cons t = 
        let mutable curr = t 
        let mutable cell = cons 
        while nonempty curr do
            let nw = [curr.(::).0] 
            cell.(::).1 <- nw;
            curr <- curr.(::).1;
            cell <- nw
        cell

    // optimized mutation-based implementation. This code is only valid in fslib, where mutation of private
    // tail cons cells is permitted in carefully written library code.
    let set_fresh_cons_tail cons t = 
        cons.(::).1 <- t

    (* let rec (@) x y = match x with [] -> y | (h::t) -> h :: (t @ y)*)
    // optimized mutation-based implementation. This code is only valid in fslib, where mutation of private
    // tail cons cells is permitted in carefully written library code.
    let append l1 l2 = 
        match l1 with
        | [] -> l2
        | (h::t) -> 
        match l2 with
        | [] -> l1
        | _ -> 
          let res = [h] 
          let cons = append_to_fresh_cons_tail res t 
          set_fresh_cons_tail cons l2;
          res

    // optimized mutation-based implementation. This code is only valid in fslib, where mutation of private
    // tail cons cells is permitted in carefully written library code.
    let rec concat_to_fresh_cons_tail cons h1 l = 
        match l with 
        | [] -> set_fresh_cons_tail cons h1
        | h2::t -> concat_to_fresh_cons_tail (append_to_fresh_cons_tail cons h1) h2 t
      
    // optimized mutation-based implementation. This code is only valid in fslib, where mutation of private
    // tail cons cells is permitted in carefully written library code.
    let rec concat_to_empty l = 
        match l with 
        | [] -> []
        | []::t -> concat_to_empty t 
        | (h::t1)::tt2 -> 
            let res = [h] 
            concat_to_fresh_cons_tail res t1 tt2;
            res

    let ie_to_list (e :> IEnumerator<_>) = 
        let mutable res = [] 
        while e.MoveNext() do
            res <- e.Current :: res
        rev res

    let concat (l : #seq<_>) = 
        match ie_to_list (l.GetEnumerator()) with 
        | [] -> []
        | [h] -> h
        | [h1;h2] -> append h1 h2
        | l -> concat_to_empty l

    let rec init_to_fresh_cons_tail cons i n f = 
        if i < n then 
          let cons2 = [f i] 
          set_fresh_cons_tail cons cons2;
          init_to_fresh_cons_tail cons2 (i+1) n f 
        else ()
      
    let init n f = 
        if n = 0 then [] 
        else let res = [f 0] in (init_to_fresh_cons_tail res 1 n f; res)

     
      
    // optimized mutation-based implementation. This code is only valid in fslib, where mutation of private
    // tail cons cells is permitted in carefully written library code.
    let rec partition_to_fresh_cons_tails consL consR p l = 
        match l with 
        | [] -> ()
        | h::t -> 
            let cons' = [h] 
            if p h then 
                set_fresh_cons_tail consL cons';
                partition_to_fresh_cons_tails cons' consR p t
            else 
                set_fresh_cons_tail consR cons';
                partition_to_fresh_cons_tails consL cons' p t
      
    let rec partition_to_fresh_cons_tail_left consL p l = 
        match l with 
        | [] -> []
        | h::t -> 
            let cons' = [h] 
            if p h then 
                set_fresh_cons_tail consL cons';
                partition_to_fresh_cons_tail_left cons'  p t
            else 
                partition_to_fresh_cons_tails consL cons' p t; 
                cons'

    let rec partition_to_fresh_cons_tail_right consR p l = 
        match l with 
        | [] -> []
        | h::t -> 
            let cons' = [h] 
            if p h then 
                partition_to_fresh_cons_tails cons' consR p t; 
                cons'
            else 
                set_fresh_cons_tail consR cons';
                partition_to_fresh_cons_tail_right cons' p t

    let rec partition p l = 
        match l with 
        | [] -> [],[]
        | [h] -> if p h then l,[] else [],l
        | h::t -> 
            let cons = [h] 
            if p h 
            then cons, (partition_to_fresh_cons_tail_left cons p t)
            else (partition_to_fresh_cons_tail_right cons p t), cons
           
    // optimized mutation-based implementation. This code is only valid in fslib, where mutation of private
    // tail cons cells is permitted in carefully written library code.
    let rec unzip_to_fresh_cons_tail cons1a cons1b x = 
        match x with 
        | [] -> ()
        | ((h1,h2)::t) -> 
          let cons2a = [h1] 
          let cons2b = [h2] 
          set_fresh_cons_tail cons1a cons2a;
          set_fresh_cons_tail cons1b cons2b;
          unzip_to_fresh_cons_tail cons2a cons2b t

    // optimized mutation-based implementation. This code is only valid in fslib, where mutation of private
    // tail cons cells is permitted in carefully written library code.
    let rec unzip x = 
        match x with 
        | [] -> [],[]
        | ((h1,h2)::t) -> 
          let res1a = [h1] 
          let res1b = [h2] 
          unzip_to_fresh_cons_tail res1a res1b t; 
          res1a,res1b

    // optimized mutation-based implementation. This code is only valid in fslib, where mutation of private
    // tail cons cells is permitted in carefully written library code.
    let rec unzip3_to_fresh_cons_tail cons1a cons1b cons1c x = 
        match x with 
        | [] -> ()
        | ((h1,h2,h3)::t) -> 
          let cons2a = [h1] 
          let cons2b = [h2] 
          let cons2c = [h3] 
          set_fresh_cons_tail cons1a cons2a;
          set_fresh_cons_tail cons1b cons2b;
          set_fresh_cons_tail cons1c cons2c;
          unzip3_to_fresh_cons_tail cons2a cons2b cons2c t

    // optimized mutation-based implementation. This code is only valid in fslib, where mutation of private
    // tail cons cells is permitted in carefully written library code.
    let rec unzip3 x = 
        match x with 
        | [] -> [],[],[]
        | ((h1,h2,h3)::t) -> 
            let res1a = [h1] 
            let res1b = [h2] 
            let res1c = [h3] 
            unzip3_to_fresh_cons_tail res1a res1b res1c t; 
            res1a,res1b,res1c

    // optimized mutation-based implementation. This code is only valid in fslib, where mutation of private
    // tail cons cells is permitted in carefully written library code.
    let rec zip_to_fresh_cons_tail cons x1 x2 = 
        match x1,x2 with 
        | [],[] -> ()
        | (h1::t1),(h2::t2) -> 
          let cons2 = [(h1,h2)] 
          set_fresh_cons_tail cons cons2;
          zip_to_fresh_cons_tail cons2 t1 t2
        | _ -> invalid_arg "List.zip"

    // optimized mutation-based implementation. This code is only valid in fslib, where mutation of private
    // tail cons cells is permitted in carefully written library code.
    let zip x1 x2 = 
        match x1,x2 with 
        | [],[] -> []
        | (h1::t1),(h2::t2) -> 
            let res = [(h1,h2)] 
            zip_to_fresh_cons_tail res t1 t2; res
        | _ -> invalid_arg "List.zip"

    // optimized mutation-based implementation. This code is only valid in fslib, where mutation of private
    // tail cons cells is permitted in carefully written library code.
    let rec zip3_to_fresh_cons_tail cons x1 x2 x3 = 
        match x1,x2,x3 with 
        | [],[],[] -> ()
        | (h1::t1),(h2::t2),(h3::t3) -> 
            let cons2 = [(h1,h2,h3)] in 
            set_fresh_cons_tail cons cons2;
            zip3_to_fresh_cons_tail cons2 t1 t2 t3
        | _ -> invalid_arg "List.zip3"

    // optimized mutation-based implementation. This code is only valid in fslib, where mutation of private
    // tail cons cells is permitted in carefully written library code.
    let zip3 x1 x2 x3 = 
        match x1,x2,x3 with 
        | [],[],[] -> []
        | (h1::t1),(h2::t2),(h3::t3) -> let res = [(h1,h2,h3)] in zip3_to_fresh_cons_tail res t1 t2 t3; res
        | _ -> invalid_arg "List.zip"

    let to_array (l:'a list) =
        let len = length l 
        let res = (PrimArray.zero_create len : 'a array) 
        let mutable lref = l 
        for i = 0 to len - 1 do 
            PrimArray.set res i lref.(::).0;
            lref <- lref.(::).1
        res

    let of_array (arr:'a array) =
        let len = PrimArray.length arr 
        let mutable res = ([]: 'a list) 
        for i = len - 1 downto 0 do 
            res <- (PrimArray.get arr i) :: res
        res

module Array = 

    let inline geta (arr: 'a[]) (m:int) : 'a nativeptr = (# "ldelema !0" type('a) arr m : 'a nativeptr #)
    let inline length (arr: 'a array) =  PrimArray.length arr
    let inline get (arr: 'a array) (n:int) =  PrimArray.get arr n
    let inline set (arr: 'a array) (n:int) (x:'a) =  PrimArray.set arr n x
    let inline zero_create (n:int) = PrimArray.zero_create n

    let make  (n:int) (x:'a) =
        let arr = (zero_create n : 'a array) 
        for i = 0 to n - 1 do 
            (set arr i x)
        arr

    let create (n:int) (x:'a) = make n x

    let init (n:int) (f: int -> 'a) = 
        let arr = (zero_create n : 'a array)  
        for i = 0 to n - 1 do 
            set arr i (f i)
        arr

    let map (f: 'a -> 'b) (arr:'a array) =
        let len = length arr 
        let res = (zero_create len : 'b array) 
        for i = 0 to len - 1 do 
            set res i (f (get arr i))
        res

    let sub (arr:'a array) (start:int) (len:int) =
        let res = (zero_create len : 'a array)  
        for i = 0 to len - 1 do 
            (set res i (get arr (start + i) : 'a))
        res

    let blit (arr1:'a array) (start1:int) (arr2: 'a array) (start2:int) (len:int) =
        for i = 0 to len - 1 do 
          (set arr2 (start2+i) (get arr1 (start1 + i) : 'a))

    let copy (arr:'a array) =
        let len = length arr 
        let res = (zero_create len : 'a array)  
        blit arr 0 res 0 len;
        res

    let fold_right (f : 'a -> 'b -> 'b) (arr:'a array) (acc: 'b) =
        let f = OptimizedClosures.FastFunc2<_,_,_>.Adapt(f)
        let mutable res = acc 
        let len = length arr 
        for i = len - 1 downto 0 do 
            res <- f.Invoke(arr.[i],res)
        res

    let fold_left (f : 'a -> 'b -> 'a) (acc: 'a) (arr:'b array) =
        let f = OptimizedClosures.FastFunc2<_,_,_>.Adapt(f)
        let mutable state = acc 
        let len = length arr 
        for i = 0 to len - 1 do 
            state <- f.Invoke(state,arr.[i])
        state

    let of_list x = List.to_array x
    let to_list x = List.of_array x


let string_of_int (x:int) = x.ToString()
let (@) l1 l2 = List.append l1 l2

module Char = 
    let code (c:char) = (# "" c : int #)
    let chr (n:int) =  (# "conv.u2" n : char #)

module CompatArray  = 

    (* Define the primitive operations. *)
    (* Note: the "type" syntax is for the type parameter for inline *)
    (* polymorphic IL. This helps the compiler inline these fragments, *)
    (* i.e. work out the correspondence between IL and F# type variables. *)
    let
    #if CLI_AT_MOST_1_1
     inline
    #endif
     length (arr: 'a[]) =  (# "ldlen conv.i4" arr : int #)  

    let
    #if CLI_AT_MOST_1_1
     inline
    #endif
     get (arr: 'a[]) (n:int) =  (# "ldelem.any !0" type ('a) arr n : 'a #)

    #if CLI_AT_LEAST_2_0
    let set (arr: 'a[]) (n:int) (x:'a) =  (# "stelem.any !0" type ('a) arr n x #)
    #else
    // The above do not work for value types when compiling without generics, since 
    // ILX is not able to erase the use of stelem.any
    let inline set (arr: 'a[]) (n:int) (x:'a) =  (# "stobj !0" type ('a) (# "ldelema !0" type ('a) arr n : System.UIntPtr #) x #)
    #endif

    let
    #if CLI_AT_MOST_1_1
     inline
    #endif
     zero_create (n:int) = (# "newarr !0" type ('a) n : 'a[] #)

    let
    #if CLI_AT_MOST_1_1
     inline
    #endif
     init (n:int) f = 
      let arr = zero_create n  
      for i = 0 to n - 1 do 
          set arr i (f i)
      arr

    let
    #if CLI_AT_MOST_1_1
     inline
    #endif
     sub arr (start:int) (len:int) =
      let res = zero_create len   in
      for i = 0 to len - 1 do 
        set res i (get arr (start + i))
      done;
      res

    let
    #if CLI_AT_MOST_1_1
     inline
    #endif
     blit arr1 (start1:int) arr2 (start2:int) (len:int) =
      for i = 0 to len - 1 do 
        set arr2 (start2+i) (get arr1 (start1 + i))
      done


module Bytearray = 

    let length (arr: byte[]) =  CompatArray.length arr
    let get (arr: byte[]) (n:int) =  (# "ldelem.u1" arr n : byte #)  
    let set (arr: byte[]) (n:int) (x:byte) =  (# "stelem.i1" arr n x #)  
    let zero_create (n:int) : byte[]= CompatArray.zero_create n
    let init (n:int) (f: int -> byte) =  CompatArray.init n f
    let sub (arr:byte[]) (start:int) (len:int) = CompatArray.sub arr start len
    let blit (arr1:byte[]) (start1:int) (arr2: byte[]) (start2:int) (len:int) = CompatArray.blit arr1 start1 arr2 start2 len

open System.Runtime.InteropServices

module NativeOps = 
    let inline of_nativeint (x:nativeint) = (# "" x : 'a nativeptr    #)
    let inline to_nativeint (x: 'a nativeptr)   = (# "" x : nativeint #)
    let inline of_ilsigptr(x: 'a nativeptr) = (# "" x : 'a nativeptr    #)
    let inline to_ilsigptr (x: 'a nativeptr)   = (# "" x : 'a nativeptr #)
    let inline pinUnscoped (obj: obj) =  GCHandle.Alloc(obj,GCHandleType.Pinned) 
    let inline pinAny (obj: obj) f = 
        let gch = pinUnscoped obj 
        try f gch
        finally
            gch.Free()

module RangeOps = 

    type range = int * int

    let inline foldR f z ((a,b):range) = 
        let mutable res = z in
        for i = a to b do
            res <- f res i
        res
    let inline sumfR f ((a,b):range) =
        let mutable res = 0.0 in
        for i = a to b do
            res <- res + f i
        res
      
