(*==========================================================================
 * (c) Microsoft Corporation 2005-2007.  
 *
 *=========================================================================*)

module Microsoft.FSharp.Compatibility.OCaml.Hashtbl
open Microsoft.FSharp.Compatibility.OCaml
open Microsoft.FSharp.Compatibility.OCaml.Pervasives
open Microsoft.FSharp.Collections
open Microsoft.FSharp.Core
#if CLI_AT_LEAST_2_0
open System.Collections.Generic
#else
open Microsoft.FSharp.Compatibility
#endif

type HashTable<'key,'a> = Microsoft.FSharp.Collections.HashMultiMap<'key,'a> 
type ('key,'a) t = Microsoft.FSharp.Collections.HashMultiMap<'key,'a> 

let inline create (n:int) = HashTable.Create n

let add (t:HashMultiMap<'key,'a>) x y = t.Add(x,y)
let of_list (l : _ list) = 
     let t = create (List.length l) in 
     List.iter (fun (x,y) -> add t x y) l; 
     t
let of_seq l = 
    let arr =Array.of_seq l in 
    let t = create (Array.length arr) in Array.iter (fun (x,y) -> add t x y) arr; 
    t
let of_IEnumerable l = of_seq l
let copy (t:HashMultiMap<'key,'a>)  = t.Copy()
let find (t:HashMultiMap<'key,'a>)  x = t.[x]
let tryfind (t:HashMultiMap<'key,'a>)  x = t.TryFind x
let find_all (t:HashMultiMap<'key,'a>)  x = t.FindAll x
let mem (t:HashMultiMap<'key,'a>)  x =  t.Contains x
let remove (t:HashMultiMap<'key,'a>)  x = t.Remove x
let replace (t:HashMultiMap<'key,'a>)  x y = t.Replace(x,y)
let fold f (t:HashMultiMap<'key,'a>)  c = t.Fold f c

let clear (t:HashMultiMap<'key,'a>)  = t.Clear ()

let iter f (t:HashMultiMap<'key,'a>)  = t.Iterate f

let stats s p os (t:HashMultiMap<'key,'a>)  = 
  let r = ref 0 in 
  t.Iterate(fun _ _ -> incr r);
  let longest = t.GetLongestChain() in
  Printf.fprintf os "Statistics for Hashtable %s:\n" s;
  Printf.fprintf os "  #elements = %d\n" !r;
  Printf.fprintf os "  maximum chain size = %d\n" (List.length longest);
  Printf.fprintf os "  longest chain = \n";
  flush os;
  List.iter (fun (x,y) -> Printf.fprintf os "  bucket: %a\n" p x) longest
  
(* --------------------------------------------------------------------
 * Hashtables that carry equality/hash functions.
 * -------------------------------------------------------------------- *)

let gen_stats (stats: Microsoft.FSharp.Collections.HashStats) s f os = 
  Printf.fprintf os "Statistics for Hashtbl %s:\n" s;
  Printf.fprintf os "  #tables = %d\n" stats.NumTables;
  Printf.fprintf os "  #resizes = %d\n" stats.NumResizes;
  Printf.fprintf os "  max chain length on resize = %d\n" stats.LongestChainLength

(* --------------------------------------------------------------------
 * Componentized Hash Tables
 * -------------------------------------------------------------------- *)


type Provider<'key,'a,'tag> 
     when 'tag :> IEqualityComparer<'key> =
  { create  : int -> Tagged.HashMultiMap<'key,'a,'tag>;
    clear   : Tagged.HashMultiMap<'key,'a,'tag> -> unit;
    add     : Tagged.HashMultiMap<'key,'a,'tag> -> 'key -> 'a -> unit;
    copy    : Tagged.HashMultiMap<'key,'a,'tag> -> Tagged.HashMultiMap<'key,'a,'tag>;
    find    : Tagged.HashMultiMap<'key,'a,'tag> -> 'key -> 'a;
    find_all: Tagged.HashMultiMap<'key,'a,'tag> -> 'key -> 'a list;
    tryfind : Tagged.HashMultiMap<'key,'a,'tag> -> 'key -> 'a option;
    mem     : Tagged.HashMultiMap<'key,'a,'tag> -> 'key -> bool;
    remove  : Tagged.HashMultiMap<'key,'a,'tag> -> 'key -> unit;
    replace : Tagged.HashMultiMap<'key,'a,'tag> -> 'key -> 'a -> unit;
    iter    : ('key -> 'a -> unit) -> Tagged.HashMultiMap<'key,'a,'tag> -> unit;
    fold    : 'b. ('key -> 'a -> 'b -> 'b) -> Tagged.HashMultiMap<'key,'a,'tag> -> 'b -> 'b;
    
    /// Provider.stats prints out statistics for all the hash tables ever created
    /// related to this provider.  It also prints out the keys in the longest 
    /// chain ever stored in any hash table managed by this component, using 
    /// the given printing function.  This is useful for debugging poor hash 
    /// functions.
    stats   : string -> (out_channel -> 'key -> unit) -> out_channel -> unit }

type Provider<'key,'a> = Provider<'key,'a,IEqualityComparer<'key>>

let MakeTagged (ops : 'tag :> IEqualityComparer<'key>) : Provider<'key,'a,'tag> = 
  let stats = HashStats.Create() in 
  { create=(fun n -> Tagged.HashMultiMap<'key,'a,'tag>.Create(ops,stats,n));
    clear=(fun c -> c.Clear());
    add=(fun c x y -> c.Add(x,y));
    copy=(fun c -> c.Copy());
    find=(fun c x -> c.Find(x));
    find_all=(fun c x -> c.FindAll(x));
    tryfind=(fun c x -> c.TryFind(x));
    mem=(fun c x -> c.Contains(x));
    remove=(fun c x -> c.Remove(x));
    replace=(fun c x y -> c.Replace(x,y));
    iter=(fun f c -> c.Iterate(f));
    fold=(fun f c acc -> c.Fold f acc);
    stats=gen_stats stats }

let Make (hash,eq) = MakeTagged (HashIdentity.Custom hash eq)

let hash x = LanguagePrimitives.GenericHash x
let hashq x = LanguagePrimitives.PhysicalHash x

type CHashOps<'key,'a> = Provider<'key,'a>
type CHashTable<'key,'a> = Tagged.HashMultiMap<'key,'a>
