(* (c) Microsoft Corporation 2005-2007.  *)

(*-------------------------------------------------------------------------
 * Derived expression manipulation and construction functions.
 *------------------------------------------------------------------------- *)

(*F# 
module Microsoft.FSharp.Compiler.Tastops 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
open Microsoft.FSharp.Compiler 
module Il = Microsoft.Research.AbstractIL.IL 
module Illib = Microsoft.Research.AbstractIL.Internal.Library
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics  
module Ilx    = Microsoft.Research.AbstractIL.Extensions.ILX 
module Ilprint = Microsoft.Research.AbstractIL.AsciiWriter 
module Ccuthunk = Microsoft.FSharp.Compiler.CcuThunk 
F#*)

open Il
open Illib
open Range
open Ast
open Tast
open Printf
open List
open Ildiag
open Ilx
open Lib
open Env
open Ccuthunk
open Layout
open Nums

(*---------------------------------------------------------------------------
!* Standard orderings, e.g. for order set/map keys
 *--------------------------------------------------------------------------*)

(* REVIEW: rename as val_spec_order and val_ref_order (or lvref_order)*)
let val_spec_order' v1 v2 = Pervasives.compare (stamp_of_val v1)   (stamp_of_val v2) 
let val_spec_order  v1 v2 = Pervasives.compare (stamp_of_lvref v1) (stamp_of_lvref v2) 
let tycon_spec_order  v1 v2 = Pervasives.compare (stamp_of_tycon v1) (stamp_of_tycon v2) 
let rfref_order  (RFRef(tcref1,nm1)) (RFRef(tcref2,nm2)) = 
     let c = tycon_spec_order (deref_tycon tcref1) (deref_tycon tcref2) in 
     if c <> 0 then c else 
     Pervasives.compare nm1 nm2
let ucref_order  (UCRef(tcref1,nm1)) (UCRef(tcref2,nm2)) = 
     let c = tycon_spec_order (deref_tycon tcref1) (deref_tycon tcref2) in 
     if c <> 0 then c else 
     Pervasives.compare nm1 nm2

(*---------------------------------------------------------------------------
!* Make some common types
 *--------------------------------------------------------------------------*)

let mk_fun_ty d r = TType_fun (d,r)
let (-->) d r = mk_fun_ty d r
let mk_forall_ty d r = TType_forall (d,r)
let try_mk_forall_ty d r = if isNil d then r else mk_forall_ty d r
let (+->) d r = try_mk_forall_ty d r
let mk_tuple_ty l = TType_tuple l
let mk_iterated_fun_ty dl r = fold_right (-->) dl r

let fake_mk_tupled_ty tys = 
  match tys with 
  | [] -> failwith "fake_mk_tupled_ty" 
  | [h] -> h 
  | _ -> mk_tuple_ty tys

let type_of_lambda_arg vs = fake_mk_tupled_ty (map type_of_val vs)
let mk_multi_lambda_ty vs rty = mk_fun_ty (type_of_lambda_arg vs) rty 
let mk_lambda_ty tps tys rty = try_mk_forall_ty tps (mk_iterated_fun_ty tys rty)

(*---------------------------------------------------------------------------
!* The tables of modules and submodules: TODO - remove this use of mutation
 *------------------------------------------------------------------------- *)

let ensure_fslib_has_submodul_at ccu path (CompPath(_,cpath)) xml =
  let scoref = (scoref_of_ccu ccu) in 
  let rec loop prior_cpath (path:ident list) cpath modul =
    let mtype = mtyp_of_modul modul in 
    match path,cpath with 
    | (hpath::tpath),((_,isModule)::tcpath)  -> 
      if not (mtyp_has_submodul hpath.idText mtype) then begin
        let smodul = new_mspec (Some(CompPath(scoref,prior_cpath))) taccessPublic hpath xml [] (notlazy (empty_mtype isModule)) in
        mtype.mtyp_submoduls <-  Map.add hpath.idText smodul (submoduls_of_mtyp mtype);
      end;
      let modul = (mtyp_get_submodul hpath.idText mtype) in 
      loop (prior_cpath@[(hpath.idText,Namespace)]) tpath tcpath modul 
    | _ -> () in
  loop [] path cpath (top_modul_of_ccu ccu)


(*---------------------------------------------------------------------------
!* Primitive destructors
 *------------------------------------------------------------------------- *)

let rec strip_expr e = 
  match e with 
  | TExpr_link eref -> strip_expr !eref
  | _ -> e    

let discrim_of_case (TCase(d,_)) = d
let dest_of_case (TCase(_,d)) = d
let mk_case (a,b) = TCase(a,b)
  

let is_tuple e = match e with TExpr_op(TOp_tuple,_,_,_) -> true | _ -> false
let try_dest_tuple e = match e with TExpr_op(TOp_tuple,_,es,_) -> es | _ -> [e]

(*---------------------------------------------------------------------------
!* Debug info for expressions
 *------------------------------------------------------------------------- *)

let rec range_of_expr x =
  match x with
  | TExpr_val (_,_,m) | TExpr_op (_,_,_,m)   | TExpr_hole (m,_) | TExpr_const (_,m,_) | TExpr_quote (_,_,m,_)
  | TExpr_obj (_,_,_,_,_,_,m,_) | TExpr_app(_,_,_,_,m) | TExpr_seq (_,_,_,m) 
  | TExpr_static_optimization (_,_,_,m) | TExpr_lambda (_,_,_,_,m,_,_) 
  | TExpr_tlambda (_,_,_,m,_,_)| TExpr_tchoose (_,_,m) | TExpr_letrec (_,_,m,_) | TExpr_let (_,_,m,_) | TExpr_match (_,_,_,m,_,_)
    -> m
  | TExpr_link(eref) -> range_of_expr !eref

(*---------------------------------------------------------------------------
!* Build nodes in decision graphs
 *------------------------------------------------------------------------- *)

let prim_mk_match(exprm,tree,targets,matchm,ty) = TExpr_match (exprm,tree,targets,matchm,ty,new_cache())

module MatchBuilder = struct

    type match_builder = { expr_mark: Range.range; mutable targets: dtree_target list }
    let create exprm = { expr_mark=exprm; targets = [] }
    let add_target builder tg = 
      let n = List.length builder.targets in 
      builder.targets <- tg :: builder.targets;
      n

    let add_and_mk_result_target builder e = TDSuccess([], add_target builder (TTarget([],e)))

    let close_targets mbuilder = List.rev mbuilder.targets

    let close dtree mbuilder m ty = prim_mk_match  (mbuilder.expr_mark,dtree,Array.of_list (close_targets mbuilder),m,ty)

end

let mk_bool_switch m g t e = TDSwitch(g,[TCase(TTest_const(TConst_bool(true)),t)],Some e,m)

let mk_cond m ty e1 e2 e3 = 
  let mbuilder = MatchBuilder.create m in
  let dtree = mk_bool_switch m e1 (MatchBuilder.add_and_mk_result_target mbuilder e2) (MatchBuilder.add_and_mk_result_target mbuilder e3) in 
  MatchBuilder.close dtree mbuilder m ty

(*---------------------------------------------------------------------------
!* These make local/non-local references to values according to whether
 * the item is globally stable ("published") or not.
 *------------------------------------------------------------------------- *)

let mk_local_vref   (v:val_spec) = mk_local_ref v
let mk_local_modref (v:modul_spec) = mk_local_ref v
let mk_local_tcref  (v:tycon_spec) = mk_local_ref v
let mk_local_ecref  (v:tycon_spec) = mk_local_ref v

let mk_nonlocal_modref nlpath nm : modul_ref = mk_nonlocal_ref nlpath (encode_modref_name nm)
let mk_nonlocal_ccu_top_modref ccu x : modul_ref = mk_nonlocal_modref (nlpath_of_ccu ccu) (name_of_modul x)
let mk_nonlocal_ccu_top_tcref ccu x : tycon_ref = mk_nonlocal_ref (nlpath_of_ccu ccu) (name_of_tycon x)

let mk_vref_in_modref   (x:modul_ref) y : val_ref = mk_subref x (name_of_val y)   y
let mk_ecref_in_modref  (x:modul_ref) y : tycon_ref  = mk_subref x (name_of_tycon y)  y
let mk_modref_in_modref (x:modul_ref) y : modul_ref = mk_subref x (encode_modref_name (name_of_modul y)) y
let mk_tcref_in_modref  (x:modul_ref) y : tycon_ref = mk_subref x (name_of_tycon y) y
let mk_rfref_in_modref (x:modul_ref) tycon (rf:ident) : recdfield_ref = mk_rfref (mk_tcref_in_modref x tycon) rf.idText 
let mk_tcref_in_tcref  (x:tycon_ref) y : tycon_ref = mk_subref x (name_of_tycon y) y

(*---------------------------------------------------------------------------
!* Primitive constructors
 *------------------------------------------------------------------------- *)

let expr_for_vref m vref =  TExpr_val(vref,NormalValUse,m)
let expr_for_val m v =  expr_for_vref m (mk_local_vref v)
let gen_mk_local m s ty mut compgen =
    let thisv = new_vspec(ident(s,m),ty,mut,compgen,None,None,taccessPublic,ValNotInRecScope,None,NormalVal,[],OptionalInline,emptyXMLDoc,false,false,false,false,None,ParentNone) in 
    thisv,expr_for_val m thisv

let mk_local         m s ty = gen_mk_local m s ty Immutable false
let mk_compgen_local m s ty = gen_mk_local m s ty Immutable true
let mk_mut_compgen_local m s ty = gen_mk_local m s ty Mutable true
let mk_mut_local m s ty = 
    let thisv,thise = gen_mk_local m s ty Mutable false in
    thisv,mk_local_vref thisv,thise


(* Type gives return type.  For type-lambdas this is the formal return type. *)
let mk_multi_lambda m vs (b,rty) = TExpr_lambda (new_uniq(), None,vs,b,m, rty, new_cache ())
let mk_basev_multi_lambda m basevopt vs (b,rty) = TExpr_lambda (new_uniq(), basevopt,vs,b,m, rty, new_cache ())
let mk_lambda m v (b,rty) = mk_multi_lambda m [v] (b,rty)
let mk_tlambda m vs (b,tau_ty) = match vs with [] -> b | _ -> TExpr_tlambda (new_uniq(), vs,b,m,tau_ty, new_cache ())
let mk_tchoose m vs b = match vs with [] -> b | _ -> TExpr_tchoose (vs,b,m)

let mk_lambdas m tps vs (b,rty) = 
  mk_tlambda m tps (List.fold_right (fun v (e,ty) -> mk_lambda m v (e,ty), (type_of_val v) --> ty) vs (b,rty))
let mk_multi_lambdas_core m vsl (b,rty) = 
  List.fold_right (fun v (e,ty) -> mk_multi_lambda m v (e,ty), type_of_lambda_arg v --> ty) vsl (b,rty)
let mk_multi_lambdas m tps vsl (b,rty) = 
  mk_tlambda m tps (mk_multi_lambdas_core m vsl (b,rty) )

let mk_basev_multi_lambdas_core m basevopt vsl (b,rty) = 
    match basevopt with
    | None -> mk_multi_lambdas_core m vsl (b,rty)
    | _ -> 
        match vsl with 
        | [] -> error(InternalError("mk_basev_multi_lambdas_core: can't attach a basev to a non-lambda expression",m))
        | h::t -> 
            let b,rty = mk_multi_lambdas_core m t (b,rty) in
            (mk_basev_multi_lambda m basevopt h (b,rty), (type_of_lambda_arg h --> rty))
        
let mk_basev_multi_lambdas m tps basevopt vsl (b,rty) = 
  mk_tlambda m tps (mk_basev_multi_lambdas_core m basevopt vsl (b,rty) )

let mk_multi_lambda_bind v m tps vsl (b,rty) = TBind(v,mk_multi_lambdas m tps vsl (b,rty))

let mk_bind v e = TBind(v,e)
let mk_binds vs es = 
  if List.length vs <> List.length es then failwith "mk_binds: invalid argument";
  map2 mk_bind vs es
(* n.b. type gives type of body *)
let mk_let_bind m bind body = TExpr_let(bind,body, m, new_cache())
let mk_lets_bind m binds body = list_fold_right (mk_let_bind m) binds body 
let mk_let m v x body = mk_let_bind m (mk_bind v x) body
let mk_lets m vs xs body = mk_lets_bind m (mk_binds vs xs) body

let mk_let_typed m v x (body,ty) = mk_let m v x body,ty

let mk_letrec_binds m binds body = if isNil binds then body else TExpr_letrec(binds,body, m, new_cache())
let mk_letrec_binds_typed m binds (body,ty) =  mk_letrec_binds m binds body, ty



(*-------------------------------------------------------------------------
 * Type schemes...
 *------------------------------------------------------------------------- *)
  
type typ_scheme   = 
    TypeScheme of 
        typars (* the truly generalized type parameters *)
      * typars (* free choice type parameters from a recursive block where this value only generalizes a subsest of the overall set of type parameters generalized *)
      * typ    (* the 'tau' type forming the body of the generalized type *)
  
let mk_poly_bind_rhs m typeScheme bodyExpr = 
    let (TypeScheme(generalizedTypars,freeChoiceTypars,tauType)) = typeScheme in
    mk_tlambda m generalizedTypars (mk_tchoose m freeChoiceTypars bodyExpr, tauType)

let is_being_generalized tp typeScheme = 
    let (TypeScheme(generalizedTypars,_,_)) = typeScheme in
    gen_mem tpspec_eq tp generalizedTypars

(*-------------------------------------------------------------------------
 * Build conditional expressions...
 *------------------------------------------------------------------------- *)

let mk_lazy_and g m e1 e2 = mk_cond m g.bool_ty e1 e2 (TExpr_const(TConst_bool false,m,g.bool_ty))
let mk_lazy_or g m e1 e2 = mk_cond m g.bool_ty e1 (TExpr_const(TConst_bool true,m,g.bool_ty)) e2

let mk_byref_typ g ty = TType_app (g.byref_tcr, [ty])
let mk_il_arr_ty g n ty = 
  if n = 1 then TType_app (g.il_arr1_tcr, [ty]) 
  else if n = 2 then TType_app (g.il_arr2_tcr, [ty]) 
  else if n = 3 then TType_app (g.il_arr3_tcr, [ty]) 
  else if n = 4 then TType_app (g.il_arr4_tcr, [ty]) 
  else failwith "F# supports a maxiumum .NET array dimension of 4"

let mk_coerce(e,to_ty,m,from_ty) = TExpr_op(TOp_coerce,[to_ty;from_ty],[e],m)

let mk_asm(code,tinst,args,rettys,m) = TExpr_op(TOp_asm(code,rettys),tinst,args,m)

let mk_constr(uc,tinst,args,m) = TExpr_op(TOp_uconstr uc,tinst,args,m)
let mk_exnconstr(uc,args,m) = TExpr_op(TOp_exnconstr uc,[],args,m)

let mk_tuple_field_get(e,tinst,i,m) = TExpr_op(TOp_tuple_field_get(i), tinst, [e],m)

let mk_recd_field_get_via_expra(e,fref,tinst,m) = TExpr_op(TOp_field_get(fref), tinst, [e],m)
let mk_recd_field_get_addr_via_expra(e,fref,tinst,m) = TExpr_op(TOp_field_get_addr(fref), tinst, [e],m)
let mk_recd_field_set_via_expra(e1,fref,tinst,e2,m) = TExpr_op(TOp_field_set(fref), tinst, [e1;e2],m)

let mk_static_rfield_get_addr(fref,tinst,m) = TExpr_op(TOp_field_get_addr(fref), tinst, [],m)
let mk_static_rfield_get(fref,tinst,m) = TExpr_op(TOp_field_get(fref), tinst, [],m)
let mk_static_rfield_set(fref,tinst,e,m) = TExpr_op(TOp_field_set(fref), tinst, [e],m)

let mk_uconstr_tag_get(e1,cref,tinst,m) = TExpr_op(TOp_constr_tag_get(cref), tinst, [e1],m)
let mk_uconstr_field_get(e1,cref,tinst,j,m) = TExpr_op(TOp_constr_field_get(cref,j), tinst, [e1],m)
let mk_uconstr_field_set(e1,cref,tinst,j,e2,m) = TExpr_op(TOp_constr_field_set(cref,j), tinst, [e1;e2],m)

let mk_exnconstr_field_get(e1,ecref,j,m) = TExpr_op(TOp_exnconstr_field_get(ecref,j), [],[e1],m)
let mk_exnconstr_field_set(e1,ecref,j,e2,m) = TExpr_op(TOp_exnconstr_field_set(ecref,j), [],[e1;e2],m)

let mk_dummy_lambda g (e,ety) = 
    let m = (range_of_expr e) in 
    mk_lambda m (fst (mk_compgen_local m "$dummy" g.unit_ty)) (e,ety)
                           
let mk_while       g (e1,e2,m)             = TExpr_op(TOp_while      ,[]  ,[mk_dummy_lambda g (e1,g.bool_ty);mk_dummy_lambda g (e2,g.unit_ty)],m)
let mk_for         g (v,e1,dir,e2,e3,m)    = TExpr_op(TOp_for dir    ,[]  ,[mk_dummy_lambda g (e1,g.int_ty) ;mk_dummy_lambda g (e2,g.int_ty);mk_lambda (range_of_expr e3) v (e3,g.unit_ty)],m)
let mk_try_catch   g (e1,vf,ef,vh,eh,m,ty) = TExpr_op(TOp_try_catch  ,[ty],[mk_dummy_lambda g (e1,ty)       ;mk_lambda (range_of_expr ef) vf (ef,ty);mk_lambda (range_of_expr eh) vh (eh,ty)],m)
let mk_try_finally g (e1,e2,m,ty)          = TExpr_op(TOp_try_finally,[ty],[mk_dummy_lambda g (e1,ty)       ;mk_dummy_lambda g (e2,g.unit_ty)],m)

let mk_ilzero (m,ty) = TExpr_const(TConst_default,m,ty) 

(*-------------------------------------------------------------------------
 * Tuples are eliminated by the F#-to-ILX code generator.  They
 * are compiled to a finite set of record types defined in env0.
 *------------------------------------------------------------------------- *)

let maxTuple = 7
let goodTupleFields = maxTuple-1
let rec split_after_acc n l1 l2 = if n <= 0 then List.rev l1,l2 else split_after_acc (n-1) ((hd l2):: l1) (tl l2) 
let split_after n l = split_after_acc n [] l

let compiled_tuple_tcref g tys = 
  let n = List.length tys in 
  if      n = 2 then g.tuple2_tcr
  else if n = 3 then g.tuple3_tcr
  else if n = 4 then g.tuple4_tcr
  else if n = 5 then g.tuple5_tcr
  else if n = 6 then g.tuple6_tcr
  else if n = 7 then g.tuple7_tcr
  else failwith "compiled_tuple_tcref"

let rec compiled_tuple_ty g tys = 
  let n = List.length tys in 
  if n <= 1 then failwith "compiled_tuple_ty"
  else if n <= maxTuple then TType_app (compiled_tuple_tcref g tys, tys)
  else 
    let a,b = split_after (maxTuple - 1) tys in 
    compiled_tuple_ty g (a@[compiled_tuple_ty g b])

let rec compiled_mk_tuple g (argtys,args,m) = 
  let n = List.length argtys in 
  if n <= 1 then failwith "compiled_mk_tuple"
  else if n <= maxTuple then  (compiled_tuple_tcref g argtys, argtys, args, m)
  else
    let argtysA,argtysB = split_after goodTupleFields argtys in
    let argsA,argsB = split_after (goodTupleFields) args in
    let a,b,c,d = compiled_mk_tuple g (argtysB, argsB, m) in
    compiled_mk_tuple g (argtysA @ [compiled_tuple_ty g argtysB],
                         argsA @ [TExpr_op(TOp_recd (RecdExpr,a),b,c,d)],
                         m)

let rec compiled_get_tuple_field g (e,tys,n,m) = 
  let ar = List.length tys in 
  if ar <= 1 then failwith "compiled_get_tuple_field"
  else if ar <= maxTuple then 
    let tcr' = compiled_tuple_tcref g tys in 
    let f = rfield_of_tycon_by_idx (deref_tycon tcr') n in 
    let fref = rfref_of_rfield tcr' f in 
    (e, fref, tys, m)
  else 
    let tysA,tysB = split_after (goodTupleFields) tys in
    let tys' = tysA@[compiled_tuple_ty g tysB] in 
    let tcr' = compiled_tuple_tcref g tys' in 
    let f = rfield_of_tycon_by_idx (deref_tycon tcr') (min n goodTupleFields) in 
    let fref = rfref_of_rfield tcr' f in 
    if n < goodTupleFields then (e, fref, tys', m)
    else compiled_get_tuple_field g (mk_recd_field_get_via_expra (e,fref,tys',m),tysB,n-goodTupleFields,m)


(*--------------------------------------------------------------------------
!* Maps tracking extra information for values
 *-------------------------------------------------------------------------*)

type 'a vspec_map = VSpecMap of 'a Imap.t
let vspec_map_find (v: local_val_ref) (VSpecMap m) = Imap.find (stamp_of_lvref v) m
let vspec_map_tryfind (v: local_val_ref) (VSpecMap m) = Imap.tryfind (stamp_of_lvref v) m
let vspec_map_mem v (VSpecMap m) = Imap.mem (stamp_of_lvref v) m
let vspec_map_add v x (VSpecMap m) = VSpecMap (Imap.add (stamp_of_lvref v) x m)
let vspec_map_remove v (VSpecMap m) = VSpecMap (Imap.remove (stamp_of_lvref v) m)
let vspec_map_empty () = VSpecMap (Imap.empty ())
let vspec_map_of_list vs = List.fold_right (fun (x,y) acc -> vspec_map_add x y acc) vs (vspec_map_empty()) 

type 'a vspec_mmap = 'a list vspec_map
let vspec_mmap_find v (m: 'a vspec_mmap) = if vspec_map_mem v m then vspec_map_find v m else []
let vspec_mmap_add v x (m: 'a vspec_mmap) = vspec_map_add v (x :: vspec_mmap_find v m) m
let vspec_mmap_empty () : 'a vspec_mmap = vspec_map_empty()

type 'a typar_map = TPMap of 'a Imap.t
let tpmap_find (v: local_typar_ref) (TPMap m) = Imap.find (stamp_of_tpref v) m
let tpmap_mem v (TPMap m) = Imap.mem (stamp_of_tpref v) m
let tpmap_add v x (TPMap m) = TPMap (Imap.add (stamp_of_tpref v) x m)
let tpmap_empty () = TPMap (Imap.empty ())

type 'a tcref_map = TCRefMap of 'a Imap.t
let tcref_map_find (v: tycon_ref) (TCRefMap m) = Imap.find (stamp_of_tcref v) m
let tcref_map_tryfind (v: tycon_ref) (TCRefMap m) = Imap.tryfind (stamp_of_tcref v) m
let tcref_map_mem v (TCRefMap m) = Imap.mem (stamp_of_tcref v) m
let tcref_map_add v x (TCRefMap m) = TCRefMap (Imap.add (stamp_of_tcref v) x m)
let tcref_map_empty () = TCRefMap (Imap.empty ())
let tcref_map_is_empty (TCRefMap m) = Zmap.is_empty m
let tcref_map_of_list vs = List.fold_right (fun (x,y) acc -> tcref_map_add x y acc) vs (tcref_map_empty()) 

type 'a tcref_mmap = 'a list tcref_map
let tcref_mmap_find v (m: 'a tcref_mmap) = if tcref_map_mem v m then tcref_map_find v m else []
let tcref_mmap_add v x (m: 'a tcref_mmap) = tcref_map_add v (x :: tcref_mmap_find v m) m
let tcref_mmap_empty () : 'a tcref_mmap = tcref_map_empty()

(*--------------------------------------------------------------------------
!* Substitute for type variables and remap type constructors 
 *-------------------------------------------------------------------------*)

type typar_inst = (local_typar_ref * typ) list
type tpenv = typar_inst

type tcref_remap = tycon_ref tcref_map

let empty_tpenv = ([] : tpenv)
let empty_tpinst = ([] : typar_inst)

type tyenv =
    { tpinst : typar_inst;
      tcref_remap : tcref_remap }
      
let empty_tcref_remap : tcref_remap = tcref_map_empty()
let empty_tyenv = { tpinst = empty_tpinst; tcref_remap =empty_tcref_remap }

let tyenv_is_empty tyenv = isNil tyenv.tpinst && tcref_map_is_empty tyenv.tcref_remap 

let inst_tpref tpinst ty tp  =
  if gen_mem_assoc typar_ref_eq tp tpinst then gen_assoc typar_ref_eq tp tpinst 
  else ty    (* avoid re-allocation of TType_app node in the common case *)

let remap_tcref tcmap tcr  =
    match tcref_map_tryfind tcr tcmap with 
    | Some tcr ->  tcr
    | None -> tcr

let remap_ucref tcmap (UCRef(tcref,nm)) = UCRef(remap_tcref tcmap tcref,nm)
let remap_rfref tcmap (RFRef(tcref,nm)) = RFRef(remap_tcref tcmap tcref,nm)

let mk_typar_inst (typars: local_typar_ref list) tyargs =  
  if length typars <> length tyargs then
    failwith ("mk_typar_inst: invalid type" ^ (sprintf " %d <> %d" (length typars) (length tyargs)));
  (List.combine typars tyargs : typar_inst)

let generalize_typar tp = mk_typar_ty (mk_local_tpref tp)
let generalize_typars tps = List.map generalize_typar tps

let rec remap_typeA (tyenv : tyenv) (ty:typ) =
  match strip_tpeqns ty with
  | TType_var tp as ty       -> inst_tpref tyenv.tpinst ty tp
  | TType_app (tcr,tinst) as ty -> 
      begin match tcref_map_tryfind tcr tyenv.tcref_remap with 
      | Some tcr ->  TType_app (tcr,remap_typesA tyenv tinst)
      | None -> 
          match tinst with 
          | [] -> ty  (* optimization to avoid re-allocation of TType_app node in the common case *)
          | _ -> TType_app (tcr,remap_typesA tyenv tinst)
      end
  | TType_tuple l         -> TType_tuple (remap_typesA tyenv l)
  | TType_fun (d,r)       -> TType_fun (remap_typeA tyenv d, remap_typeA tyenv r)
  | TType_forall (tps,ty) -> 
      let tps',tyenv = copy_remap_and_bind_typars tyenv tps in
      TType_forall (tps', remap_typeA tyenv ty)
  | TType_unknown
  | TType_modul_bindings          -> ty
and remap_typesA tyenv l = list_map (remap_typeA tyenv) l
and remap_typar_constraintsA tyenv cs =
   cs |>  chooseList (fun x -> 
         match x with 
         | TTyparCoercesToType(TTyparSubtypeConstraintFromFS ty,m) -> 
             Some(TTyparCoercesToType (TTyparSubtypeConstraintFromFS (remap_typeA tyenv ty),m))
         | TTyparCoercesToType(TTyparSubtypeConstraintFromIL _,m) -> 
             warning(Error("inst_typar_constraints tcenv: typar originated from IL code and should have gone throuh copy_or_import_typar_constraints",m));
             None
         | TTyparMayResolveMemberConstraint(traitInfo,m) -> 
             Some(TTyparMayResolveMemberConstraint (remap_traitA tyenv traitInfo,m))
         | TTyparDefaultsToType(priority,ty,m) -> Some(TTyparDefaultsToType(priority,remap_typeA tyenv ty,m))
         | TTyparIsEnum(uty,m) -> 
             Some(TTyparIsEnum(remap_typeA tyenv uty,m))
         | TTyparIsDelegate(uty1,uty2,m) -> 
             Some(TTyparIsDelegate(remap_typeA tyenv uty1,remap_typeA tyenv uty2,m))
         | TTyparSimpleChoice(tys,m) -> Some(TTyparSimpleChoice(map (remap_typeA tyenv) tys,m))
         | TTyparSupportsNull _ | TTyparIsNotNullableValueType _ 
         | TTyparIsReferenceType _ | TTyparRequiresDefaultConstructor _ -> Some(x))

and remap_traitA tyenv (TTrait(typs,nm,mf,tys,rty)) =
  TTrait(remap_typesA tyenv typs,nm,mf,List.map (remap_typeA tyenv) tys, remap_typeA tyenv rty)
and fixup_typar_constraints tp cs =
  (data_of_typar tp).typar_constraints <-  cs

and bind_typars tps tyargs tpinst =   
  map2 (fun tp tyarg -> (tp,tyarg)) tps tyargs @ tpinst 

(* copies bound typars, extends tpinst *)
and copy_remap_and_bind_typars tyenv tps =
  let tps' = copy_typars tps in
  let tyenv = { tyenv with tpinst = bind_typars tps (generalize_typars tps') tyenv.tpinst } in 
  List.iter2 (fun tporig tp -> fixup_typar_constraints tp (remap_typar_constraintsA tyenv  (constraints_of_typar tporig))) tps tps';
  tps',tyenv


let remap_type  tyenv x = if tyenv_is_empty tyenv then x else remap_typeA tyenv x
let remap_types tyenv x = if tyenv_is_empty tyenv then x else map (remap_typeA tyenv) x

let remap_param tyenv (TSlotParam(nm,typ,fl1,fl2,fl3,attribs)) = TSlotParam(nm,remap_typeA tyenv typ,fl1,fl2,fl3,attribs) 
let remap_slotsig tyenv (TSlotSig(nm,typ, ctps,methTypars,paraml, rty)) =
    let typ' = remap_typeA tyenv typ in 
    let ctps',tyenvinner = copy_remap_and_bind_typars tyenv ctps in
    let methTypars',tyenvinner = copy_remap_and_bind_typars tyenvinner methTypars in
    TSlotSig(nm,typ', ctps',methTypars',map (remap_param tyenvinner) paraml,remap_typeA tyenvinner rty) 

let mk_inst_tyenv tpinst = { tcref_remap= empty_tcref_remap; tpinst=tpinst }

(* entry points for "typar -> typ" instantiation *)
let inst_type              tpinst x = if isNil tpinst then x else remap_typeA  (mk_inst_tyenv tpinst) x
let inst_types             tpinst x = if isNil tpinst then x else remap_typesA (mk_inst_tyenv tpinst) x
let inst_trait             tpinst x = if isNil tpinst then x else remap_traitA (mk_inst_tyenv tpinst) x
let inst_typar_constraints tpinst x = if isNil tpinst then x else remap_typar_constraintsA (mk_inst_tyenv tpinst) x
let inst_slotsig tpinst ss = remap_slotsig (mk_inst_tyenv tpinst) ss
let copy_slotsig ss = remap_slotsig empty_tyenv ss


let mk_typar_to_typar_renaming tpsorig tps = 
  let tinst = generalize_typars tps in 
  mk_typar_inst tpsorig tinst,tinst


let mk_tycon_inst tycon tinst = mk_typar_inst (typars_of_tycon tycon) tinst
let mk_tcref_inst tcref tinst = mk_tycon_inst (deref_tycon tcref) tinst


(*---------------------------------------------------------------------------
!* Remove inference equations and abbreviations from types
 *------------------------------------------------------------------------- *)
    
let reduce_tcref_abbrev tc tyargs = 
  let abbrev = abbrev_of_tycon (deref_tycon tc) in 
  if not (isSome abbrev) then invalid_arg "reduce_tcref_abbrev";
  let abbrev_ty = the abbrev in 
  if isNil tyargs then abbrev_ty else inst_type (mk_tcref_inst tc tyargs) abbrev_ty

let rec strip_tpeqns_and_tcabbrevsA canShortcut ty = 
  match strip_tpeqnsA canShortcut ty with 
  | TType_app (tcref,tinst) when is_abbrev_tcref tcref  ->  
      strip_tpeqns_and_tcabbrevsA canShortcut (reduce_tcref_abbrev tcref tinst) 
  | ty -> ty

let strip_tpeqns_and_tcabbrevs ty = strip_tpeqns_and_tcabbrevsA false ty

let rec strip_eqns_from_ecref eref = 
  let exnc = deref_exnc eref in 
  match exn_repr_of_tycon exnc with
  | TExnAbbrevRepr eref -> strip_eqns_from_ecref eref
  | _ -> exnc

let dest_forall_typ  ty = ty |> strip_tpeqns_and_tcabbrevs |> (function TType_forall (tyvs,tau) -> (tyvs,tau) | _ -> failwith "dest_forall_typ: not a forall type")
let dest_fun_typ     ty = ty |> strip_tpeqns_and_tcabbrevs |> (function TType_fun (tyv,tau) -> (tyv,tau) | _ -> failwith "dest_fun_typ: not a function type")
let dest_tuple_typ   ty = ty |> strip_tpeqns_and_tcabbrevs |> (function TType_tuple l -> l | _ -> failwith "dest_tuple_typ: not a tuple type")
let dest_typar_typ   ty = ty |> strip_tpeqns_and_tcabbrevs |> (function TType_var v -> v | _ -> failwith "dest_typar_typ: not a typar type")
let is_fun_ty        ty = ty |> strip_tpeqns_and_tcabbrevs |> (function TType_fun _ -> true | _ -> false)
let is_forall_ty     ty = ty |> strip_tpeqns_and_tcabbrevs |> (function TType_forall _ -> true | _ -> false)
let is_tuple_ty      ty = ty |> strip_tpeqns_and_tcabbrevs |> (function TType_tuple _ -> true | _ -> false)
let is_union_ty      ty = ty |> strip_tpeqns_and_tcabbrevs |> (function TType_app(tcr,_) -> is_union_tcref tcr | _ -> false)
let is_abstract_ty   ty = ty |> strip_tpeqns_and_tcabbrevs |> (function TType_app(tcr,_) -> is_abstract_tcref tcr | _ -> false)
let is_fsobjmodel_ty ty = ty |> strip_tpeqns_and_tcabbrevs |> (function TType_app(tcr,_) -> is_fsobjmodel_tcref tcr | _ -> false)
let is_recd_ty       ty = ty |> strip_tpeqns_and_tcabbrevs |> (function TType_app(tcr,_) -> is_recd_tcref tcr | _ -> false)
let is_typar_ty      ty = ty |> strip_tpeqns_and_tcabbrevs |> (function TType_var _ -> true | _ -> false)


let try_dest_forall_typ ty = 
  if is_forall_ty ty then dest_forall_typ ty else ([],ty) 

let mk_tyapp_ty tcref tyargs = TType_app(tcref,tyargs)
let is_stripped_tyapp_typ   ty = ty |> strip_tpeqns_and_tcabbrevs |> (function TType_app _ -> true | _ -> false) 
let dest_stripped_tyapp_typ ty = ty |> strip_tpeqns_and_tcabbrevs |> (function TType_app(tcref,tinst) -> tcref,tinst | _ -> failwith "dest_stripped_tyapp_typ") 
let tcref_of_stripped_typ   ty = ty |> strip_tpeqns_and_tcabbrevs |> (function TType_app(tcref,_) -> tcref | _ -> failwith "tcref_of_stripped_typ") 
let tinst_of_stripped_typ   ty = ty |> strip_tpeqns_and_tcabbrevs |> (function TType_app(_,tinst) -> tinst | _ -> []) 
let tycon_of_stripped_typ   ty = deref_tycon (tcref_of_stripped_typ ty)

let mk_inst_for_stripped_typ typ = 
  if is_stripped_tyapp_typ typ then 
    let tcref,tinst = dest_stripped_tyapp_typ typ in 
    mk_tcref_inst tcref tinst
  else []

let domain_of_fun_typ ty = fst(dest_fun_typ ty)
let range_of_fun_typ ty = snd(dest_fun_typ ty)

(*---------------------------------------------------------------------------
!* Type information about records, constructors etc.
 *------------------------------------------------------------------------- *)
 
let formal_typ_of_rfield fspec = fspec.rfield_type
let static_of_rfield fspec = fspec.rfield_static
let static_of_rfref x = x |> deref_rfield |> snd |> static_of_rfield

let literal_value_of_rfield fspec = 
    match fspec.rfield_const  with 
    | None -> None
    | Some(TConst_default) -> None
    | Some(k) -> Some(k)

let zero_init_of_rfield fspec = 
    match fspec.rfield_const  with 
    | None -> false 
    | Some(TConst_default) -> true 
    | _ -> false
let literal_value_of_rfref x = x |> deref_rfield |> snd |> literal_value_of_rfield
let typ_of_rfield inst fspec  = inst_type inst (formal_typ_of_rfield fspec)

let typs_of_rfields inst rfields = List.map (typ_of_rfield inst) rfields

let typs_of_tcref_rfields inst tcref = typs_of_rfields inst (instance_rfields_of_tcref tcref) 

let uconstr_of_ucref x = (snd (deref_uconstr x))
let rfield_tables_of_ucref x = (uconstr_of_ucref x).uconstr_rfields 
let rfields_of_ucref x = rfield_tables_of_ucref x |> all_rfields_of_rfield_tables
let rfield_of_ucref_by_idx x n = rfield_by_idx (rfield_tables_of_ucref x) n

let rty_of_ucref x = (uconstr_of_ucref x).uconstr_rty
let typs_of_ucref_rfields inst x = typs_of_rfields inst (rfields_of_ucref x)
let typ_of_ucref_rfield_by_idx x tinst j = 
  let tcref = tcref_of_ucref x in 
  let inst = mk_tcref_inst tcref tinst in 
  typ_of_rfield inst (rfield_of_ucref_by_idx x j)
let rty_of_uctyp x tinst = 
  let tcref = tcref_of_ucref x in 
  let inst = mk_tcref_inst tcref tinst in 
  inst_type inst (rty_of_ucref x)

let rfields_of_ecref x = instance_rfields_of_tycon (strip_eqns_from_ecref x)
let rfield_of_ecref_by_idx x n = rfield_of_tycon_by_idx (strip_eqns_from_ecref x) n

let typs_of_ecref_rfields x = typs_of_rfields [] (rfields_of_ecref x)
let typ_of_ecref_rfield x j = typ_of_rfield [] (rfield_of_ecref_by_idx x j)

(* REVIEW: these could be faster, e.g. by storing the index in the namemap *)
let ucref_index (UCRef(tcref,id)) = try firstPos (fun ucspec -> name_of_uconstr ucspec = id) (uconstrs_array_of_tcref tcref) with Not_found -> error(Error(Printf.sprintf "constructor %s not found in type %s" id (name_of_tcref tcref), range_of_tcref tcref))
let rfref_index (RFRef(tcref,id)) = try firstPos (fun rfspec -> name_of_rfield rfspec = id) (rfields_array_of_tcref tcref) with Not_found -> error(Error(Printf.sprintf "field %s not found in type %s" id (name_of_tcref tcref), range_of_tcref tcref))

let ucrefs_of_tcref tcref = List.map (ucref_of_uconstr tcref) (uconstrs_of_tcref tcref)
let instance_rfrefs_of_tcref tcref = List.map (rfref_of_rfield tcref) (instance_rfields_of_tcref tcref)
let all_rfrefs_of_tcref tcref = List.map (rfref_of_rfield tcref) (all_rfields_of_tcref tcref)

let actual_typ_of_rfield tycon tinst fspec = 
  inst_type (mk_tycon_inst tycon tinst) fspec.rfield_type

let actual_rtyp_of_rfref fref tinst = 
  let tycon,fspec = deref_rfield fref in 
  actual_typ_of_rfield tycon tinst fspec


(*---------------------------------------------------------------------------
!* 
 *------------------------------------------------------------------------- *)
 
let formal_typ_of_tcref g tcref = TType_app(tcref,map mk_typar_ty (typars_of_tcref tcref))

let enclosing_formal_typ_of_val g v = formal_typ_of_tcref g (apparent_parent_of_vspr_val v)

    
(*---------------------------------------------------------------------------
!* Apply types to type functions 
 *------------------------------------------------------------------------- *)

let rec strip_fun_typ ty = 
  if is_fun_ty ty then 
    let (d,r) = dest_fun_typ ty in let more,rty = strip_fun_typ r in d::more, rty
  else [],ty

let reduce_forall_typ ty tyargs = 
  let tyvs,tau = dest_forall_typ ty in 
  inst_type (mk_typar_inst tyvs tyargs) tau

let reduce_iterated_fun_ty ty args = 
  fold_left (fun ty _ -> 
    if not (is_fun_ty ty) then failwith "reduce_iterated_fun_ty";
    snd (dest_fun_typ ty)) ty args

let apply_types functy (tyargs,argtys) = 
  let after_tyapp_ty = if is_forall_ty functy then reduce_forall_typ functy tyargs else functy in 
  reduce_iterated_fun_ty after_tyapp_ty  argtys

let formal_apply_types functy (tyargs,args) = 
  reduce_iterated_fun_ty 
    (if isNil tyargs then functy else snd (dest_forall_typ functy))
    args

let rec strip_fun_typ_upto n ty = 
  assert (n >= 0);
  if n > 0 && is_fun_ty ty then 
    let (d,r) = dest_fun_typ ty in 
    let more,rty = strip_fun_typ_upto (n-1) r in d::more, rty
  else [],ty

let dest_tuple_typ_upto n ty = 
  if n = 1 then [ty] else dest_tuple_typ ty

let try_dest_tuple_typ ty = if is_tuple_ty ty then dest_tuple_typ ty else [ty]

(* A 'tau' type is one with its type paramaeters stripped off *)
let dest_top_tau_type (argInfos: topArgInfo list list) tau =
    let argtys,rty = strip_fun_typ_upto (length argInfos) tau in
    if length argInfos <> length argtys then 
      failwith (sprintf "dest_top_tau_type internal error, #arities = %d, #argtys = %d" (length argInfos) (length argtys));
    let argtysl = map2 (fun argInfo argty -> combine (dest_tuple_typ_upto (length argInfo) argty) argInfo) argInfos argtys in
    argtysl,rty


let normalizeDeclaredtyparsForEquiRecursiveInference tps = 
    tps |> map (fun tp -> 
      let ty =  mk_typar_ty tp in
      if is_typar_ty ty then dest_typar_typ ty else tp)
    

let dest_top_forall_type (TopValInfo (ntps,argInfos,retInfo) as arity_info) ty =
    let tps,tau = (if ntps = 0 then [],ty else try_dest_forall_typ ty) in 
    if length tps <> ntps then failwith (sprintf "dest_top_forall_type: internal error, #tps = %d, #ntps = %d" (length tps) ntps);
    (* tps may be have been equated to other tps in equi-recursive type inference. Normalize them here *)
    let tps = normalizeDeclaredtyparsForEquiRecursiveInference tps in 
    tps,tau

let dest_top_type (TopValInfo (ntps,argInfos,retInfo) as arity_info) ty =
  let tps,tau = dest_top_forall_type arity_info ty in
  let argtysl,rty = dest_top_tau_type argInfos tau in
  tps,argtysl,rty,retInfo

(*-------------------------------------------------------------------------
 * Multi-dimensional array types...
 *------------------------------------------------------------------------- *)

let is_il_arr_tcref g tcr = 
  g.tcref_eq tcr g.il_arr1_tcr || 
  g.tcref_eq tcr g.il_arr2_tcr || 
  g.tcref_eq tcr g.il_arr3_tcr || 
  g.tcref_eq tcr g.il_arr4_tcr 

let rank_of_il_arr_tcref g tcr = 
  if g.tcref_eq tcr g.il_arr1_tcr then 1
  else if g.tcref_eq tcr g.il_arr2_tcr then 2
  else if g.tcref_eq tcr g.il_arr3_tcr then 3
  else if g.tcref_eq tcr g.il_arr4_tcr then 4
  else failwith "rank_of_il_arr_tcref: unsupported array rank"

(*-------------------------------------------------------------------------
 * Misc functions on F# types
 *------------------------------------------------------------------------- *)

let get_array_element_typ (g:tcGlobals) ty =
    let tcr,tinst = dest_stripped_tyapp_typ ty in 
    if List.length tinst <> 1 then failwith "get_array_element_typ";
    List.hd tinst

let is_il_arr_typ g ty =
  is_stripped_tyapp_typ ty && is_il_arr_tcref g (tcref_of_stripped_typ ty)

let is_il_arr1_typ g ty =
  is_stripped_tyapp_typ ty && g.tcref_eq (tcref_of_stripped_typ ty) g.il_arr1_tcr 

let dest_il_arr1_typ g ty = get_array_element_typ g ty

let is_compat_array_typ g ty = is_stripped_tyapp_typ ty && g.tcref_eq g.array_tcr (tcref_of_stripped_typ ty)
let is_unit_typ g ty = is_stripped_tyapp_typ ty && g.tcref_eq g.unit_tcr_canon (tcref_of_stripped_typ ty)

let is_il_named_typ ty = is_stripped_tyapp_typ ty && is_il_tcref (tcref_of_stripped_typ ty)


let is_il_class_typ ty = 
  (is_il_named_typ ty &&
   let tcr,tinst = dest_stripped_tyapp_typ ty in 
   let _,_,tdef = dest_il_tcref tcr in 
   (tdef.tdKind = TypeDef_class))

let is_il_interface_typ ty = 
  (is_il_named_typ ty && 
   let tcr,tinst = dest_stripped_tyapp_typ ty in 
   let _,_,tdef = dest_il_tcref tcr in 
   (tdef.tdKind = TypeDef_interface))

let is_il_ref_typ g ty = 
  (is_il_named_typ ty && 
   let tcr,tinst = dest_stripped_tyapp_typ ty in 
   let _,_,tdef = dest_il_tcref tcr in 
   not (is_value_tdef tdef)) ||
  is_il_arr_typ g ty

let is_il_struct_tycon tycon = 
  (is_il_tycon tycon && 
   let _,_,tdef = dest_il_tycon tycon in 
   is_value_tdef tdef)

let is_il_enum_tycon tycon = 
  (is_il_tycon tycon && 
   let _,_,tdef = dest_il_tycon tycon in 
   is_enum_tdef tdef)

let is_il_interface_tycon tycon = 
  (is_il_tycon tycon && 
   let _,_,tdef = dest_il_tycon tycon in 
   (tdef.tdKind = TypeDef_interface))

let is_il_struct_typ ty = 
  (is_il_named_typ ty && 
   let tcr,tinst = dest_stripped_tyapp_typ ty in 
   let _,_,tdef = dest_il_tcref tcr in 
   is_value_tdef tdef)

let is_il_delegate_tcref tcref = 
  if is_il_tcref tcref then 
    let _,_,tdef = dest_il_tcref  tcref in 
    match tdef.tdKind with
    | TypeDef_delegate -> true
    | _ -> false
  else false

let is_any_array_typ g ty =  is_il_arr_typ g ty || is_compat_array_typ g ty
let dest_any_array_typ g ty = get_array_element_typ g ty
let rank_of_any_array_typ g ty = 
 if is_il_arr_typ g ty then rank_of_il_arr_tcref g (tcref_of_stripped_typ ty)
 else 1

let is_fsobjmodel_ref_typ ty = 
  is_fsobjmodel_ty ty && 
  let tcr,tinst = dest_stripped_tyapp_typ ty in 
  match (tycon_objmodel_data_of_tcref tcr).tycon_objmodel_kind with 
  | TTyconClass | TTyconInterface   | TTyconDelegate _ -> true
  | TTyconStruct | TTyconEnum -> false

let is_tycon_kind_struct k = 
  match k with 
  | TTyconClass | TTyconInterface   | TTyconDelegate _ -> false
  | TTyconStruct | TTyconEnum -> true

let is_tycon_kind_enum k = 
  match k with 
  | TTyconStruct | TTyconClass | TTyconInterface   | TTyconDelegate _ -> false
  | TTyconEnum -> true

let is_fsobjmodel_struct_tycon x = 
  is_fsobjmodel_tycon x &&
  is_tycon_kind_struct (tycon_objmodel_data_of_tycon x).tycon_objmodel_kind 

let is_fsobjmodel_enum_tycon x = 
  is_fsobjmodel_tycon x &&
  is_tycon_kind_enum (tycon_objmodel_data_of_tycon x).tycon_objmodel_kind 

let is_fsobjmodel_class_tycon x = 
  is_fsobjmodel_tycon x &&
  match (tycon_objmodel_data_of_tycon x).tycon_objmodel_kind with TTyconClass -> true | _ -> false

let is_fsobjmodel_interface_tycon c = 
  is_fsobjmodel_tycon c &&
  match (tycon_objmodel_data_of_tycon c).tycon_objmodel_kind with TTyconInterface -> true | _ -> false

let is_fsobjmodel_delegate_tycon c = 
  is_fsobjmodel_tycon c &&
  match (tycon_objmodel_data_of_tycon c).tycon_objmodel_kind with TTyconDelegate _ -> true | _ -> false

let is_fsobjmodel_class_typ    ty = is_stripped_tyapp_typ ty && is_fsobjmodel_class_tycon (tycon_of_stripped_typ ty)
let is_fsobjmodel_struct_typ    ty = is_stripped_tyapp_typ ty && is_fsobjmodel_struct_tycon (tycon_of_stripped_typ ty)
let is_fsobjmodel_interface_typ ty = is_stripped_tyapp_typ ty && is_fsobjmodel_interface_tycon (tycon_of_stripped_typ ty)
let is_fsobjmodel_delegate_typ  ty = is_stripped_tyapp_typ ty && is_fsobjmodel_delegate_tycon (tycon_of_stripped_typ ty)

let is_delegate_typ ty = 
  is_fsobjmodel_delegate_typ ty ||
  (is_stripped_tyapp_typ ty && is_il_delegate_tcref (tcref_of_stripped_typ ty ))

let is_interface_typ ty = 
  is_il_interface_typ ty || 
  is_fsobjmodel_interface_typ ty

let is_class_typ ty = 
  is_il_class_typ ty || 
  is_fsobjmodel_class_typ ty

let is_ref_typ g ty = 
  is_union_ty ty || 
  is_compat_array_typ g ty ||
  is_tuple_ty ty || 
  is_recd_ty ty || 
  is_il_ref_typ g ty ||
  is_fun_ty ty || 
  is_abstract_ty ty || 
  is_fsobjmodel_ref_typ ty || 
  is_unit_typ g ty

let is_struct_typ ty = 
  is_il_struct_typ ty || is_fsobjmodel_struct_typ ty

let is_struct_tycon ty = 
  is_il_struct_tycon ty || is_fsobjmodel_struct_tycon ty

let is_enum_tycon ty = 
  is_il_enum_tycon ty || is_fsobjmodel_enum_tycon ty

let is_interface_tycon ty = 
  is_il_interface_tycon ty || is_fsobjmodel_interface_tycon ty

let is_enum_tcref tcref = is_enum_tycon (deref_tycon tcref)
let is_struct_tcref tcref = is_struct_tycon (deref_tycon tcref)
let is_interface_tcref tcref = is_interface_tycon (deref_tycon tcref)

let is_enum_typ ty = 
  is_stripped_tyapp_typ ty && is_enum_tcref (tcref_of_stripped_typ ty)

let actual_rty_of_slotsig parentTyInst methTyInst (TSlotSig(_,_,parentFormalTypars,methFormalTypars,_,formalRetTy)) = 
  let methTyInst = mk_typar_inst methFormalTypars methTyInst in
  let parentTyInst = mk_typar_inst parentFormalTypars parentTyInst in
  inst_type (parentTyInst @ methTyInst) formalRetTy

let slotsig_has_void_rty g (TSlotSig(_,_,_,_,_,formalRetTy)) = 
  is_unit_typ g formalRetTy 

let rty_of_tmethod (TMethod((TSlotSig(_,parentTy,_,_,_,_) as ss),methFormalTypars,_,_,m)) =
  let tinst = tinst_of_stripped_typ parentTy in 
  let methTyInst = generalize_typars methFormalTypars in
  actual_rty_of_slotsig tinst methTyInst ss
  
let enclosing_typ_of_slotsig (TSlotSig(_,parentTy,_,_,_,_)) = parentTy


(* Is the type 'abstract' *)
let is_partially_implemented_tycon tycon = 
  if is_fsobjmodel_tycon tycon then 
    not (is_fsobjmodel_delegate_tycon tycon) && 
    let tcaug = (tcaug_of_tycon tycon) in 
    tcaug.tcaug_abstract 
    (* Note this is a static analysis to determine if we know the type is abstract or not *)
    (* Now that we support an AbstractClass attribute and correctly report/error when classes are not marked Abstract then we *)
    (* no longer need to use this technique to determine abstract. *)
(*
    tcaug.tcaug_adhoc 
    |> Namemap.range_multi 
    |> exists (fun vref -> 
      let membInfo = (the (member_info_of_vref vref)) in 
      membInfo.vspr_flags.memFlagsAbstract && not membInfo.vspr_implemented)
*)
  else 
    (is_il_tycon tycon && 
     let _,_,tdef = dest_il_tycon tycon in 
     tdef.tdAbstract)

(*---------------------------------------------------------------------------
!* Find all type variables in a type, apart from those that have had 
 * an equation assigned by type inference.
 *------------------------------------------------------------------------- *)

let empty_free_locvals = Zset.empty val_spec_order
let union_free_locvals s1 s2 = 
  if s1 == empty_free_locvals then s2
  else if s2 == empty_free_locvals then s1
  else Zset.union s1 s2

let empty_free_rfields = Zset.empty rfref_order
let union_free_rfields s1 s2 = 
  if s1 == empty_free_rfields then s2
  else if s2 == empty_free_rfields then s1
  else Zset.union s1 s2

let empty_free_uconstrs = Zset.empty ucref_order
let union_free_uconstrs s1 s2 = 
  if s1 == empty_free_uconstrs then s2
  else if s2 == empty_free_uconstrs then s1
  else Zset.union s1 s2

let empty_free_loctycons = Zset.empty tycon_spec_order
let union_free_loctycons s1 s2 = 
  if s1 == empty_free_loctycons then s2
  else if s2 == empty_free_loctycons then s1
  else Zset.union s1 s2

let empty_free_loctypars = Zset.empty (fun lv1 lv2 -> compare (stamp_of_tpref lv1) (stamp_of_tpref lv2))
let union_free_loctypars s1 s2 = 
  if s1 == empty_free_loctypars then s2
  else if s2 == empty_free_loctypars then s1
  else Zset.union s1 s2

let empty_free_tyvars =  
  { free_loctycons=empty_free_loctycons; 
    free_loctypars=empty_free_loctypars}

let union_free_tyvars fvs1 fvs2 = 
  if fvs1 == empty_free_tyvars then fvs2 else 
  if fvs2 == empty_free_tyvars then fvs1 else
  { free_loctycons           = union_free_loctycons fvs1.free_loctycons fvs2.free_loctycons;
    free_loctypars           = union_free_loctypars fvs1.free_loctypars fvs2.free_loctypars }

let acc_free_loctycon x acc = if Zset.mem x acc.free_loctycons then acc else {acc with free_loctycons = Zset.add x acc.free_loctycons } 

let acc_free_tycon (tcr:tycon_ref) acc = 
  match tcr with 
  | Ref_private v -> acc_free_loctycon v acc
  | _ -> acc

let rec bound_typars tps acc = 
  (* Bound type vars form a recursively-referential set due to constraints, e.g.  A : I<B>, B : I<A> *)
  (* So collect up free vars in all constraints first, then bind all variables *)
  let acc = fold_right (constraints_of_typar >> acc_free_in_typar_constraints) tps acc in
  List.fold_right (fun tp acc -> {acc with free_loctypars = Zset.remove (mk_local_tpref tp) acc.free_loctypars}) tps acc

and acc_free_in_typar_constraints cxs acc =
  List.fold_right acc_free_in_typar_constraint cxs acc

and acc_free_in_typar_constraint tpc acc =
  match tpc with 
  | TTyparCoercesToType(TTyparSubtypeConstraintFromFS(typ),m) -> acc_free_in_type typ acc
  | TTyparCoercesToType(TTyparSubtypeConstraintFromIL _,m) -> 
      warning(Error("acc_free_in_typar_constraint: unexpected TTyparSubtypeConstraintFromIL",m));
      acc
  | TTyparMayResolveMemberConstraint (traitInfo,_) -> acc_free_in_trait traitInfo acc
  | TTyparDefaultsToType(_,rty,_) -> acc_free_in_type rty acc
  | TTyparSimpleChoice(tys,_) -> acc_free_in_types tys acc
  | TTyparIsEnum(uty,m) -> acc_free_in_type uty acc
  | TTyparIsDelegate(aty,bty,m) -> acc_free_in_type aty (acc_free_in_type bty acc)
  | TTyparSupportsNull _ | TTyparIsNotNullableValueType _ | TTyparIsReferenceType _ 
  | TTyparRequiresDefaultConstructor _ -> acc

and acc_free_in_trait (TTrait(typs,_,_,argtys,rty)) acc = 
  acc_free_in_types typs (acc_free_in_types argtys (acc_free_in_type rty acc))  

and acc_free_tpref tp acc = 
  if Zset.mem tp acc.free_loctypars then acc
  else 
    acc_free_in_typar_constraints (constraints_of_typar tp)
      {acc with free_loctypars=Zset.add tp acc.free_loctypars}

and acc_free_in_type ty acc  = 
  match strip_tpeqns ty with 
  | TType_tuple l -> acc_free_in_types l acc
  | TType_app (tc,tinst) -> acc_free_in_types tinst (acc_free_tycon tc  acc)
  | TType_fun (d,r) -> acc_free_in_type d (acc_free_in_type r acc)
  | TType_var r -> acc_free_tpref r acc
  | TType_forall (tps,r) -> union_free_tyvars (bound_typars tps (free_in_type r)) acc
  | TType_modul_bindings -> failwith "acc_free_in_type: naked struct"
  | TType_unknown -> failwith "acc_free_in_type: naked unknown"
and acc_free_in_types tys acc = 
  match tys with 
  | [] -> acc
  | h :: t -> acc_free_in_type h (acc_free_in_types t acc)
and free_in_type ty = acc_free_in_type ty empty_free_tyvars
let free_in_types tys = acc_free_in_types tys empty_free_tyvars
let acc_free_in_val v acc = acc_free_in_type (deref_osgn v).val_type acc
let free_in_val v = acc_free_in_val v empty_free_tyvars
let free_in_typar_constraints v = acc_free_in_typar_constraints v empty_free_tyvars
let acc_free_tprefs tps acc = list_fold_right acc_free_tpref tps acc
        

(*--------------------------------------------------------------------------
!* Free in type, left-to-right order preserved. This is used to determine the
 * order of type variables for top-level definitions based on their signature,
 * so be careful not to change the order.  We accumulate in reverse
 * order.
 *-------------------------------------------------------------------------*)

let empty_free_typars_lr = []
let union_free_typars_lr fvs1 fvs2 = gen_union_favour_right typar_ref_eq fvs1 fvs2

let rec bound_typars_lr cxFlag thruFlag acc tps = 
  (* Bound type vars form a recursively-referential set due to constraints, e.g.  A : I<B>, B : I<A> *)
  (* So collect up free vars in all constraints first, then bind all variables *)
  let acc = fold_left (fun acc tp -> acc_free_in_typar_constraints_lr cxFlag thruFlag acc (constraints_of_typar tp)) tps acc in
  List.fold_right (mk_local_tpref >> gen_remove typar_ref_eq) tps acc

and acc_free_in_typar_constraints_lr cxFlag thruFlag acc cxs =
  List.fold_left (acc_free_in_typar_constraint_lr cxFlag thruFlag) acc cxs 

and acc_free_in_typar_constraint_lr cxFlag thruFlag acc tpc =
  match tpc with 
  | TTyparCoercesToType(TTyparSubtypeConstraintFromFS(typ),m) -> acc_free_in_type_lr cxFlag thruFlag acc typ 
  | TTyparCoercesToType(TTyparSubtypeConstraintFromIL _,m) -> 
      warning(Error("acc_free_in_typar_constraint_lr: unexpected TTyparSubtypeConstraintFromIL",m));
      acc
  | TTyparMayResolveMemberConstraint (traitInfo,_) -> acc_free_in_trait_lr cxFlag thruFlag acc traitInfo 
  | TTyparDefaultsToType(_,rty,_) -> acc_free_in_type_lr cxFlag thruFlag acc rty 
  | TTyparSimpleChoice(tys,_) -> acc_free_in_types_lr cxFlag thruFlag acc tys 
  | TTyparIsEnum(uty,m) -> acc_free_in_type_lr cxFlag thruFlag acc uty
  | TTyparIsDelegate(aty,bty,m) -> acc_free_in_type_lr cxFlag thruFlag (acc_free_in_type_lr cxFlag thruFlag acc aty) bty  
  | TTyparSupportsNull _ | TTyparIsNotNullableValueType _ | TTyparIsReferenceType _ 
  | TTyparRequiresDefaultConstructor _ -> acc

and acc_free_in_trait_lr cxFlag thruFlag acc (TTrait(typs,_,_,argtys,rty))  = 
  let acc = acc_free_in_types_lr cxFlag thruFlag acc typs in 
  let acc = acc_free_in_types_lr cxFlag thruFlag acc argtys in
  let acc = acc_free_in_type_lr cxFlag thruFlag acc rty in 
  acc

and acc_free_tpref_lr cxFlag thruFlag acc tp = 
  if gen_mem typar_ref_eq tp acc 
  then acc
  else 
      let acc = (gen_insert typar_ref_eq tp acc) in 
      if cxFlag then 
          acc_free_in_typar_constraints_lr cxFlag thruFlag acc (constraints_of_typar tp)
      else 
          acc

and acc_free_in_type_lr cxFlag thruFlag acc ty  = 
  if verbose then  dprintf0 "--> acc_free_in_type_lr \n";
  match (if thruFlag then strip_tpeqns_and_tcabbrevs ty else strip_tpeqns ty) with 
  | TType_tuple l -> acc_free_in_types_lr cxFlag thruFlag acc l 
  | TType_app (tc,tinst) -> acc_free_in_types_lr cxFlag thruFlag acc tinst 
  | TType_fun (d,r) -> acc_free_in_type_lr cxFlag thruFlag (acc_free_in_type_lr cxFlag thruFlag acc d ) r
  | TType_var r -> acc_free_tpref_lr cxFlag thruFlag acc r 
  | TType_forall (tps,r) -> union_free_typars_lr (bound_typars_lr cxFlag thruFlag tps (acc_free_in_type_lr cxFlag thruFlag empty_free_typars_lr r)) acc
  | TType_modul_bindings -> failwith "acc_free_in_type_lr: naked struct"
  | TType_unknown -> failwith "acc_free_in_type_lr: naked unknown"
and acc_free_in_types_lr cxFlag thruFlag acc tys = 
  match tys with 
  | [] -> acc
  | h :: t -> acc_free_in_types_lr cxFlag thruFlag (acc_free_in_type_lr cxFlag thruFlag acc h) t
    
let free_in_type_lr thruFlag ty = acc_free_in_type_lr true thruFlag empty_free_typars_lr ty |> List.rev
let free_in_types_lr thruFlag ty = acc_free_in_types_lr true thruFlag empty_free_typars_lr ty |> List.rev
let free_in_types_lr_no_cxs ty = acc_free_in_types_lr false true empty_free_typars_lr ty |> List.rev

let var_of_bind (TBind(v,_)) = v
let rhs_of_bind (TBind(_,r)) = r

let typar_spec_order v1 v2 = Pervasives.compare (stamp_of_tpref v1) (stamp_of_tpref v2) (* type instanced *) 
let bind_order v1 v2 = val_spec_order (var_of_bind v1) (var_of_bind v2)


(*---------------------------------------------------------------------------
!* Equivalence types up to alpha-equivalence (NOTE: approximate during inference,
 * fully accurate post unification)
 *------------------------------------------------------------------------- *)

type type_equiv_env = 
    { ae_typars: typ typar_map;
      ae_tcrefs: tcref_remap}

let tyeq_env_empty = { ae_typars=tpmap_empty(); ae_tcrefs=empty_tcref_remap }

let bind_tyeq_env_types tps1 tys2 aenv =
  {aenv with ae_typars=fold_right2 (fun x y acc -> tpmap_add (mk_local_tpref x) y acc) tps1 tys2 aenv.ae_typars}

let bind_tyeq_env_typars tps1 tps2 aenv =
  bind_tyeq_env_types tps1 (map mk_typar_ty tps2) aenv

let bind_tyeq_env_tpinst tpinst aenv =
  let tps,tys = split tpinst in 
  bind_tyeq_env_types tps tys aenv

let mk_tyeq_env tps1 tps2 = bind_tyeq_env_typars tps1 tps2 tyeq_env_empty

let rec traits_aequiv g aenv (TTrait(typs1,nm,mf1,argtys,rty)) (TTrait(typs2,nm2,mf2,argtys2,rty2)) =
   gen_set_eq (type_aequiv g aenv) typs1 typs2 &&
   mf1 = mf2 && 
   type_aequiv g aenv rty rty2 && 
   List.length argtys = List.length argtys2 &&
   List.for_all2 (type_aequiv g aenv) argtys argtys2 &&
   nm = nm2
    
and typarConstraints_aequiv g aenv tpc1 tpc2 =
  match tpc1,tpc2 with
  | TTyparCoercesToType(TTyparSubtypeConstraintFromFS acty,_),
    TTyparCoercesToType(TTyparSubtypeConstraintFromFS fcty,_) -> 
      type_aequiv g aenv acty fcty

  | TTyparMayResolveMemberConstraint(trait1,_),
      TTyparMayResolveMemberConstraint(trait2,_) -> 
      traits_aequiv g aenv trait1 trait2 

  | TTyparDefaultsToType(_,acty,_),
    TTyparDefaultsToType(_,fcty,_) -> 
      type_aequiv g aenv acty fcty

  | TTyparIsEnum(uty1,_),TTyparIsEnum(uty2,_) -> 
      type_aequiv g aenv uty1 uty2

  | TTyparIsDelegate(aty1,bty1,_),TTyparIsDelegate(aty2,bty2,_) -> 
      type_aequiv g aenv aty1 aty2 && 
      type_aequiv g aenv bty1 bty2 

  | TTyparSimpleChoice (tys1,_),TTyparSimpleChoice(tys2,_) -> 
      gen_set_eq (type_aequiv g aenv) tys1 tys2

  | TTyparSupportsNull _               ,TTyparSupportsNull _ 
  | TTyparIsNotNullableValueType _    ,TTyparIsNotNullableValueType _
  | TTyparIsReferenceType _           ,TTyparIsReferenceType _ 
  | TTyparRequiresDefaultConstructor _, TTyparRequiresDefaultConstructor _ -> true
  | _ -> false

and typarConstraintSets_aequiv g aenv tp1 tp2 = 
    (static_req_of_typar tp1 = static_req_of_typar tp2) &&
    gen_set_eq (typarConstraints_aequiv g aenv) (constraints_of_typar tp1) (constraints_of_typar tp2)

and typar_decls_aequiv g aenv tps1 tps2 = 
    length tps1 = length tps2 &&
    let aenv = bind_tyeq_env_typars tps1 tps2 aenv in 
    List.for_all2 (typarConstraintSets_aequiv g aenv) tps1 tps2

and tcref_aequiv g aenv tc1 tc2 = 
             (g.tcref_eq tc1 tc2 || (tcref_map_mem tc1 aenv.ae_tcrefs && g.tcref_eq (tcref_map_find tc1 aenv.ae_tcrefs) tc2)) 

and type_aequiv g aenv ty1 ty2 = 
    if verbose then  dprintf0 "--> type_aequiv...\n";
    match strip_tpeqns_and_tcabbrevs ty1,strip_tpeqns_and_tcabbrevs ty2 with 
    | TType_forall(tps1,rty1), TType_forall(tps2,rty2) -> 
             typar_decls_aequiv g aenv tps1 tps2 && type_aequiv g (bind_tyeq_env_typars tps1 tps2 aenv) rty1 rty2
    | TType_var tp1, TType_var tp2 when typar_ref_eq tp1 tp2 -> true
    | TType_var tp1, _ when tpmap_mem tp1 aenv.ae_typars -> type_equiv g (tpmap_find tp1 aenv.ae_typars) ty2
    | TType_app (tc1,b1)  ,TType_app (tc2,b2) -> 
             tcref_aequiv g aenv tc1 tc2
          && types_aequiv g aenv b1 b2
    | TType_tuple l1,TType_tuple l2 -> types_aequiv g aenv l1 l2
    | TType_fun (dtys1,rty1),TType_fun (dtys2,rty2) -> 
             type_aequiv g aenv dtys1 dtys2 && type_aequiv g aenv rty1 rty2
    | _ -> false

and types_aequiv g aenv l1 l2 = (length l1 = length l2 && for_all2 (type_aequiv g aenv) l1 l2)
and type_equiv g ty1 ty2 =  type_aequiv g tyeq_env_empty ty1 ty2

(*--------------------------------------------------------------------------
!* Special representations (.NET member functions on F# types)
 *-------------------------------------------------------------------------*)

let member_val_is_instance  v    = (the (member_info_of_val  v   )).vspr_flags.memFlagsInstance
let member_vref_is_instance vref = member_val_is_instance (deref_val vref)

let num_obj_args_of_member membInfo = 
  if membInfo.vspr_flags.memFlagsInstance then 1 else 0

let num_obj_args_of_member_vref vref = 
  match member_info_of_vref vref with 
  | Some membInfo -> num_obj_args_of_member membInfo
  | None -> failwith "num_obj_args_of_member_vref"

(* Pull apart the type for an F# value that represents an object model method *)
let dest_vspr_typ membInfo arities ty = 
    let tps,argInfos,rty,retInfo = dest_top_type arities ty in 
    let nobj = num_obj_args_of_member membInfo in
    (* Slam all the arguments together. This limits the appliciability of this function *)
    (* REVIEW: this may not always be quite right *)
    let argInfos = List.concat argInfos in 
    let argInfos = 
        if nobj = 1 then 
          if isNil argInfos then (warning(Failure("warning: value does not have a valid member type")); argInfos) 
          else List.tl argInfos 
        else argInfos in 
    tps,argInfos,rty,retInfo

(* Check that an F# value represents an object model method. *)
(* It will also always have an arity (inferred from syntax). *)
let check_member_val membInfo arity m =
    match membInfo, arity with 
    | None,_ -> error(InternalError("dest_member_vref_typ - no membInfo" , m))
    | _,None -> error(InternalError("dest_member_vref_typ - no arity", m))
    | Some membInfo,Some arity ->  (membInfo,arity)

let check_vspr_vref vref =
    check_member_val (member_info_of_vref vref) (arity_of_vref vref) (range_of_vref vref)
     
(* Pull apart the type for an F# value that represents an object model method *)
(* Detect methods with no arguments by looking for form argument type of 'unit'. *)
(* REVIEW: this technique is OK in practice but should be reviewed as it may requrie further special *)
(* checks. *)
(* Review: Does dest_vspr_typ have any other direct callers? *)
let dest_member_typ g membInfo arity typ =
    let tps,argInfos,rty,retInfo = dest_vspr_typ membInfo arity typ in 
    let argInfos = 
        match argInfos with 
        | [(argType,_)] when type_equiv g g.unit_ty (fst (hd argInfos)) -> []
        | _ -> argInfos in
    (tps,argInfos,rty,retInfo)

let dest_member_vref_typ g vref =
    let membInfo,arity = check_vspr_vref vref in
    dest_member_typ g membInfo arity (type_of_vref vref)  

let retty_of_member_vref vref =
    let membInfo,arity = check_vspr_vref vref in
    let _,_,rty,_ = dest_vspr_typ membInfo arity (type_of_vref vref)  in 
    rty
  
(* Match up the type variables on an member value with the type *)
(* variables on the enclosing class *)
let partition_val_typars v  = 
     match arity_of_val v with 
     | None -> error(InternalError("partition_val_typars: not a top value", range_of_val v))
     | Some arities -> 
         let fullTypars,_ = dest_top_forall_type arities (type_of_val v)  in 
         let parent = apparent_parent_of_vspr_val v in 
         let parentTypars = typars_of_tcref parent in
         let nparentTypars = length parentTypars in 
         if nparentTypars <= length fullTypars then 
             let memberParentTypars,memberMethodTypars = chop_at nparentTypars fullTypars in
             let memberToParentInst,tinst = mk_typar_to_typar_renaming memberParentTypars parentTypars in 
             Some(parentTypars,memberParentTypars,memberMethodTypars,memberToParentInst,tinst)
         else None

let partition_vref_typars vref = partition_val_typars (deref_val vref) 

let name_of_slotsig (TSlotSig(nm,_,_,_,_,_)) = nm
let id_of_method (TMethod(slotsig,methFormalTypars,_,_,m)) = mksyn_id m (name_of_slotsig slotsig)
let logical_name_of_member membInfo = 
    match membInfo.vspr_implements_slotsig with 
    | Some slotsig -> name_of_slotsig slotsig
    | _ -> membInfo.vspr_il_name 

(* Get the arguments for an F# value that represents an object model method *)
let arginfos_of_member_val g v = 
    let m = (range_of_val v) in 
    let membInfo,arity = check_member_val (member_info_of_val v) (arity_of_val v) m in
    let typ = (type_of_val v)  in
    let _,arginfos,_,_ = dest_member_typ g membInfo arity typ in 
    if length arginfos = 1 && type_equiv g g.unit_ty (fst (hd arginfos)) then []
    else arginfos

let arginfos_of_member_vref g vref = 
    arginfos_of_member_val g (deref_val vref)

(** Get the property "type" (getter return type) for an F# value that represents a getter or setter *)
(** of an object model property. *)
let vtyp_of_property_val g v = 
    let m = (range_of_val v) in 
    let membInfo,arity = check_member_val (member_info_of_val v) (arity_of_val v) m in
    let typ = (type_of_val v)  in
    match membInfo.vspr_flags.memFlagsKind with 
    | MemberKindPropertySet ->
        let _,arginfos,_,_ = dest_member_typ g membInfo arity typ  in 
        if isNil arginfos then  begin
            warning(Error("warning: value did not have a valid property setter type", m));
            g.unit_ty
        end else begin
            arginfos |> front_n_back |> snd |> fst
        end
    | MemberKindPropertyGet ->
        let _,_,rty,_ = dest_member_typ g membInfo arity typ in 
        rty
    | _ -> error(InternalError("vtyp_of_prop_vref",m))


(** Get the property arguments for an F# value that represents a getter or setter *)
(** of an object model property. *)
let arginfos_of_propery_val g v = 
    let m = (range_of_val v) in 
    let membInfo,arity = check_member_val (member_info_of_val v) (arity_of_val v) m in
    let typ = (type_of_val v)  in
    match membInfo.vspr_flags.memFlagsKind with 
    | MemberKindPropertyGet ->
        arginfos_of_member_val g v
    | MemberKindPropertySet ->
        let _,arginfos,_,_ = dest_member_typ g membInfo arity typ in 
        if isNil arginfos then  begin
            warning(Error("warning: value did not have a valid property setter type", m));
            arginfos
        end else begin
            arginfos |> front_n_back |> fst 
        end
    | _ -> error(InternalError("arginfos_of_propery_val",m))


(*-------------------------------------------------------------------------
!* Property name mangling.
 * Expecting s to be in the form (as returned by qualified_name_of_tcref) of:
 *    get_P                         or  set_P
 *    Names/Space/Class/NLPath-get_P  or  Names/Space/Class/NLPath.set_P
 * Required to return "P"
 *------------------------------------------------------------------------*)

let chopTo s c =
  (* chopTo "abcdef" 'c' --> "def" *)
  if String.contains s c then
    let i = String.index s c + 1 in
    String.sub s i (String.length s - i)
  else
    s

let chop_property_name s =
  let s = chopTo s '.' in
  if String.length s <= 4 || (let s = String.sub s 0 4 in s <> "get_" && s <> "set_") then
    (errorR(Failure("Invalid internal property name: '"^s^"'")); s)
  else 
    String.sub s 4 (String.length s - 4)  (* chop "get_" or "set_" *)

let get_property_name membInfo =
  chop_property_name (logical_name_of_member membInfo)


let mustinline = function PseudoValue | AlwaysInline -> true | OptionalInline | NeverInline -> false

(*---------------------------------------------------------------------------
!* Generalize type constructors to types
 *--------------------------------------------------------------------------*)

let generalize_tcref_tinst tc =  generalize_typars (typars_of_tycon (deref_tycon tc))
let generalize_tcref tc = 
  let tinst = generalize_tcref_tinst tc in 
  tinst,TType_app(tc, tinst)

let isTTyparSupportsStaticMethod = function TTyparMayResolveMemberConstraint _ -> true | _ -> false
let isTTyparCoercesToType = function TTyparCoercesToType _ -> true | _ -> false

(*--------------------------------------------------------------------------
!* Print Signatures/Types - prelude
 *-------------------------------------------------------------------------- *)

let fully_split_tref tref = 
  List.concat (List.map Il.split_namespace (Il.enclosing_tnames_of_tref tref @ [Il.ungenericize_tname (Il.tname_of_tref tref)])) 

let demangle_dotnet_generic_overloading n = 
  let sym = '`' in
  if  String.contains n sym &&
      (* check what comes after the symbol is a number *)
      begin
        let m = String.rindex n sym in 
        let res = ref (m < String.length n - 1) in 
        for i = m + 1 to String.length n - 1 do
          res := !res && String.get n i >= '0' && String.get n i <= '9';
        done;
        !res
      end
  then 
    let pos = String.rindex n sym in 
    String.sub n 0 pos 
  else n


let prefix_of_static_req s =
    match s with 
    | NoStaticReq -> "'"
    | CompleteStaticReq -> " $"
    | HeadTypeStaticReq -> " ^"

let prefix_of_rigid typar =   
  if (rigid_of_typar typar <> TyparRigid) then "_" else ""

let display_name_of_tycon tc = demangle_dotnet_generic_overloading (name_of_tycon tc)
let display_name_of_typar tp = let nm = name_of_typar tp in if nm = "?" then "?"^string_of_int (stamp_of_typar tp) else nm
let display_name_of_tcref tcref = display_name_of_tycon (deref_tycon tcref)

(*---------------------------------------------------------------------------
!* Prettify: prettyTyparNames/prettifyTypes - make typar names human friendly
 *------------------------------------------------------------------------- *)

module PrettyTypes = struct

    let new_pretty_typar tp nm = new_typar (rigid_of_tpref tp,Typar(ident(nm, range_of_tpref tp),static_req_of_typar tp,false),false,[])

    let newPrettyTypars renaming tps names = 
        let nice_tps = map2 new_pretty_typar tps names in 
        let renaming = renaming @ mk_typar_inst tps (generalize_typars nice_tps) in
        List.iter2 (fun tp tpnice -> fixup_typar_constraints tpnice (inst_typar_constraints renaming (constraints_of_typar tp))) tps nice_tps;
        nice_tps, renaming

    let prettyTyparNames pred base alreadyInUse tps = 
        let rec choose i tp = 
            if pred tp then 
                let nm = 
                    if compgen_of_tpref tp then 
                        if i < 26 then String.make 1 (Char.chr(Char.code base + i)) 
                        else  String.make 1 base ^ string_of_int i
                    else name_of_tpref tp^string_of_int i in 
                if mem nm alreadyInUse then choose(i+1) tp else nm 
            else name_of_typar tp in 
        let start = List.length alreadyInUse in 
        list_mapi (fun n tp -> choose (n+start) tp) tps 

    let prettifyTypes foldTys mapTys tys = 
        let ftps = (foldTys (acc_free_in_type_lr true false) empty_free_typars_lr tys) in   
        (* let ftps = (foldTys (fun x acc -> acc_free_in_type_lr false acc x) tys empty_free_typars_lr) in   *)
        let ftps = List.rev ftps in 
        (* ftps |> List.iter (fun tp -> dprintf1 "free typar: %d\n" (stamp_of_typar tp)); *)
        let rec computeKeep keep change tps = 
            match tps with 
            | [] -> List.rev keep, List.rev change 
            | tp :: rest -> 
                if not (compgen_of_tpref tp) && (not (List.exists (fun tp2 -> name_of_tpref tp = name_of_tpref tp2) keep))  then
                    computeKeep (tp :: keep) change rest
                else 
                    computeKeep keep (tp :: change) rest in
        let keep,change = computeKeep [] [] ftps in 
        
        (* change |> List.iter (fun tp -> dprintf3 "change typar: %s %s %d\n" (name_of_typar tp) (display_name_of_typar tp) (stamp_of_typar tp));  *)
        (* keep |> List.iter (fun tp -> dprintf3 "keep typar: %s %s %d\n" (name_of_typar tp) (display_name_of_typar tp) (stamp_of_typar tp));  *)
        let alreadyInUse = map name_of_tpref keep in 
        let names = prettyTyparNames (fun x -> List.memq x change) 'a' alreadyInUse ftps in 

        let nice_tps, renaming = newPrettyTypars [] ftps names  in
        let pretty_tys = mapTys (inst_type renaming) tys in
        (* nice_tps |> List.iter (fun tp -> dprintf1 "nice typar: %d\n" (stamp_of_typar tp)); *) 
        let tpconstraints  = nice_tps |> mapConcat (fun tpnice -> map (fun tpc -> tpnice,tpc) (constraints_of_tpref tpnice)) in

        renaming,
        pretty_tys,
        tpconstraints

    let prettify1        x = prettifyTypes (fun f -> f) (fun f -> f) x
    let prettify2        x = prettifyTypes (fun f -> foldl'2 (f,f)) (fun f -> map'2 (f,f)) x
    let prettifyN   x = prettifyTypes fold_left map   x
    let prettifyTypesN1  x = prettifyTypes (fun f -> foldl'2 (fold_left (foldl1'2  f), f)) (fun f -> map'2 (map (map1'2  f),f)) x
    let prettifyTypesNN1 x = prettifyTypes (fun f -> foldl'3 (fold_left f,fold_left (foldl1'2 f),f)) (fun f -> map'3 (map f, map (map1'2  f), f)) x

end
    (*--------------------------------------------------------------------------
    !* typeSimplificationInfo notes singleton/inplace typars and constraints
     *------------------------------------------------------------------------*)

module SimplifyTypes = struct

    (* CAREFUL! This function does NOT walk constraints *)
    let rec foldType f z typ =
        let typ = strip_tpeqns typ  in 
        let z = f z typ in
        match typ with
        | TType_forall (tps,body) -> foldType f z body
        | TType_app (tycon,tinst) -> List.fold_left (foldType f) z tinst
        | TType_tuple typs        -> List.fold_left (foldType f) z typs
        | TType_fun (s,t)         -> foldType f (foldType f z s) t
        | TType_var tp            -> z
        | TType_unknown           -> z
        | TType_modul_bindings            -> z

    let incM x m =
        if Zmap.mem x m then Zmap.add x (1 + Zmap.find x m) m
        else Zmap.add x 1 m

    let accTyparCounts z typ =
        (* Walk type to determine typars and their counts (for pprinting decisions) *)
        foldType (fun z typ -> match typ with | TType_var tp when rigid_of_tpref tp = TyparRigid  -> incM tp z | _ -> z) z typ

    let emptyTyparCounts = Zmap.empty typar_spec_order

    (* print multiple fragments of the same type using consistent naming and formatting *)
    let accTyparCountsMulti acc l = List.fold_left accTyparCounts acc l

    type typeSimplificationInfo =
        { singletons         : typar_spec Zset.set;
          inplaceConstraints : (typar_spec,typar_constraint_typ) Zmap.map;
          postfixConstraints : (typar_spec * typar_constraint) list; }
          
    let typeSimplificationInfo0 = 
        { singletons         = Zset.empty typar_spec_order;
          inplaceConstraints = Zmap.empty typar_spec_order;
          postfixConstraints = [] }

    let categorizeConstraints simplify m cxs =
        let singletons = if simplify then Zmap.chooseL (fun tp n -> if n=1 then Some tp else None) m else [] in
        let singletons = Zset.addL singletons (Zset.empty typar_spec_order) in
        (* Here, singletons are typars that occur once in the type.
         * However, they may also occur in a type constraint.
         * If they do, they are really multiple occurance - so we should remove them.
         *)
        let constraintTypars = (free_in_typar_constraints (map snd cxs)).free_loctypars in
        let usedInTypeConstraint typar = Zset.mem typar constraintTypars in
        let singletons = singletons |> Zset.filter (usedInTypeConstraint >> not)  in
        (* Here, singletons should really be used once *)
        let inplace,postfix =
          cxs |> partition (fun (tp,tpc) -> 
            simplify &&
            isTTyparCoercesToType tpc && 
            Zset.mem tp singletons && 
            length (constraints_of_typar tp) = 1) in
        let inplace = inplace |> List.map (function (tp,TTyparCoercesToType(ty,m)) -> tp,ty | _ -> failwith "not isTTyparCoercesToType") in
        
        { singletons         = singletons;
          inplaceConstraints = Zmap.of_list (Zmap.empty typar_spec_order) inplace;
          postfixConstraints = postfix;
        }
    let collectInfo simplify tys cxs = 
        categorizeConstraints simplify (accTyparCountsMulti emptyTyparCounts tys) cxs 
        
end

let rec iterType ((fStripped,fTypars) as f) typ =
    let typ = strip_tpeqns typ  in 
    fStripped typ;
    match typ with
    | TType_forall (tps,body) -> 
        iterType f body;           
        tps |> List.iter fTypars;
        tps |> List.iter (constraints_of_tpref >> List.iter (iterTypeContraint f))

    | TType_app (tycon,tinst) -> iterTypes f tinst
    | TType_tuple typs        -> iterTypes f typs
    | TType_fun (s,t)         -> iterType f s; iterType f t
    | TType_var tp            -> fTypars tp; 
    | TType_unknown           -> ()
    | TType_modul_bindings            -> ()
and iterTypes f tys = List.iter (iterType f) tys
and iterTypeContraint ((fStripped,fTypars) as f) x =
     match x with 
     | TTyparCoercesToType(TTyparSubtypeConstraintFromFS ty,m) -> iterType f ty
     | TTyparCoercesToType(TTyparSubtypeConstraintFromIL _,m) -> 
         warning(InternalError("iterTypeContraint: typar originated from IL code and should have gone throuh copy_or_import_typar_constraints",m));
         ()
     | TTyparMayResolveMemberConstraint(traitInfo,m) -> iterTraitInfo f traitInfo
     | TTyparDefaultsToType(priority,ty,m) -> iterType f ty
     | TTyparSimpleChoice(tys,m) -> iterTypes f tys
     | TTyparIsEnum(uty,m) -> iterType f uty
     | TTyparIsDelegate(aty,bty,m) -> iterType f aty; iterType f bty
     | TTyparSupportsNull _ | TTyparIsNotNullableValueType _ 
     | TTyparIsReferenceType _ | TTyparRequiresDefaultConstructor _ -> ()
and iterTraitInfo f (TTrait(typs,_,_,argtys,rty))  = iterTypes f typs; iterTypes f argtys; iterType f rty 

(*--------------------------------------------------------------------------
!* Print Signatures/Types
 *-------------------------------------------------------------------------- *)

type displayEnv = 
  { html: bool;
    htmlHideRedundantKeywords: bool;
    htmlAssemMap: string namemap; (* where can the docs for f# assemblies be found? *)
    openTopPaths: (string list) list; 
    showTyparBinding: bool; 
    showImperativeTyparAnnotations: bool;
    showMemberContainers:bool;
    showAttributes:bool;
    showConstraintTyparAnnotations: bool;
    showTyparDefaultConstraints : bool;
    g: tcGlobals; }
let empty_denv tcGlobals = 
  { html=false;
    htmlHideRedundantKeywords=false;
    htmlAssemMap=Namemap.empty;
    openTopPaths=[]; 
    showTyparBinding = false;
    showImperativeTyparAnnotations=false;
    showMemberContainers=false;
    showAttributes=false;
    showConstraintTyparAnnotations=true;
    showTyparDefaultConstraints=false;
    g=tcGlobals; }

let denv_add_open_path path denv = 
  { denv with openTopPaths= path :: denv.openTopPaths}

let denv_add_open_modref modref denv = 
    match pubpath_of_modref modref with 
    | None -> denv
    | Some(PubPath(p,n)) -> 
      let path = p@[n] in 
      denv_add_open_path path denv 

let demangle_operator nm = 
    let nm = decompileOpName nm in 
    if isOpName nm then "( "^nm^" )" else nm 

let core_display_name_of_val v = 
    match member_info_of_val v with 
    | Some membInfo -> 
        begin match membInfo.vspr_flags.memFlagsKind with 
        | MemberKindClassConstructor 
        | MemberKindConstructor 
        | MemberKindMember -> membInfo.vspr_il_name
        | MemberKindPropertyGetSet 
        | MemberKindPropertySet
        | MemberKindPropertyGet -> get_property_name membInfo 
        end
    | None -> name_of_val v 

let display_name_of_val v = 
    demangle_operator (core_display_name_of_val v)

let display_name_of_vref v = 
    display_name_of_val (deref_val v)

let compiled_name_of_val v = 
    match member_info_of_val v with 
    | Some membInfo -> membInfo.vspr_il_name
    | None -> name_of_val v

let full_name_of_nlpath (NLPath(ccu,p) : nonlocal_path) =  text_of_path p 
let (+.+) s1 s2 = s1^"."^s2
let full_name_of_item_ref nmF stripF ppF x = 
    match x with 
    | Ref_private x -> 
        begin match ppF x with 
        | None -> nmF x 
        | Some (PubPath(p,nm)) -> text_of_path (p@[nm])
        end
    | Ref_nonlocal nlr -> 
        let nm = stripF nlr.nlr_item in
        match nlr.nlr_nlpath with 
        | (NLPath(ccu,[])) -> nm 
        | _ -> full_name_of_nlpath nlr.nlr_nlpath +.+ nm
  
let full_name_of_modref r = full_name_of_item_ref name_of_modul try_decode_modref_name pubpath_of_modul r
let full_name_of_vref   r = full_name_of_item_ref name_of_val   (fun x -> x)           pubpath_of_val   r
let full_name_of_tcref  r = full_name_of_item_ref name_of_tycon try_decode_modref_name pubpath_of_tycon r
let full_name_of_ecref  r = full_name_of_item_ref name_of_tycon try_decode_modref_name pubpath_of_exnc  r

let full_name_of_ucref ucref = full_name_of_tcref (tcref_of_ucref ucref) +.+ (name_of_ucref ucref)
let full_name_of_rfref rfref = full_name_of_tcref (tcref_of_rfref rfref) +.+ (name_of_rfref rfref)

let full_path_to_tcref tcref = 
    match tcref with 
    | Ref_private ltc -> (match pubpath_of_tcref tcref with None -> [] | Some (PubPath(p,nm)) -> p)
    | Ref_nonlocal nlr -> path_of_nlpath (nlpath_of_nlref nlr)
  
let qualified_name_of_tcref tcref nm = 
    String.concat "-" (full_path_to_tcref tcref @ [ name_of_tcref tcref ^ "-" ^ nm ])

let rec firstEq p1 p2 = 
    match p1 with
    | [] -> true 
    | h1::t1 -> 
        match p2 with 
        | h2::t2 -> h1 = h2 && firstEq t1 t2
        | _ -> false 

let rec firstRem p1 p2 = 
   match p1 with [] -> p2 | h1::t1 -> firstRem t1 (List.tl p2)

let trim_path_by_denv denv path =
    let findOpenedNamespace opened_path = 
        if  firstEq opened_path path then 
          let t2 = firstRem opened_path path in 
          if t2 <> [] then Some(text_of_path t2^".")
          else Some("")
        else None in 
    (* sort the opened namespaces to find the best possible match - YUCK *)
    let sorted = List.sort (fun p1 p2 -> -(compare p1 p2)) denv.openTopPaths in 
    match Lib.choose findOpenedNamespace sorted with
    | Some(s) -> s
    | None ->  if isNil path then "" else text_of_path path ^ "."


let adhoc_of_tycon tycon = 
    Namemap.range_multi (tcaug_of_tycon tycon).tcaug_adhoc |> 
    filter (compgen_of_vref >> not)

let super_of_tycon g tycon = match (tcaug_of_tycon tycon).tcaug_super with None -> g.obj_ty | Some ty -> ty 
let implements_of_tycon (g:tcGlobals) tycon = (tcaug_of_tycon tycon).tcaug_implements |> map (fun (x,_,_) -> x)

(*----------------------------------------------------------------------------
 * Detect attributes
 *--------------------------------------------------------------------------*)

(* AbsIL view of attributes (we read these from .NET binaries) *)
let is_il_attrib tref attr = 
  tname_of_tspec (tspec_of_typ (enclosing_typ_of_mspec attr.customMethod)) = tname_of_tref tref
let ilthing_has_il_attrib tref attrs = List.exists (is_il_attrib tref) (dest_custom_attrs attrs)
let ilthing_find_il_attrib tref attrs = List.find (is_il_attrib tref) (dest_custom_attrs attrs)

let ilthing_decode_il_attrib g tref attrs = 
  if ilthing_has_il_attrib tref attrs then Some(decode_cattr_data g.ilg (ilthing_find_il_attrib tref attrs)) else None


(* F# view of attributes (these get converted to AbsIL attributes in ilxgen) *)
let is_fs_attrib g (AttribInfo(tref,tcref)) (Attrib(k,args,props)) = 
    match k with 
    | (ILAttrib(mref)) -> 
        (tname_of_tref (tref_of_mref mref) = tname_of_tref tref)
    | (FSAttrib(vref)) ->
        let _,_,rty,_ = dest_member_vref_typ g vref in 
        g.tcref_eq (tcref_of_stripped_typ rty) tcref 
let fsthing_has_attrib g tref attrs = List.exists (is_fs_attrib g tref) attrs
let fsthing_find_attrib g tref attrs = List.find (is_fs_attrib g tref) attrs
let fsthing_tryfind_attrib g tref attrs = tryfind (is_fs_attrib g tref) attrs

let fsthing_bool_attrib g nm attrs = 
    match fsthing_tryfind_attrib g nm attrs with
    | Some(Attrib(_,[ ],_)) -> Some(true)
    | Some(Attrib(_,[ TExpr_const (TConst_bool(b),_,_) ],_)) -> Some(b)
    | _ -> None

let fsthing_unit_attrib g nm attrs = 
    match fsthing_tryfind_attrib g nm attrs with
    | Some(Attrib(_,[ ],_)) -> Some()
    | _ -> None

let fsthing_int32_attrib g nm attrs = 
    match fsthing_tryfind_attrib g nm attrs with
    | Some(Attrib(_,[ TExpr_const (TConst_int32(b),_,_) ],_)) -> Some b
    | _ -> None
    
let tcref_bind_attrib g (AttribInfo(atref,_) as args) tcref f1 f2 = 
    if is_il_tcref tcref then 
      let _,_,tdef = dest_il_tcref  tcref in 
      begin match ilthing_decode_il_attrib g atref tdef.tdCustomAttrs with 
      | Some attr -> f1 attr
      | _ -> None
      end
    else 
      begin match fsthing_tryfind_attrib g args (attribs_of_tcref tcref) with 
      | Some attr -> f2 attr
      | _ -> None
      end
      
let tcref_has_attrib g args tcref = tcref_bind_attrib g args tcref (fun _ -> Some()) (fun _ -> Some()) |> isSome 


(*-------------------------------------------------------------------------
 * List and reference types...
 *------------------------------------------------------------------------- *)

let is_byref_ty g ty     = is_stripped_tyapp_typ ty && g.tcref_eq g.byref_tcr (tcref_of_stripped_typ ty)
let dest_byref_ty g ty   = if is_byref_ty g ty then List.hd (tinst_of_stripped_typ ty) else failwith "dest_byref_ty: not a byref type"

let is_refcell_ty g ty   = is_stripped_tyapp_typ ty && g.tcref_eq g.refcell_tcr (tcref_of_stripped_typ ty)
let dest_refcell_ty g ty = if is_refcell_ty g ty then List.hd (tinst_of_stripped_typ ty) else failwith "dest_refcell_ty: not a ref type"
let mk_refcell_ty  g ty = TType_app(g.refcell_tcr_nice,[ty])

let mk_lazy_ty g ty = TType_app(g.lazy_tcr_nice,[ty])
let mk_lazystatus_ty g ty = TType_app(g.lazystatus_tcr,[ty])
let mk_format_ty g aty bty cty dty ety = TType_app(g.format_tcr, [aty;bty;cty;dty; ety])

let mk_option_ty g ty = TType_app (g.option_tcr_nice, [ty])
let mk_list_ty g ty = TType_app (g.list_tcr_nice, [ty])
let is_arity1_ty g tcr ty = is_stripped_tyapp_typ ty && g.tcref_eq tcr (tcref_of_stripped_typ ty)
let is_option_ty g ty = is_arity1_ty g g.option_tcr ty

let try_dest_option_ty g ty = 
  match tinst_of_stripped_typ ty with 
  | [ty1]  when is_option_ty g ty  -> Some(ty1)
  | _ -> None

let dest_option_ty g ty = 
  match try_dest_option_ty g ty with 
  | Some(ty) -> ty
  | None -> failwith "dest_option_ty: not an option type"


let mk_none_ucref g = mk_ucref g.option_tcr "None"
let mk_some_ucref g = mk_ucref g.option_tcr "Some"


let vref_is_abstract vref = 
    match (member_info_of_vref vref) with 
    | Some membInfo -> membInfo.vspr_flags.memFlagsAbstract 
    | None -> false
          

(*--------------------------------------------------------------------------
!* Print Signatures/Types 
 *-------------------------------------------------------------------------- *)

module NicePrint = struct


    open Layout
    open PrettyTypes


    (* Note: We need nice printing of constants in order to print literals and attributes *)
    let tconstL c =
        let str = 
            match c with
            | TConst_bool x        -> if x then "true" else "false"
            | TConst_int8 x        -> (x |> Nums.i8_to_i32       |> Int32.to_string)^"y"
            | TConst_uint8 x       -> (x |> Nums.u8_to_i32       |> Int32.to_string)^"uy"
            | TConst_int16 x       -> (x |> Nums.i16_to_i32      |> Int32.to_string)^"s"
            | TConst_uint16 x      -> (x |> Nums.u16_to_i32      |> Int32.to_string)^"us"
            | TConst_int32 x       -> (x                         |> Int32.to_string)
            | TConst_uint32 x      -> (x |> Nums.u32_to_i64      |> Int64.to_string)^"u"
            | TConst_int64 x       -> (x                         |> Int64.to_string)^"L"
            | TConst_uint64 x      -> (x |> Nums.u64_to_i64      |> Int64.to_string)^"UL"
            | TConst_nativeint x   -> (x                         |> Int64.to_string)^"n"
            | TConst_unativeint x  -> (x |> Nums.u64_to_i64      |> Int64.to_string)^"un"
            | TConst_float32 x     -> (x |> Nums.ieee32_to_float |> string_of_float)^"f"
            | TConst_float x       -> (x |> Nums.ieee64_to_float |> string_of_float)
            | TConst_char x        -> 
                let i32 = x |> Nums.unichar_to_u16 |> Nums.u16_to_i32 in
                "'" ^ Bytes.unicode_bytes_as_string (Bytes.of_intarray [| b0 i32; b1 i32 |])  ^ "'" 
            | TConst_string bs     -> "\"" ^ Bytes.unicode_bytes_as_string bs ^ "\"" 
            | TConst_bigint bs     -> Bytes.unicode_bytes_as_string bs ^ "I" 
            | TConst_decimal bs    -> Bytes.unicode_bytes_as_string bs ^ "M" 
            | TConst_bignum bs     -> Bytes.unicode_bytes_as_string bs ^ "N" 
            | TConst_unit          -> "()" 
            | TConst_default       -> "default" in
        wordL str

    let bracketIfL x lyt = if x then bracketL lyt else lyt
    let hlinkL (url:string) l = linkL url l
    let squareAngleL x = leftL "[<" $$ x $$ rightL ">]"
    let angleL x = sepL "<" $$ x $$ rightL ">"  
    let braceL x = leftL "{" $$ x $$ rightL "}"  
    let boolL = function true -> wordL "true" | false -> wordL "false"

    (* attributes classes *)
    let valA  ly = tagL "VAL"  ly


    let trefL denv tref =
        let path = fully_split_tref tref in 
        let p2,n = frontAndBack path in
        if denv.html then
            hlinkL (text_of_path path ^ ".html") (wordL n)
        else 
            leftL (trim_path_by_denv denv p2) $$ wordL n
        
    (** Layout a reference to a type or value, perhaps emitting a HTML hyperlink *)
    let item_refL isExn denv nmF ppF v = 
        match v with 
        | Ref_private _ -> 
            begin match ppF v with 
            | None -> wordL (nmF v)
            | Some (PubPath(path,nm)) -> 
                let demangled = demangle_dotnet_generic_overloading nm in 
                let demangled = if isExn then demangle_exception_name demangled else demangled in 
                if denv.html then 
                    let nm = (text_of_path (path@["type_" ^ underscore_lowercase demangled])) in    (* text must tie up with fsc.ml *)
                    hlinkL (sprintf "%s.html" nm)  (wordL demangled)
                else
                    let pathText = trim_path_by_denv denv path in 
                    let tyconTextL = wordL demangled in 
                    (if pathText = "" then tyconTextL else leftL pathText $$ tyconTextL)
            end

        | Ref_nonlocal nlref -> 
            let (NLPath(ccu,path)) as nlpath = nlpath_of_nlref nlref in 
            let ccun = (name_of_ccu_thunk ccu) in 
            let item = (item_of_nlref nlref) in 
            let demangled = demangle_dotnet_generic_overloading item in 
            let demangled = if isExn then demangle_exception_name demangled else demangled in 
            let demangled = try_decode_modref_name demangled in 
            if denv.html then 
                let href = 
                    match ccun with 
                    | "mscorlib" | "System" | "System.Windows.Forms" 
                    | "System.Xml" | "System.Drawing" | "System.Data" -> 
                        (* cross link to the MSDN 2.0 documentation.  Generic types don't seem to have stable names :-( *)
                        if demangled = item then 
                          Some (sprintf "http://msdn2.microsoft.com/en-us/library/%s.aspx" (text_of_path (path@[demangle_dotnet_generic_overloading item])))
                        else None
                    | _ -> 
                        if ccu_is_fsharp ccu then 
                            let nm = (text_of_path (path@["type_" ^ underscore_lowercase demangled])) in     (* text must tie up with fsc.ml *)
                            match Namemap.tryfind ccun denv.htmlAssemMap with 
                            | Some root -> Some (sprintf "%s/%s.html" root nm)  
                            (* otherwise assume it is installed parallel to this tree *)
                            | None -> Some (sprintf "../%s/%s.html" ccun nm)  
                        else
                            None in 
                           
                match href with 
                | Some href -> hlinkL href (wordL demangled)
                | None      -> wordL demangled
            else
              let pathText = trim_path_by_denv denv path in 
              let tyconTextL = wordL demangled in 
              (if pathText = "" then tyconTextL else leftL pathText $$ tyconTextL)

    (** Layout a reference to a type *)
    let tcrefL denv tcr = 
        item_refL false denv name_of_tcref pubpath_of_tcref tcr

    (** Layout the flags of a member *)
    let memFlagsL hide memFlags = 
        let stat = if hide || memFlags.memFlagsInstance || (memFlags.memFlagsKind = MemberKindConstructor) then emptyL else wordL "static" in
        let stat = if not memFlags.memFlagsAbstract && memFlags.memFlagsVirtual then stat ++ wordL "virtual" 
                   else if not hide && memFlags.memFlagsAbstract then stat ++ wordL "abstract" 
                   else if memFlags.memFlagsOverride then stat ++ wordL "override" 
                   else stat in
        let stat = 
        
            if memFlags.memFlagsOverride then stat 
            else  
               match memFlags.memFlagsKind with 
              | MemberKindClassConstructor  
              | MemberKindConstructor 
              | MemberKindPropertyGetSet -> stat
              | MemberKindMember 
              | MemberKindPropertyGet 
              | MemberKindPropertySet -> stat ++ wordL "member" in

        (* let stat = if memFlags.memFlagsFinal then stat ++ wordL "final" else stat in  *)
        stat


    (* NOTE: The primed' functions take an "env" - determines inplace printing of typar and constraints *)
    (* NOTE: "denv" is the displayEnv - "env" is the typeSimplificationInfo *)      

    (** Layout type parameters, taking typeSimplificationInfo into account  *)
    let rec typarDeclsL' denv env nmL prefix typars =
        let tpcs = mapConcat (fun tp -> map (fun tpc -> tp,tpc) (constraints_of_typar tp)) typars in
        match typars,tpcs with 
        | [],[]  -> 
            nmL

        | [h],[] when not prefix -> 
            typarL' denv env h --- nmL

        | _ -> 
            let tpcsL = constraintsL denv env tpcs in 
            let coreL = sepListL (sepL ",") (map (typarL' denv env) typars) in 
            (if prefix or nonNil(tpcs) then nmL $$ angleL (coreL --- tpcsL) else bracketL coreL --- nmL)

    (** Layout a single type parameter declaration, taking typeSimplificationInfo into account  *)
    (* There are several printing-cases for a typar:
     * 
     *   'a              - is multiple  occurance.
     *   _               - singleton occurance, an underscore prefered over 'b. (OCAML accepts but does not print)
     *   #Type           - inplace coercion constraint and singleton.
     *   ('a :> Type)    - inplace coercion constraint not singleton.
     *   ('a.opM : S->T) - inplace operator constraint.
     * 
     *)  
    and typarL' denv env typar =
        let varL =
          wordL (sprintf "%s%s%s"
                   (if denv.showConstraintTyparAnnotations then prefix_of_static_req (static_req_of_typar typar) else "'")
                   (if denv.showImperativeTyparAnnotations then prefix_of_rigid typar else "")
                   (display_name_of_typar typar))  in

        match Zmap.tryfind typar env.SimplifyTypes.inplaceConstraints with
        | Some (typarConstrTyp) ->
            if Zset.mem typar env.SimplifyTypes.singletons then
                leftL "#" $$ typarSubtypeConstraintL denv env typarConstrTyp
            else
                (varL $$ sepL ":>" $$ typarSubtypeConstraintL denv env typarConstrTyp) |> bracketL

        | _ -> varL

      
    (** Layout type parameter constraints, taking typeSimplificationInfo into account  *)
    and constraintsL denv env cxs = 
        
        
        (* Internally member constraints get attached to each type variable in their support. *)
        (* This means we get too many constraints being printed. *)
        (* So we normalize the constraints to eliminate duplicate member constraints *)
        let cxs = 
            cxs  
            |> gen_setify (fun (_,cx1) (_,cx2) ->
                     match cx1,cx2 with 
                     | TTyparMayResolveMemberConstraint(traitInfo1,_),
                       TTyparMayResolveMemberConstraint(traitInfo2,_) -> traits_aequiv denv.g tyeq_env_empty traitInfo1 traitInfo2
                     | _ -> false) in
                     
        let cxsL = mapConcat (constraintL' denv env) cxs in
        match cxsL with 
        | [] -> emptyL 
        | _ -> wordL "when" $$ sepListL (wordL "and") cxsL

    (** Layout constraints, taking typeSimplificationInfo into account  *)
    and constraintL' denv env (tp,tpc) =
        match tpc with 
        | TTyparCoercesToType(tpct,m) -> 
            [typarL' denv env tp $$ wordL ":>" --- typarSubtypeConstraintL denv env tpct]
        | TTyparMayResolveMemberConstraint(traitInfo,_) ->
            [traitL denv env traitInfo]
        | TTyparDefaultsToType(_,ty,m) ->
           if denv.showTyparDefaultConstraints then [wordL "default" $$ typarL' denv env tp $$ wordL " :" $$ typL' denv env ty]
           else []
        | TTyparIsEnum(ty,m) ->
           [typarL' denv env tp $$ wordL ":" $$ tyappL denv env (wordL "enum") 2 true [ty]]
        | TTyparIsDelegate(aty,bty,m) ->
           [typarL' denv env tp $$ wordL ":" $$ tyappL denv env (wordL "delegate") 2 false [aty;bty]]
        | TTyparSupportsNull(m) ->
           [typarL' denv env tp $$ wordL ":" $$ wordL "null" ]
        | TTyparIsNotNullableValueType(m) ->
           [typarL' denv env tp $$ wordL ":" $$ wordL "struct" ]
        | TTyparIsReferenceType(m) ->
           [typarL' denv env tp $$ wordL ":" $$ wordL "not struct" ]
        | TTyparSimpleChoice(tys,m) ->
           [typarL' denv env tp $$ wordL ":" $$ bracketL (sepListL (sepL "|") (map (typL' denv env) tys)) ]
        | TTyparRequiresDefaultConstructor(m) -> 
           [typarL' denv env tp $$ wordL ":" $$ bracketL (wordL "new : unit -> " $$ (typarL' denv env tp))]

    (** Layout a subtype constraint *)
    and typarSubtypeConstraintL denv env tpct =
        match tpct with
        | TTyparSubtypeConstraintFromFS ty            -> typL' denv env ty
        | TTyparSubtypeConstraintFromIL (scoref,ilty) -> wordL "obj (* unexpected IL type *)"

    and traitL denv env (TTrait(tys,nm,memFlags,argtys,rty)) =
      let stat = memFlagsL denv.htmlHideRedundantKeywords memFlags in 
      let tys = gen_setify (type_equiv denv.g) tys in
      let tysL = 
          match tys with 
          | [ty] -> typL' denv env ty 
          | tys -> bracketL (typesWithPrecL denv env 2 (wordL "or") tys) in 
      tysL $$ wordL ":"  ---  
          bracketL (stat ++ wordL (demangle_operator nm) $$ wordL ":" ---
                  ((typesWithPrecL denv env 2 (wordL "*") argtys ++ wordL "->") ++ (typL' denv env rty)))


    (** Layout type arguments, either NAME<ty,...,ty> or (ty,...,ty) NAME *)
    and tyappL denv env tcL prec prefix args =
        if prefix  then 
            match args with
            | [] -> tcL
            | [arg] -> tcL $$ sepL "<" $$ (typeWithPrecL denv env 4 arg) $$ rightL ">"          
            | args -> bracketIfL (prec <= 1) (tcL $$ angleL (typesWithPrecL denv env 2 (sepL ",") args))
        else
            begin match args with
            | []    -> tcL
            | [arg] ->  typeWithPrecL denv env 2 arg $$ tcL
            | args  -> bracketIfL (prec <= 1) (bracketL (typesWithPrecL denv env 2 (sepL ",") args) --- tcL)
            end

    (** Layout a type, taking precedence into account to insert brackets where needed *)
    and typeWithPrecL denv env prec typ =

        match strip_tpeqns typ with 

        (* Layout a type application *)
        | TType_app (tc,args) -> 
            tyappL denv env (tcrefL denv tc) prec (prefix_display_of_tcref tc) args 

        (* Layout a tuple type *)
        | TType_tuple t ->
            bracketIfL (prec <= 2) (typesWithPrecL denv env 2 (wordL "*") t)

        (* Layout a first-class generic type. *)
        | TType_forall (tps,tau) ->
            let tauL = typeWithPrecL denv env prec tau in
            begin match tps with 
            | []  -> tauL
            | [h] -> typarL' denv env h $$ rightL "." --- tauL
            | (h::t) -> spaceListL (map (typarL' denv env) (h::t)) $$ rightL "." --- tauL
            end

        (* Layout a function type. *)
        | TType_fun (dty,rty) ->
            let rec fun_typL soFarL = function
              | TType_fun (dty,rty) ->
                  let soFarL = soFarL ++ (typeWithPrecL denv env 4 dty $$ wordL "->") in
                  fun_typL soFarL rty
              | rty -> soFarL ++ typeWithPrecL denv env 5 rty in
            bracketIfL (prec <= 4) (fun_typL emptyL (TType_fun (dty,rty)))

        (* Layout a type variable . *)
        | TType_var r ->
            typarL' denv env r

        | TType_unknown -> wordL "'??" 
        | TType_modul_bindings -> wordL "<struct>"

    (** Layout a list of types, separated with the given separator, either '*' or ',' *)
    and typesWithPrecL denv env prec sep typl = 
        sepListL sep (map (typeWithPrecL denv env prec) typl)

    (** Layout a single type, taking typeSimplificationInfo into account *)
    and typL' denv env typ = 
        typeWithPrecL denv env 5 typ

    (** Layout a single type used as the type of a member or value *)
    let topTypeL denv env argInfos rty cxs =
        if denv.html && exists (snd >> isTTyparSupportsStaticMethod) cxs then
            wordL "overloaded"
        else
            (* Parenthesize the return type to match the arity *)      
            let rtyL  = typeWithPrecL denv env 4 rty in
            let cxsL = constraintsL denv env cxs in
            match argInfos with
            | [] | [[]]    -> rtyL --- cxsL
            | _     -> 

               (* Formay each argument, in cluding its name and type *)
               let argL (ty,TopArgData(argAttribs,idOpt)) = 
                   
                   (* Detect an optional argument *)
                   let isOptionalArg = fsthing_has_attrib denv.g denv.g.attrib_OptionalArgumentAttribute argAttribs in
                   match idOpt, isOptionalArg, try_dest_option_ty denv.g ty with 
                   (* Layout an optional argument *)
                   | Some(id), true, Some(ty) -> 
                       leftL ("?"^id.idText) $$ sepL ":" $$ typeWithPrecL denv env 2 ty 
                   (* Layout an unnamed argument *)
                   | None, _,_ -> 
                       typeWithPrecL denv env 2 ty
                   (* Layout a named argument *)
                   | Some id,_,_ -> 
                        leftL id.idText $$ sepL ":" $$ typeWithPrecL denv env 2 ty in

               let allArgsL = argInfos |> map (map argL >> sepListL (wordL "*")) in
               ((sepListL (wordL "->") allArgsL ++ wordL "->") ++ rtyL) --- cxsL

    let typarDeclsL denv nmL prefix typars = 
        typarDeclsL' denv SimplifyTypes.typeSimplificationInfo0 nmL prefix typars 

    let typL denv typ  = 
        typL' denv SimplifyTypes.typeSimplificationInfo0 typ

    let constraintL denv typars = 
        match constraintL' denv SimplifyTypes.typeSimplificationInfo0 typars  with h::_ -> h | [] -> emptyL

    let typesAndConstraintsL denv taus =
        let _,ptaus,cxs = prettifyN taus in
        let env = SimplifyTypes.collectInfo true ptaus cxs in
        List.map (typL' denv env) ptaus,constraintsL denv env env.SimplifyTypes.postfixConstraints

    let topPrettifiedTypesAndConstraintsL denv argInfos tau cxs = 
        let env = SimplifyTypes.collectInfo true (tau:: map fst argInfos) cxs in
        topTypeL denv env [argInfos] tau env.SimplifyTypes.postfixConstraints

    let topTypAndConstraintsL denv argInfos tau = 
        let _,(argInfos,tau),cxs = prettifyTypesN1 (argInfos,tau) in
        topPrettifiedTypesAndConstraintsL denv argInfos tau cxs

    let memberTypeAndConstraintsL denv argInfos retTy parentTyparTys = 
        let _,(parentTyparTys,argInfos,retTy),cxs = prettifyTypesNN1 (parentTyparTys,argInfos,retTy) in
        (* Filter out the parent typars, which don't get shown in the member signature *)
        let cxs = cxs |> filter (fun (tp,_) -> not (gen_exists (dest_typar_typ >> typar_ref_eq tp) parentTyparTys))  in
        topPrettifiedTypesAndConstraintsL denv argInfos retTy cxs

    (* Layout: type spec - class, datatype, record, abbrev *)

    let memberTypeCoreL denv memberToParentInst (methTypars,argInfos,retTy) = 
       let niceMethodTypars, allTyparInst = 
           let methTyparNames = methTypars |> list_mapi (fun i tp -> if compgen_of_typar tp then sprintf "a%d" (length memberToParentInst + i) else name_of_typar tp) in 
           newPrettyTypars memberToParentInst methTypars methTyparNames in

       let retTy = inst_type allTyparInst retTy in 
       let argInfos = map (map1'2 (inst_type allTyparInst)) argInfos in 

       (* Also format dummy types corresponding to any type variables on the container to make sure they *)
       (* aren't chosen as names for displayed variables. *)
       let memberParentTypars = map fst memberToParentInst in 
       let parentTyparTys = map (mk_typar_ty >> inst_type allTyparInst) memberParentTypars in 

       niceMethodTypars,memberTypeAndConstraintsL denv argInfos retTy parentTyparTys

    let memberTypeL denv v tps argInfos retTy = 
        match partition_val_typars v with
        | Some(_,memberParentTypars,memberMethodTypars,memberToParentInst,_) ->  
           memberTypeCoreL denv memberToParentInst (memberMethodTypars, argInfos, retTy)
        | None -> 
           [],topTypAndConstraintsL denv argInfos retTy 

    let memberSigL denv  (memberToParentInst,nm,methTypars,argInfos,retTy) = 
        let niceMethodTypars,tauL = memberTypeCoreL denv memberToParentInst (methTypars, argInfos, retTy) in
        let nameL = 
            let nameL = wordL (demangle_operator nm) in
            let nameL = if denv.showTyparBinding then typarDeclsL denv nameL true niceMethodTypars else nameL in
            nameL in 
        nameL $$ wordL ":" $$ tauL


    (** Layout a single attibute arg, following the cases of 'gen_attr_arg' in ilxgen.ml *)
    (** This is the subset of expressions we display in the NicePrint pretty printer *)
    (** See also dataExprL - there is overlap between these that should be removed *)
    let rec attribArgL denv arg = 
        match arg with 
        | TExpr_const(c,_,ty) -> 
            if is_enum_typ ty then 
                wordL "enum" $$ angleL (typL denv ty) $$ bracketL (tconstL c)
            else
                tconstL c

        | TExpr_op(TOp_array,[elemTy],args,m) ->
            leftL "[|" $$ semiListL (map (attribArgL denv) args) $$ rightL "|]"

        (* Detect 'typeof<ty>' calls *)
        | TExpr_app(TExpr_val(vref,_,_),_,[ty],[],m) when denv.g.vref_eq vref denv.g.typeof_vref  ->
            leftL "typeof<" $$ typL denv ty $$ rightL ">"

        (* Detect 'typedefof<ty>' calls *)
        | TExpr_app(TExpr_val(vref,_,_),_,[ty],[],m) when denv.g.vref_eq vref denv.g.typedefof_vref  ->
            leftL "typedefof<" $$ typL denv ty $$ rightL ">"

        | TExpr_op(TOp_coerce,_,[arg2],_) ->
            attribArgL denv arg2

        | TExpr_app(TExpr_val(vref,_,_),_,_,[arg1;arg2],_) when denv.g.vref_eq vref denv.g.bitwise_or_vref  ->
            attribArgL denv arg1 $$ wordL "|||" $$ attribArgL denv arg2

        (* Detect explicit enum values *)
        | TExpr_app(TExpr_val(vref,_,_),_,_,[arg1],_) when denv.g.vref_eq vref denv.g.enum_vref  ->
            wordL "enum" ++ bracketL (attribArgL denv arg1)


        | _ -> wordL "(* unsupported attribute argument *)"

    (** Layout arguments of an attribute 'arg1, ..., argN' *)
    and attribArgsL denv args = 
        sepListL (rightL ",") (map (attribArgL denv) args)

    (** Layout an attribute 'Type(arg1, ..., argN)' *)
    and attribL denv (Attrib(k,args,props)) = 
        let argsL = bracketL (attribArgsL denv args) in
        match k with 
        | (ILAttrib(mref)) -> 
            let trimmedName = 
                let name =  (Il.tname_of_tref mref.Il.mrefParent) in 
                match try_drop_suffix name "Attribute" with 
                | Some shortName -> shortName
                | None -> name in
            let tref = { mref.Il.mrefParent with Il.trefName = trimmedName} in
            trefL denv tref ++ argsL

        | (FSAttrib(vref)) -> 
            (* REVIEW: this is not trimming "Attribute" *)
            let rty = retty_of_member_vref vref in 
            let tcref = tcref_of_stripped_typ  rty in 
            tcrefL denv tcref ++ argsL


    (** Layout '[<attribs>]' above another block *)
    and attribsL denv attrs restL = 
        
        if denv.showAttributes then
            (* Don't display DllImport attributes in generated signatures *)
            let attrs = attrs |> filter (is_fs_attrib denv.g denv.g.attrib_DllImportAttribute >> not) in
            let attrs = attrs |> filter (is_fs_attrib denv.g denv.g.attrib_ContextStaticAttribute >> not) in
            let attrs = attrs |> filter (is_fs_attrib denv.g denv.g.attrib_ThreadStaticAttribute >> not) in
            let attrs = attrs |> filter (is_fs_attrib denv.g denv.g.attrib_EntryPointAttribute >> not) in
            let attrs = attrs |> filter (is_fs_attrib denv.g denv.g.attrib_MarshalAsAttribute >> not) in
            let attrs = attrs |> filter (is_fs_attrib denv.g denv.g.attrib_ReflectedDefinitionAttribute >> not) in
            let attrs = attrs |> filter (is_fs_attrib denv.g denv.g.attrib_StructLayoutAttribute >> not) in
            let attrs = attrs |> filter (is_fs_attrib denv.g denv.g.attrib_AutoSerializableAttribute >> not) in
            match attrs with
            | [] -> restL 
            | _  -> squareAngleL (sepListL (rightL ";") (map (attribL denv) attrs)) @@ 
                    restL
        else restL

    let memberL denv v = 
        let membInfo = the(member_info_of_val v) in 
        let arity = the(arity_of_val v) in 
        let id = id_of_val v in 
        let ty = type_of_val v in 
        let attrs = attribs_of_val v in
        let stat = memFlagsL denv.htmlHideRedundantKeywords membInfo.vspr_flags in 
        let tps,argInfos,rty,_ = dest_vspr_typ membInfo arity ty in 
        let mkNameL niceMethodTypars name =       
            let name = demangle_operator name in
            let nameL = if denv.showMemberContainers then tcrefL denv (tcref_of_stripped_typ (enclosing_formal_typ_of_val denv.g v)) $$ rightL ("." ^ name) else wordL name in 
            let nameL = valA nameL in 
            let nameL = if denv.showTyparBinding then typarDeclsL denv nameL true niceMethodTypars else nameL in
            nameL in 

        match membInfo.vspr_flags.memFlagsKind with 
        | MemberKindMember -> 
            let niceMethodTypars,tauL = memberTypeL denv v tps argInfos rty in
            let nameL = mkNameL niceMethodTypars membInfo.vspr_il_name in
            stat --- (nameL $$ wordL ":" $$ tauL)
        | MemberKindClassConstructor  
        | MemberKindConstructor -> 
            let niceMethodTypars,tauL = memberTypeL denv v tps argInfos rty in
            stat ++ wordL "new :" $$ tauL
        | MemberKindPropertyGetSet -> stat
        | MemberKindPropertyGet -> 
            if isNil argInfos then error(Error("invalid form for a property getter. At least one '()' argument is required when using the explicit syntax",id.idRange));
            let argInfos = 
              (* This is gross - it is our way of telling whether the thing is an indexer or not *)
              (* REVIEW: reconsider *)
              if length argInfos > 1 || (length argInfos = 1 && not (is_stripped_tyapp_typ (fst (hd argInfos)) && name_of_tcref (tcref_of_stripped_typ (fst (hd argInfos))) = "Unit")) then 
                argInfos
              else [] in 
            let niceMethodTypars,tauL = memberTypeL denv v tps argInfos rty in 
            let nameL = mkNameL niceMethodTypars (get_property_name membInfo) in
            stat --- (nameL $$ wordL ":" $$ (if isNil argInfos then tauL else tauL --- wordL "with get"))
        | MemberKindPropertySet -> 
            let argInfos,rty = 
              if isNil argInfos then error(Error("invalid form for a property setter. At least one argument is required",id.idRange));
              frontAndBack argInfos in 
            let niceMethodTypars,tauL = memberTypeL denv v tps argInfos (fst rty) in 
            let nameL = mkNameL niceMethodTypars (get_property_name membInfo) in 
            stat --- (nameL $$ wordL ":" $$ (tauL --- wordL "with set"))

    let nonMemberValSpecL denv (tps,v,tau,cxs) =
        let env = SimplifyTypes.collectInfo true [tau] cxs in
        let cxs = env.SimplifyTypes.postfixConstraints in
        let argInfos,rty = dest_top_tau_type (arity2_of_val v  |> TopValData.getArgInfos) tau in
        (* Drop the names from value arguments when printing them *)
        let argInfos = map (map (fun (ty,info) -> ty,TopValData.unnamedTopArg1)) argInfos in
        let nameL = wordL (display_name_of_val v) |> valA (* VAL attribute can be shown in bold *) in
        let nameL = if mutability_of_val v = Mutable then wordL "mutable" ++ nameL else nameL in
        let nameL = 
            if not denv.html && mustinline (inlineFlag_of_val v) then 
                wordL "inline" ++ nameL 
            else 
                nameL in        

        let isOverGeneric = List.length (Zset.elements (free_in_type tau).free_loctypars) < List.length tps in  (* Bug: 1143 *)
        let isTyFunction  = is_tyfunc_of_val v in                                                   (* Bug: 1143, and innerpoly tests *)
        let typarBindingsL = 
            if isTyFunction || isOverGeneric || denv.showTyparBinding then 
                typarDeclsL denv nameL true tps 
            else nameL in
        (wordL "val"  $$ typarBindingsL --- wordL ":") --- topTypeL denv env argInfos rty cxs


    let valSpecL denv v =
        let vL = 
            match member_info_of_val v with 
            | None -> 
                let tps,tau = try_dest_forall_typ (type_of_val v) in
                let tprenaming,ptau,cxs = PrettyTypes.prettify1 tau in 
                let ptps = tps |> generalize_typars |> map (inst_type tprenaming >> dest_typar_typ) in 
                nonMemberValSpecL denv (ptps,v,ptau,cxs)
            | Some _ -> 
                memberL denv v in
        attribsL denv (attribs_of_val v) vL


    let uconstrArgTypesL denv argtys = 
        sepListL (wordL "*") (map (typeWithPrecL denv SimplifyTypes.typeSimplificationInfo0 2) argtys)

    let uconstrL denv prefixL constr =
        let nmL = wordL (demangle_operator constr.uconstr_id.idText) in
        match constr |> rfields_of_uconstr |> List.map formal_typ_of_rfield with
        | []     -> (prefixL $$ nmL)
        | argtys -> (prefixL $$ nmL $$ wordL "of") --- uconstrArgTypesL denv argtys

    let uconstrsL denv constrs =
        let prefixL = if length constrs > 1 then wordL "|" else emptyL in
        map (uconstrL denv prefixL) constrs
        
    let rfspecL denv fld =
        let lhs = wordL fld.rfield_id.idText in
        let lhs = if fld.rfield_mutable then wordL "mutable" --- lhs else lhs in
        (lhs $$ rightL ":") --- typL denv fld.rfield_type

    let tyconReprL denv (repr,tycon) = 
        match repr with 
        | TRecdRepr flds ->
            let rfrefL fld = rfspecL denv fld $$ rightL ";" in
            flds |> true_rfields_of_rfield_tables |> map rfrefL |> aboveListL |> braceL 
        | TFsObjModelRepr r -> 
            begin match r.tycon_objmodel_kind with 
            | TTyconDelegate (TSlotSig(nm,typ, _,_,paraml, rty)) ->
                wordL "delegate of" --- topTypeL denv SimplifyTypes.typeSimplificationInfo0 [map (fun (TSlotParam(_,ty,_,_,_,_)) -> (ty, TopValData.unnamedTopArg1)) paraml] rty []
            | _ ->
                match r.tycon_objmodel_kind with
                | TTyconEnum -> 
                    r.fsobjmodel_rfields 
                    |> true_rfields_of_rfield_tables 
                    |> map (fun f -> (match literal_value_of_rfield f with 
                                      | None -> emptyL
                                      | Some c -> wordL "| " $$ wordL (name_of_rfield f) $$ wordL " = " $$ tconstL c))
                    |> aboveListL
                | _ -> 
                    let start = 
                        match r.tycon_objmodel_kind with
                        | TTyconClass -> "class" 
                        | TTyconInterface -> "interface" 
                        | TTyconStruct -> "struct" 
                        | TTyconEnum -> "enum" 
                        | _ -> failwith "???" in
                    let inherits = 
                       match r.tycon_objmodel_kind, (tcaug_of_tycon tycon).tcaug_super with
                       | TTyconClass,Some super -> [wordL  "inherit" $$ (typL denv super)] 
                       | TTyconInterface,_ -> 
                         let tcaug = tcaug_of_tycon tycon in
                         tcaug.tcaug_implements 
                           |> List.filter (fun (ity,compgen,_) -> not compgen)
                           |> List.map (fun (ity,compgen,_) -> wordL  "inherit" $$ (typL denv ity))
                       | _ -> [] in
                    let vsprs = 
                        adhoc_of_tycon tycon 
                        |> filter (fun v -> isNone (the(member_info_of_vref v)).vspr_implements_slotsig) 
                        |> filter vref_is_abstract 
                        |> map (fun vref -> valSpecL denv (deref_val vref)) in
                    let vals  = 
                        r.fsobjmodel_rfields 
                        |> true_rfields_of_rfield_tables 
                        |> map (fun f -> (if f.rfield_static then wordL "static" else emptyL) $$ wordL "val" $$ rfspecL denv f) in
                    (wordL start @@-- aboveListL (inherits @ vsprs @ vals)) @@ wordL "end"
            end
        | TFiniteUnionRepr constrs        -> constrs.funion_constrs |> uconstrs_of_uconstr_tables |> uconstrsL denv |> aboveListL 
        | TAsmRepr s                      -> wordL "(# ...)"
        | TIlObjModelRepr (scoref,enc,td) -> trefL denv (Il.mk_nested_tref(scoref,List.map Il.name_of_tdef enc,Il.name_of_tdef td))

    let breakTypeDefnEqn repr =
      (* Q: When to force a break? "type tyname = <HERE> repn"
       * A: When repn is class or datatype constructors (not single one).
       *)
        match repr with 
        | TFiniteUnionRepr r    -> (length (r.funion_constrs |> uconstrs_of_uconstr_tables) > 1) 
        | TRecdRepr _ | TFsObjModelRepr _ 
        | TAsmRepr _ | TIlObjModelRepr _  -> false

    let tyconSpecL all denv typewordL tycon =
        let prefix = prefix_display_of_tycon tycon in
        let nameL = wordL (display_name_of_tycon tycon) in
        let lhsL =
            let tps = typars_of_tycon tycon in 
            let tpsL = typarDeclsL denv nameL prefix tps in 
            typewordL $$ tpsL in
        let suffixL = 
            if all then 
                let tcaug = tcaug_of_tycon tycon in
                let adhoc = 
                    adhoc_of_tycon tycon  
                    |> filter (vref_is_abstract >> not) 
                (* Don't print individual methods forming interface implementations - these are currently never exported *)
                    |> filter (fun v -> match (the(member_info_of_vref v)).vspr_implements_slotsig with 
                                        | Some (TSlotSig(_,oty,_,_,_,_)) -> not (is_interface_typ oty)
                                        | None -> true) in 
                let iimpls = 
                    match repr_of_tycon tycon with 
                    | Some (TFsObjModelRepr r) when r.tycon_objmodel_kind = TTyconInterface -> []
                    | _ -> tcaug.tcaug_implements in 
                let iimpls = iimpls |> filter (fun (ty,compgen,m) -> not compgen) in
                (* if TTyconInterface, the iimpls should be printed as inheritted interfaces *)
                if (isNil adhoc && isNil iimpls) 
                then emptyL 
                else 
                  let iimplsLs = iimpls |> map (fun (ty,compgen,m) -> wordL "interface" --- typL denv ty) in
                  let adhocLs  = adhoc  |> map (fun vref -> valSpecL denv (deref_val vref)) in
                  (wordL "with" @@-- aboveListL (iimplsLs @ adhocLs)) @@ wordL "end"
            else
                emptyL in
        let reprL = 
            match repr_of_tycon tycon with 
            | Some a -> let brk  = breakTypeDefnEqn a in
                        let rhsL = tyconReprL denv (a,tycon) @@ suffixL in
                        if brk then (lhsL $$ wordL "=") @@-- rhsL
                        else        (lhsL $$ wordL "=") ---  rhsL
            | None   -> match abbrev_of_tycon tycon with
                              | None   -> lhsL @@-- suffixL
                              | Some a -> (lhsL $$ wordL "=") --- (typL denv a @@ suffixL) in
        attribsL denv (attribs_of_tycon tycon) reprL

    let prettyTypeL denv typ = 
        let tprenaming,typ,cxs = PrettyTypes.prettify1 typ in 
        let env = SimplifyTypes.collectInfo true [typ] cxs in
        let cxsL = constraintsL denv env env.SimplifyTypes.postfixConstraints in
        typeWithPrecL denv env 2 typ  --- cxsL

    (* Layout: exception spec *)
      
    let exnSpecReprL denv repr =
        match repr with 
        | TExnAbbrevRepr ecref -> wordL "=" --- item_refL true denv demangled_name_of_ecref pubpath_of_tcref ecref
        | TExnAsmRepr tref     -> wordL "=" --- wordL "(# ...)"
        | TExnNone             -> emptyL
        | TExnFresh r          -> match true_rfields_of_rfield_tables r with
                                  | []  -> emptyL
                                  | r -> wordL "of" --- uconstrArgTypesL denv (List.map formal_typ_of_rfield r)

    let exnSpecL denv constr =
        let nm = (demangled_name_of_exnc constr) in 
        let exnL = wordL "exception" $$ wordL nm in
        exnL $$ exnSpecReprL denv (exn_repr_of_tycon constr)


    (* Layout: module spec *)

    let tycon_specsL denv tycons =
        match tycons with 
        | [] -> emptyL
        | [h] when tycon_is_exnc h -> exnSpecL denv h
        | h :: t -> 
            let x  = tyconSpecL true denv (wordL "type") h in
            let xs = map (tyconSpecL true denv (wordL "and")) t in
            aboveListL (x::xs)

    (* "dblock" is a local for output_module *)
   let inferred_sig_of_structL showHeader denv expr =

        let rec imexprL denv x = 
            match x with  
            (* | TMTyped(mty,e,m) -> imexprL denv  e  *)
            | TMTyped(mty,def,m) -> imdefL denv def
            
        and imdefsL denv x = aboveListL (map (imdefL denv) x)

        and imdefL denv x = 
            let filterVal v = not (compgen_of_val v) && isNone (member_info_of_val v) in
            match x with 
            | TMDefRec(tycons,binds,m) -> 
                 tycon_specsL denv tycons @@ 
                 (binds |> map var_of_bind |> filter filterVal |> map (valSpecL denv) |> aboveListL)
            | TMDefLet(bind,m)  -> ([bind] |> map var_of_bind |> filter filterVal |> map (valSpecL denv) |> aboveListL)
            | TMDefs(defs) -> imdefsL denv defs
            | TMAbstract(mexpr) -> imexprL denv mexpr
            | TMDefModul(TMBind(tycon, def)) -> 

                let id = id_of_modul tycon in 
                let cpath = cpath_of_modul tycon in 
                let k = mkind_of_modul tycon in 

                let denv = denv_add_open_path (path_of_cpath cpath) denv in 
                let basic = (imdefL denv def) in 
                if k = Namespace 
                then basic
                else if (length (access_path_of_cpath cpath) > 0 && 
                         let front,last = frontAndBack (access_path_of_cpath cpath) in 
                         List.for_all (fun (_,istype) -> istype = Namespace) front) 
                then (if showHeader then wordL ("module "^text_of_path (path_of_cpath cpath)) else emptyL) @@-- basic
                else ((wordL "module" $$ wordL id.idText $$ wordL ": begin") @@-- basic) @@ wordL "end" in 
        imexprL denv expr

    type decl_spec = DVal of val_spec | DTycon of tycon_spec | DModul of modul_spec
    let range_of_decl = function
      | DVal   v -> range_of_val   v
      | DTycon t -> range_of_tycon t
      | DModul m -> range_of_modul m      

    let rec modul_specL denv (mtype:modul_typ) =
      (* Have:
       *   modul - provides (valspec)* - and also types, exns and submodules.
       * Each defines a decl block on a given range.
       * Can sort on the ranges to recover the original declaration order.
       *)
      (* REVIEW: consider a better way to keep decls in order. *)
      let decl_specs : decl_spec list =
        concat
          [mtype.mtyp_vals       |> Namemap.range |> filter (fun v -> not (compgen_of_val v) && member_info_of_val v = None) |> map (fun v -> DVal v);
           mtype.mtyp_tycons     |> Namemap.range |> map (fun x -> DTycon x);
           mtype.mtyp_submoduls  |> Namemap.range |> map (fun x -> DModul x);
          ]
      in
      let decl_specs = sort (orderOn range_of_decl range_ord) decl_specs in
      let decl_specL = function
        | DVal  vspec  -> valSpecL   denv vspec
        | DTycon tycon -> if tycon_is_exnc tycon then exnSpecL   denv tycon else tyconSpecL true denv (wordL "type") tycon
        | DModul mspec -> modulL denv mspec in
      aboveListL (map decl_specL decl_specs)

    and modulL denv (mspec: modul_spec) = 
      let istype = mkind_of_mtyp (mtyp_of_modul mspec) in 
      let nm     = name_of_modul mspec in 
      let denv   = denv_add_open_modref (mk_local_modref mspec) denv in
      ((wordL (if istype <> Namespace then "module" else "namespace") $$ wordL nm $$ wordL ": begin") @@-- (modul_specL denv (mtyp_of_modul mspec))) @@ wordL "end"

    (*--------------------------------------------------------------------------
    !* Nice printing of a subset of expressions, e.g. for refutations in pattern matching
     *------------------------------------------------------------------------*)

    let rec dataExprL denv expr = dataExprWrapL denv false expr
    and atomL denv expr = dataExprWrapL denv true  expr (* true means bracket if needed to be atomic expr *)

    and dataExprWrapL denv isAtomic expr =
        let wrap = bracketIfL isAtomic in (* wrap iff require atomic expr *)
        match expr with
        | TExpr_const (c,m,ty)                          -> 
            if is_enum_typ ty then 
                wordL "enum" $$ angleL (typL denv ty) $$ bracketL (tconstL c)
            else
                tconstL c

        | TExpr_val (v,flags,m)                         -> wordL (display_name_of_vref v)
        | TExpr_link rX                                 -> dataExprWrapL denv isAtomic (!rX)
        | TExpr_op(TOp_uconstr(c),tyargs,args,m)        -> 
            if denv.g.ucref_eq c denv.g.nil_ucref then wordL "[]"
            else if denv.g.ucref_eq c denv.g.cons_ucref then 
                let rec strip = function (TExpr_op(TOp_uconstr(c),tyargs,[h;t],m)) -> h::strip t | _ -> [] in 
                listL (dataExprL denv) (strip expr)
            else if isNil(args) then 
                wordL (name_of_ucref c) 
            else 
                (wordL (name_of_ucref c) ++ bracketL (commaListL (dataExprsL denv args)))
            
        | TExpr_op(TOp_exnconstr(c),_,args,m)           ->  (wordL (name_of_tcref c) ++ bracketL (commaListL (dataExprsL denv args)))
        | TExpr_op(TOp_tuple,tys,xs,m)                  -> tupleL (dataExprsL denv xs)
        | TExpr_op(TOp_recd (ctor,tc),tinst,xs,m)       -> let fields = instance_rfields_of_tycon (deref_tycon tc) in
                                                           let lay fs x = (wordL fs.rfield_id.idText $$ sepL "=") --- (dataExprL denv x) in
                                                           leftL "{" $$ semiListL (map2 lay fields xs) $$ rightL "}" 
        | TExpr_op(TOp_array,[ty],xs,m)                 -> leftL "[|" $$ semiListL (dataExprsL denv xs) $$ rightL "|]"
        | _ -> wordL "?"
    and dataExprsL denv xs = map (dataExprL denv) xs

    (*--------------------------------------------------------------------------
    !* Print Signatures/Types - ouput functions - old style
     *-------------------------------------------------------------------------- *)
        
    (* A few old-style o/p functions are used, e.g. in tc.ml *)    
    let output_tref                 denv os x    = trefL denv x                 |> bufferL os
    let output_tcref                denv os x    = x |> tcrefL denv             |> bufferL os    
    let output_val_spec             denv os x    = x |> valSpecL denv          |> bufferL os
    let output_typ                  denv os x    = x |> typL denv               |> bufferL os  
    let output_exnc                 denv os x    = x |> exnSpecL denv          |> bufferL os
    let output_typar_constraints    denv os x    = x |> constraintsL denv SimplifyTypes.typeSimplificationInfo0  |> bufferL os
    let string_of_typar_constraints denv x       = x |> constraintsL denv SimplifyTypes.typeSimplificationInfo0  |> showL
    let output_rfield               denv os x    = x |> rfspecL denv |> bufferL os
    let output_tycon all            denv os x    = tyconSpecL all denv (emptyL) x      |> bufferL os
    let output_uconstr              denv os x    = uconstrL denv (wordL "|") x |> bufferL os
    let output_typars               denv nm os tps  = (typarDeclsL denv  (wordL nm) true tps)  |> bufferL os
    let output_typar_constraint     denv os tpc  = output_typar_constraints denv os [tpc]
    let string_of_typ               denv    typ  = typL denv typ                                 |> showL
    let pretty_string_of_typ        denv    typ  = prettyTypeL denv typ                          |> showL
    (* Print members with a qualification showing the type they are contained in *)
    let output_qualified_val_spec denv os v = output_val_spec { denv with showMemberContainers=true; } os v

end

(*--------------------------------------------------------------------------
!* DEBUG layout
 *------------------------------------------------------------------------*)

module DebugPrint = struct
    open Layout
    open PrettyTypes
    (* layout: print #stamp on each id -- very verbose - but sometimes useful *)    
    let layout_stamps = ref false
    let layout_ranges = ref false  

    let intL n          = wordL (string_of_int n )

    let keywordA ly = tagL "Keyword" ly
    let typeA    ly = tagL "Type"    ly

    let namemapL xL xmap = Namemap.fold_range (fun x z -> z @@ xL x) xmap emptyL

    let exconL x = wordL (demangled_name_of_exnc x)
    let bracketIfL x lyt = if x then bracketL lyt else lyt

    let lvalopL x = 
        match x with 
        | LGetAddr  -> wordL "LGetAddr"
        | LByrefGet -> wordL "LByrefGet"
        | LSet      -> wordL "LSet"
        | LByrefSet -> wordL "LByrefSet"

    let angleBracketL l = leftL "<" $$ l $$ rightL ">"
    let angleBracketListL l = angleBracketL (sepListL (sepL ",") l)


    let memFlagsL hide memFlags = 
        let stat = if hide || memFlags.memFlagsInstance || (memFlags.memFlagsKind = MemberKindConstructor) then emptyL else wordL "static" in
        let stat = if not memFlags.memFlagsAbstract && memFlags.memFlagsVirtual then stat ++ wordL "virtual" 
                   else if not hide && memFlags.memFlagsAbstract then stat ++ wordL "abstract" 
                   else if memFlags.memFlagsOverride then stat ++ wordL "override" 
                   else stat in
        (* let stat = if memFlags.memFlagsFinal then stat ++ wordL "final" else stat in  *)
        stat

    let tyconL tc = 
        let tcL = wordL (display_name_of_tycon tc) |> typeA in 
        if !layout_stamps then tcL $$ sepL "#" $$ intL (stamp_of_tycon tc) else tcL
    let tcrefL tc = tyconL (deref_tycon tc)


    let rec auxTypeL env typ = auxTypeWrapL env false typ

    and auxTypeAtomL env typ = auxTypeWrapL env true  typ

    and auxTyparsL env tcL prefix tinst = 
       match tinst with 
       | [] -> tcL
       | [t] -> 
         let tL = auxTypeAtomL env t in 
         if prefix then        tcL $$ angleBracketL tL 
         else            tL $$ tcL 
       | _ -> 
         let tinstL = map (auxTypeL env) tinst in 
         if prefix then                   
             tcL $$ angleBracketListL tinstL
         else  
             tupleL tinstL $$ tcL
           
    and auxTypeWrapL env isAtomic typ = 
        let wrap x = NicePrint.bracketIfL isAtomic x in (* wrap iff require atomic expr *)
        match strip_tpeqns typ with
        | TType_forall (typars,rty) -> 
           (leftL "!" $$ typarDeclsL typars --- auxTypeL env rty) |> wrap
        | TType_app (tcref,tinst)   -> 
           let prefix = prefix_display_of_tcref tcref in 
           let tcL = tcrefL tcref in 
           auxTyparsL env tcL prefix tinst
        | TType_tuple typs          -> sepListL (wordL "*") (map (auxTypeAtomL env) typs) |> wrap
        | TType_fun (f,x)           -> ((auxTypeAtomL env f $$ wordL "->") ++ auxTypeL env x) |> wrap
        | TType_var typar           -> auxTyparWrapL env isAtomic typar 
        | TType_unknown             -> wordL "unknownT"
        | TType_modul_bindings              -> wordL "structT"

    and auxTyparWrapL env isAtomic typar =
          let wrap x = NicePrint.bracketIfL isAtomic x in (* wrap iff require atomic expr *)  
          (* There are several cases for pprinting of typar.
           * 
           *   'a              - is multiple  occurance.
           *   #Type           - inplace coercion constraint and singleton
           *   ('a :> Type)    - inplace coercion constraint not singleton
           *   ('a.opM : S->T) - inplace operator constraint
           *)
          let tpL =
            wordL (prefix_of_static_req (static_req_of_typar typar)
                   ^ prefix_of_rigid typar
                   ^ display_name_of_typar typar) in
          let varL =
              if !layout_stamps then 
                  tpL $$ sepL "#" $$ intL (stamp_of_typar typar) 
              else tpL in

          match Zmap.tryfind typar env.SimplifyTypes.inplaceConstraints with
          | Some (typarConstrTyp) ->
              if Zset.mem typar env.SimplifyTypes.singletons then
                leftL "#" $$ auxTyparConstraintTypL env typarConstrTyp
              else
                (varL $$ sepL ":>" $$ auxTyparConstraintTypL env typarConstrTyp) |> wrap
          | _ -> varL

    and auxTypar2L     env typar = auxTyparWrapL env false typar

    and auxTyparAtomL env typar = auxTyparWrapL env true  typar

    and auxTyparConstraintTypL env x = 
        match x with 
        | TTyparSubtypeConstraintFromFS ty -> auxTypeL env ty
        | TTyparSubtypeConstraintFromIL _  -> wordL "obj (* unexpected IL type *)"

    and auxTraitL env (TTrait(tys,nm,memFlags,argtys,rty)) =
        let stat = memFlagsL false memFlags in 
        let argsL = sepListL (wordL "*") (map (auxTypeAtomL env) argtys) in
        let resL  = auxTypeL env rty in
        let methodTypeL = (argsL $$ wordL "->") ++ resL in
        bracketL (stat ++ bracketL (sepListL (wordL "or") (map (auxTypeAtomL env) tys)) ++ wordL "member" --- (wordL nm $$ wordL ":" -- methodTypeL))

    and auxTyparConstraintL env (tp,tpc) = 
        match tpc with
        | TTyparCoercesToType(typarConstrTyp,m) ->
            auxTypar2L env tp $$ wordL ":>" --- auxTyparConstraintTypL env typarConstrTyp
        | TTyparMayResolveMemberConstraint(traitInfo,_) ->
            auxTypar2L env tp $$ wordL ":"  --- auxTraitL env traitInfo
        | TTyparDefaultsToType(_,ty,m) ->
            wordL "default" $$ auxTypar2L env tp $$ wordL ":" $$ auxTypeL env ty
        | TTyparIsEnum(ty,m) ->
            auxTypar2L env tp $$ wordL ":" $$ auxTyparsL env (wordL "enum") true [ty]
        | TTyparIsDelegate(aty,bty,m) ->
            auxTypar2L env tp $$ wordL ":" $$ auxTyparsL env (wordL "delegate") false [aty; bty]
        | TTyparSupportsNull(m) ->
            auxTypar2L env tp $$ wordL ":" $$ wordL "null"
        | TTyparIsNotNullableValueType(m) ->
            auxTypar2L env tp $$ wordL ":" $$ wordL "struct"
        | TTyparIsReferenceType(m) ->
            auxTypar2L env tp $$ wordL ":" $$ wordL "not struct"
        | TTyparSimpleChoice(tys,m) ->
            auxTypar2L env tp $$ wordL ":" $$ bracketL (sepListL (sepL "|") (map (auxTypeL env) tys))
        | TTyparRequiresDefaultConstructor(m) ->
            auxTypar2L env tp $$ wordL ":" $$ bracketL (wordL "new : unit -> " $$ (auxTypar2L env tp))

    and auxTyparConstraintsL env x = 
        match x with 
        | []   -> emptyL
        | cxs -> wordL "when" --- aboveListL (map (auxTyparConstraintL env) cxs)    

    and typarL     tp = auxTypar2L     SimplifyTypes.typeSimplificationInfo0 tp 
    and typarAtomL tp = auxTyparAtomL SimplifyTypes.typeSimplificationInfo0 tp

    and typeAtomL tau =
        let tau,cxs = tau,[] in
        let env = SimplifyTypes.collectInfo false [tau] cxs in  
        match env.SimplifyTypes.postfixConstraints with
        | [] -> auxTypeAtomL env tau
        | _ -> bracketL (auxTypeL env tau --- auxTyparConstraintsL env env.SimplifyTypes.postfixConstraints)
          
    and typeL tau =
        let tau,cxs = tau,[] in  
        let env = SimplifyTypes.collectInfo false [tau] cxs in
        match env.SimplifyTypes.postfixConstraints with
        | [] -> auxTypeL env tau |> typeA
        | _ -> (auxTypeL env tau --- auxTyparConstraintsL env env.SimplifyTypes.postfixConstraints) |> typeA

    and typarDeclL tp =
        let tau,cxs = mk_typar_ty tp,(map (fun x -> (tp,x)) (constraints_of_typar tp)) in  
        let env = SimplifyTypes.collectInfo false [tau] cxs in
        match env.SimplifyTypes.postfixConstraints with
        | [] -> auxTypeL env tau |> typeA
        | _ -> (auxTypeL env tau --- auxTyparConstraintsL env env.SimplifyTypes.postfixConstraints) |> typeA
    and typarDeclsL tps = angleBracketListL (map typarDeclL       tps) 

    (*--------------------------------------------------------------------------
    !* DEBUG layout - types
     *------------------------------------------------------------------------*)
      
    let rangeL m = wordL (string_of_range m)

    let instL tyL tys =
        match tys with
        | []  -> emptyL
        | tys -> sepL "@[" $$ commaListL (map tyL tys) $$ rightL "]"

    let vrefL  vr  = 
        let vrL = wordL (name_of_vref vr) in 
        if !layout_stamps then vrL $$ sepL "#" $$ intL (stamp_of_vref vr) else vrL

    let attribL (Attrib(k,args,props)) = 
        leftL "[<" $$ 
        (match k with 
         | ILAttrib (ilmeth) -> wordL (Il.name_of_mref ilmeth)
         | FSAttrib (vref)   -> vrefL vref) $$
        rightL ">]"
    
    let attribsL attribs = aboveListL (map attribL attribs)

    let arityInfoL (TopValInfo (ntps,_,_) as tvd) = 
        let ns = TopValData.aritiesOfArgs tvd in 
        intL ntps $$ sepL "," $$ commaListL (map intL ns)

    let vspecL vspec =
        let vsL = wordL (decompileOpName (name_of_val vspec)) in
        if not !layout_stamps then vsL else 
        let vsL = vsL $$ sepL "#" $$ intL (stamp_of_val vspec) in
        let vsL = vsL $$ rightL (if isSome(pubpath_of_val vspec) then "+" else "-") in
        let vsL = vsL -- attribsL (attribs_of_val vspec) in
        vsL

    let vspecTyL      v =
        (vspecL v
          $$ (if  mustinline (inlineFlag_of_val v) then wordL "inline " else emptyL) 
          $$ (if mutability_of_val v = Mutable then wordL "mutable " else emptyL)
          $$ wordL ":") -- typeL (type_of_val v)

    let tslotsigL(TSlotSig(nm,typ,tps1,tps2,pms,rty)) =
        (wordL "slot" --- (wordL nm) $$ wordL "@" $$ typeL typ) --
          (wordL "LAM" --- spaceListL (map typarL       tps1) $$ rightL ".") ---
          (wordL "LAM" --- spaceListL (map typarL       tps2) $$ rightL ".") ---
          (wordL "...") ---
          (typeL rty) 

    let rec vsprL membInfo = 
        (aboveListL [ wordL membInfo.vspr_il_name ;
                      wordL "membInfo-slotsig!" $$ optionL tslotsigL membInfo.vspr_implements_slotsig ]) 
    and vspecAtBindL  v = 
        let mutL w = 
            match mutability_of_val v with
            | Immutable         -> w
            | Mutable -> wordL "mutable"     ++ w in
        let vL = if not !layout_stamps then vspecL v else (vspecL v $$ wordL ":") --- typeAtomL (type_of_val v) in
        (mutL vL) --- (aboveListL [ wordL ":" $$ typeL (type_of_val v);
                                  wordL "!" $$ optionL vsprL (member_info_of_val v);
                                  wordL "#" $$ optionL arityInfoL (arity_of_val v)])

    let exncL ec = wordL (demangled_name_of_exnc ec)
    let ucrefL ucr = wordL (name_of_ucref ucr)
    let rfrefL ucr = wordL (name_of_rfref ucr)

    (*--------------------------------------------------------------------------
    !* DEBUG layout - bind, expr, dtree etc.
     *------------------------------------------------------------------------*)

    let identL id = wordL id.idText  

    
    let tyconSpecL tycon =
        let lhsL = wordL "type" $$ tyconL tycon $$ typarDeclsL (typars_of_tycon tycon) in
        let lhsL = lhsL --- attribsL (attribs_of_tycon tycon) in
        let suffixL = 
            let tcaug = tcaug_of_tycon tycon in
            let adhoc = adhoc_of_tycon tycon  |> filter (vref_is_abstract >> not) in
            (* Don't print individual methods forming interface implementations - these are currently never exported *)
            let adhoc = adhoc |> filter (fun v -> isNone (the(member_info_of_vref v)).vspr_implements_slotsig) in 
            let iimpls = 
                match repr_of_tycon tycon with 
                | Some (TFsObjModelRepr r) when r.tycon_objmodel_kind = TTyconInterface -> []
                | _ -> tcaug.tcaug_implements in 
            let iimpls = iimpls |> filter (fun (ty,compgen,m) -> not compgen) in
            (* if TTyconInterface, the iimpls should be printed as inheritted interfaces *)
            if (isNil adhoc && isNil iimpls) 
            then emptyL 
            else 
                let iimplsLs = iimpls |> map (fun (ty,compgen,m) -> wordL "interface" --- typeL ty) in
                let adhocLs  = adhoc  |> map (fun vref -> vspecAtBindL  (deref_val vref)) in
                (wordL "with" @@-- aboveListL (iimplsLs @ adhocLs)) @@ wordL "end" in

        let uconstrArgTypesL argtys = sepListL (wordL "*") (map typeL argtys) in

        let uconstrL prefixL constr =
            let nmL = wordL (demangle_operator constr.uconstr_id.idText) in
            match constr |> rfields_of_uconstr |> List.map formal_typ_of_rfield with
            | []     -> (prefixL $$ nmL)
            | argtys -> (prefixL $$ nmL $$ wordL "of") --- uconstrArgTypesL argtys in

        let uconstrsL constrs =
            let prefixL = if length constrs > 1 then wordL "|" else emptyL in
            map (uconstrL prefixL) constrs in
            
        let rfspecL fld =
            let lhs = wordL fld.rfield_id.idText in
            let lhs = if fld.rfield_mutable then wordL "mutable" --- lhs else lhs in
            (lhs $$ rightL ":") --- typeL fld.rfield_type in

        let tyconReprL (repr,tycon) = 
            match repr with 
            | TRecdRepr flds ->
                let rfrefL fld = rfspecL fld $$ rightL ";" in
                flds |> true_rfields_of_rfield_tables |> map rfrefL |> aboveListL  
            | TFsObjModelRepr r -> 
                begin match r.tycon_objmodel_kind with 
                | TTyconDelegate (TSlotSig(nm,typ, _,_,paraml, rty)) ->
                    wordL "delegate ..."
                | _ ->
                    let start = match r.tycon_objmodel_kind with
                        TTyconClass -> "class" | TTyconInterface -> "interface" | TTyconStruct -> "struct" | TTyconEnum -> "enum" | _ -> failwith "???" in
                    let inherits = 
                       match r.tycon_objmodel_kind, (tcaug_of_tycon tycon).tcaug_super with
                       | TTyconClass,Some super -> [wordL  "inherit" $$ (typeL super)] 
                       | TTyconInterface,_ -> 
                         let tcaug = tcaug_of_tycon tycon in
                         tcaug.tcaug_implements 
                           |> List.filter (fun (ity,compgen,_) -> not compgen)
                           |> List.map (fun (ity,compgen,_) -> wordL  "inherit" $$ (typeL ity))
                       | _ -> [] in
                    let vsprs = adhoc_of_tycon tycon |> filter vref_is_abstract |> map (fun vref -> vspecAtBindL (deref_val vref)) in
                    let vals  = r.fsobjmodel_rfields |> true_rfields_of_rfield_tables |> map (fun f -> (if f.rfield_static then wordL "static" else emptyL) $$ wordL "val" $$ rfspecL f) in
                    (wordL start @@-- aboveListL (inherits @ vsprs @ vals)) @@ wordL "end"
                end
            | TFiniteUnionRepr constrs        -> constrs.funion_constrs |> uconstrs_of_uconstr_tables |> uconstrsL |> aboveListL 
            | TAsmRepr s                      -> wordL "(# ... #)"
            | TIlObjModelRepr (_,_,td) -> wordL td.tdName in
        let reprL = 
            match repr_of_tycon tycon with 
            | Some a -> let rhsL = tyconReprL (a,tycon) @@ suffixL in
                        (lhsL $$ wordL "=") @@-- rhsL
            | None   -> match abbrev_of_tycon tycon with
                              | None   -> lhsL @@-- suffixL
                              | Some a -> (lhsL $$ wordL "=") --- (typeL a @@ suffixL) in
        reprL

        
    (*--------------------------------------------------------------------------
    !* layout - bind, expr, dtree etc.
     *------------------------------------------------------------------------*)

    let rec bindL (TBind(v,repr)) =
        vspecAtBindL v --- (wordL "=" $$ exprL repr)

    and exprL expr = exprWrapL false expr
    and atomL expr = exprWrapL true  expr (* true means bracket if needed to be atomic expr *)

    and letRecL binds bodyL = 
        let eqnsL = mapHT (fun bind -> wordL "rec" $$ bindL bind $$ wordL "in")
                          (fun bind -> wordL "and" $$ bindL bind $$ wordL "in") binds in
        (aboveListL eqnsL @@ bodyL) 

    and letL bind bodyL = 
        let eqnL = wordL "let" $$ bindL bind $$ wordL "in" in
        (eqnL @@ bodyL) 
                                                               
    and exprWrapL isAtomic expr =
        let wrap = bracketIfL isAtomic in (* wrap iff require atomic expr *)
        let lay =
            match expr with
            | TExpr_const (c,m,ty)                          -> NicePrint.tconstL c
            | TExpr_val (v,flags,m)                         -> let xL = vspecL (deref_val v) in
                                                               let xL =
                                                                   if not !layout_stamps then xL else
                                                                   let tag = match v with
                                                                     | Ref_private _    -> ""
                                                                     | Ref_nonlocal _ -> "!!" in
                                                                   xL $$ rightL tag in
                                                               let xL =
                                                                   match flags with
                                                                     | CtorValUsedAsSelfInit    -> xL $$ rightL "<selfinit>"
                                                                     | CtorValUsedAsSuperInit -> xL $$ rightL "<superinit>"
                                                                     | VSlotDirectCall -> xL $$ rightL "<vdirect>"
                                                                     | NormalValUse -> xL in
                                                               xL
            | TExpr_seq (x0,x1,flag,m)                      -> (let flag = match flag with
                                                                  | NormalSeq   -> "; (*Seq*)"
                                                                  | ThenDoSeq   -> "; (*ThenDo*)" in
                                                                ((exprL x0 $$ rightL flag) @@ exprL x1) |> wrap)
            | TExpr_lambda(lambda_id ,basevopt,argvs,body,m,rty,_)   -> ((wordL "lam"    $$ optionL vspecAtBindL basevopt $$ spaceListL (map vspecAtBindL argvs)     $$ rightL ".") ++ exprL body) |> wrap
            | TExpr_tlambda(lambda_id,argtyvs,body,m,rty,_) -> ((wordL "LAM"    $$ spaceListL (map typarL       argtyvs) $$ rightL ".") ++ exprL body) |> wrap
            | TExpr_tchoose(argtyvs,body,m)                 -> ((wordL "CHOOSE" $$ spaceListL (map typarL       argtyvs) $$ rightL ".") ++ exprL body) |> wrap
            | TExpr_app (f,fty,tys,argtys,m)              -> let flayout = atomL f in
                                                               appL flayout tys argtys |> wrap
            | TExpr_letrec (binds,body,m,_)                 -> letRecL binds (exprL body) |> wrap
            | TExpr_let    (bind,body,m,_)                 -> letL bind (exprL body) |> wrap
            | TExpr_link rX                                 -> (wordL "RecLink" --- atomL (!rX)) |> wrap
            | TExpr_match (exprm,dtree,targets,m,ty,_)      -> leftL "[" $$ (dtreeL dtree @@ aboveListL (list_mapi targetL (targets |> Array.to_list)) $$ rightL "]")
            | TExpr_op(TOp_uconstr (c),tyargs,args,m)                -> ((ucrefL c (*$$ (instL typeL tyargs)*)) ++ spaceListL (map atomL args)) |> wrap
            | TExpr_op(TOp_exnconstr (ecr),_,args,m)                -> exncL (deref_exnc ecr) $$ bracketL (commaListL (map atomL args))
            | TExpr_op(TOp_tuple,tys,xs,m)                          -> tupleL (map exprL xs)
            | TExpr_op(TOp_recd (ctor,tc),tinst,xs,m)               -> let fields = instance_rfields_of_tycon (deref_tycon tc) in
                                                                       let lay fs x = (wordL fs.rfield_id.idText $$ sepL "=") --- (exprL x) in
                                                                       let ctorL = match ctor with
                                                                         | RecdExpr             -> emptyL
                                                                         | RecdExprIsObjInit-> wordL "(new)" in
                                                                       leftL "{" $$ semiListL (map2 lay fields xs) $$ rightL "}" $$ ctorL
            | TExpr_op(TOp_field_set (rf),tinst,[rx;x],m)        -> (atomL rx --- wordL ".") $$ (rfrefL rf $$ wordL "<-" --- exprL x)
            | TExpr_op(TOp_field_set (rf),tinst,[x],m)        -> (rfrefL rf $$ wordL "<-" --- exprL x)
            | TExpr_op(TOp_field_get (rf),tinst,[rx],m)          -> (atomL rx $$ rightL ".#" $$ rfrefL rf)
            | TExpr_op(TOp_field_get (rf),tinst,[],m)          -> (rfrefL rf)
            | TExpr_op(TOp_field_get_addr (rf),tinst,[rx],m)     -> leftL "&" $$ bracketL (atomL rx $$ rightL ".!" $$ rfrefL rf)
            | TExpr_op(TOp_field_get_addr (rf),tinst,[],m)     -> leftL "&" $$ (rfrefL rf)
            | TExpr_op(TOp_constr_tag_get (tycr),tinst,[x],m)         -> wordL ("#" ^ name_of_tcref tycr ^ ".tag") $$ atomL x
            | TExpr_op(TOp_constr_field_get (c,i),tinst,[x],m)        -> wordL ("#" ^ name_of_ucref c ^ "." ^ string_of_int i) --- atomL x
            | TExpr_op(TOp_constr_field_set (c,i),tinst,[x;y],m)      -> ((atomL x --- (rightL ("#" ^ name_of_ucref c ^ "." ^ string_of_int i))) $$ wordL ":=") --- exprL y
            | TExpr_op(TOp_tuple_field_get (i),tys,[x],m)             -> wordL ("#" ^ string_of_int i) --- atomL x
            | TExpr_op(TOp_coerce,[typ;typ2],[x],m)                   -> atomL x --- (wordL ":>" $$ typeL typ) (* check: or is it typ2? *)
            | TExpr_op(TOp_asm (a,tys),tyargs,args,m)      -> let n = List.length a in
                                                               let instrs = map Ilprint.string_of_instr a in
                                                               let instrs = map wordL instrs in
                                                               let instrs = spaceListL instrs in
                                                               let instrs = leftL "(#" $$ instrs $$ rightL "#)" in
                                                               (appL instrs tyargs args ---
                                                                  wordL ":" $$ spaceListL (map typeAtomL tys)) |> wrap
            | TExpr_op(TOp_lval_op (lvop,vr),_,args,m)       -> (lvalopL lvop $$ vrefL vr --- bracketL (commaListL (map atomL args))) |> wrap
            | TExpr_op(TOp_ilcall ((virt,protect,valu,newobj,superInit,prop,isDllImport,boxthis,mref),tinst,minst,tys),tyargs,args,m)
                                                             -> let meth = Il.name_of_mref mref in
                                                                wordL "ILCall" $$ aboveListL [wordL "meth  " --- wordL meth;
                                                                                              wordL "tinst " --- listL typeL tinst;
                                                                                              wordL "minst " --- listL typeL minst;
                                                                                              wordL "tyargs" --- listL typeL tyargs;
                                                                                              wordL "args  " --- listL exprL args] |> wrap
            | TExpr_op(TOp_array,[ty],xs,m)                 -> leftL "[|" $$ commaListL (map exprL xs) $$ rightL "|]"
            | TExpr_op(TOp_while,[],[x1;x2],m)              -> wordL "while" $$ exprL x1 $$ wordL "do" $$ exprL x2 $$ rightL "}"
            | TExpr_op(TOp_for _,[],[x1;x2;x3],m)           -> wordL "for" $$ aboveListL [(exprL x1 $$ wordL "to" $$ exprL x2 $$ wordL "do"); exprL x3 ] $$ rightL "done"
            | TExpr_op(TOp_try_catch,[_],[x1;x2],m)         -> wordL "try" $$ exprL x1 $$ wordL "with" $$ exprL x2 $$ rightL "}"
            | TExpr_op(TOp_try_finally,[_],[x1;x2],m)       -> wordL "try" $$ exprL x1 $$ wordL "finally" $$ exprL x2 $$ rightL "}"
            | TExpr_op(TOp_bytes _,[],[],m)                 -> wordL "bytearray"
            | TExpr_op(TOp_bytes _,_ ,_ ,m)                 -> wordL "bytes++"       
            | TExpr_op(TOp_get_ref_lval,tyargs,args,m)      -> wordL "GetRefLVal..."
            | TExpr_op(TOp_trait_call _,tyargs,args,m)      -> wordL "traitcall..."
            | TExpr_op(TOp_exnconstr_field_get _,tyargs,args,m) -> wordL "TOp_exnconstr_field_get..."
            | TExpr_op(TOp_exnconstr_field_set _,tyargs,args,m) -> wordL "TOp_exnconstr_field_set..."
            | TExpr_op(TOp_try_finally,tyargs,args,m) -> wordL "TOp_try_finally..."
            | TExpr_op(TOp_try_catch  ,tyargs,args,m) -> wordL "TOp_try_catch..."
            | TExpr_op(_,tys,args,m)                        -> wordL "TExpr_op ..." $$ bracketL (commaListL (map atomL args)) (* TODO *)
            | TExpr_quote (raw,a,m,_)                       -> leftL "<<" $$ atomL a $$ rightL ">>"
            | TExpr_hole      _                             -> wordL "_"
            | TExpr_obj (n,typ,basev,ccall,
                           overrides,iimpls,_,_)              -> let ccallL (vu,mr,tinst,args) = appL (wordL "ccall") tinst args in
                                                               wordL "OBJ:" $$ aboveListL [typeL typ;
                                                                                           exprL ccall;
                                                                                           optionL vspecAtBindL basev;
                                                                                           aboveListL (map overrideL overrides);
                                                                                           aboveListL (map iimplL iimpls)]
            | TExpr_static_optimization (tcs,csx,x,m)       -> let tconstraintL = function TTyconEqualsTycon (s,t) -> (typeL s $$ wordL "=") --- typeL t in
                                                               (wordL "opt" @@- (exprL x)) @@--
                                                                  (wordL "|" $$ exprL csx --- (wordL "when..." (* --- sepListL (wordL "and") (map tconstraintL tcs) *) ))
            in
        (* For tracking ranges through expr rewrites *)
        if !layout_ranges 
        then leftL "{" $$ (rangeL (range_of_expr expr) $$ rightL ":") ++ lay $$ rightL "}"
        else lay

    and mimplsL (TAssembly(modinfos)) = 
        aboveListL (map modinfoL modinfos)
    
    and appL flayout tys args =
        let z = flayout in
        let z = z $$ instL typeL tys in
        let z = z --- sepL "`" --- (spaceListL (map atomL args)) in
        z
       
    and modinfoL (TImplFile(qnm,e)) =
        aboveListL [(wordL "top implementation ") @@-- mexprL e]

    and mexprL x =
        match x with 
        (* | TMTyped(mtyp,rest,_) -> aboveListL [wordL "CONSTRAIN" @@-- mexprL rest @@- (wordL ":"  @@-  mtypeL mtyp)] *)
        | TMTyped(mtyp,defs,m) -> mdefL defs  @@- (wordL ":"  @@-  mtypeL mtyp)
    and mdefsL defs = wordL "Module Defs" @@-- aboveListL(map mdefL defs)
    and mdefL x = 
        match x with 
        | TMDefRec(tycons ,binds,m) ->  aboveListL ((tycons |> map tyconSpecL) @ [letRecL binds emptyL])
        | TMDefLet(bind,m) -> letL bind emptyL
        | TMDefs(defs) -> mdefsL defs; 
        | TMAbstract(mexpr) -> mexprL mexpr
        | TMDefModul(TMBind(tycon, rhs)) -> 
            let nm = id_of_modul tycon in 
            (wordL ("module/namespace" ^ nm.idText) @@-- mdefL rhs); 

    and mtypeL ms =
        aboveListL [namemapL mspecL (submoduls_of_mtyp ms);
                    namemapL vspecTyL ms.mtyp_vals;
                    namemapL tyconSpecL  ms.mtyp_tycons;]    

    and mspecL ms =
        let header = wordL "module" $$ wordL  (name_of_modul ms) $$ wordL ":" in
        let footer = wordL "end" in
        let body = mtypeL (mtyp_of_modul ms) in
        (header @@-- body) @@ footer

    and ccuL     ccu = mspecL (top_modul_of_ccu ccu)

    and structValsL svs =
        aboveListL [namemapL (deref_val >> vspecTyL) svs.modVals;
                    namemapL structValsL             svs.modMods]

    and dtreeL x = 
        match x with 
        | TDBind (bind,body)            -> let bind = wordL "let" $$ bindL bind $$ wordL "in" in (bind @@ dtreeL body) 
        | TDSuccess (args,n)            -> wordL "Success" $$ leftL "T" $$ intL n $$ tupleL (map exprL args)
        | TDSwitch (test,dcases,dflt,r) -> (wordL "Switch" --- exprL test) @@--
                                            (aboveListL (map dcaseL dcases) @@
                                             match dflt with
                                               None       -> emptyL
                                             | Some dtree -> wordL "dflt:" --- dtreeL dtree)

    and dcaseL (TCase (test,dtree)) = (dtestL test $$ wordL "//") --- dtreeL dtree

    and dtestL x = 
        match x with 
        |  (TTest_unionconstr (c,tinst)) -> wordL "is" $$ ucrefL c $$ instL typeL tinst
        |  (TTest_array_length (n,ty)) -> wordL "length" $$ intL n $$ typeL ty
        |  (TTest_const       c        ) -> wordL "is" $$ NicePrint.tconstL c
        |  (TTest_isnull               ) -> wordL "isnull"
        |  (TTest_isinst (_,typ)           ) -> wordL "isinst" $$ typeL typ
        |  (TTest_query (exp,_,_,idx,_)) -> wordL "query" $$ exprL exp
            
    and targetL i (TTarget (argvs,body)) = leftL "T" $$ intL i $$ tupleL (map vspecL argvs) $$ rightL ":" --- exprL body

    and tmethodL (TMethod(TSlotSig(nm,_,_,_,_,_),tps,vs,e,m)) =
        (wordL "TMethod" --- (wordL nm) $$ wordL "=") --
          (wordL "METH-LAM" --- angleBracketListL (map typarL       tps) $$ rightL ".") ---
          (wordL "meth-lam" --- spaceListL (map vspecAtBindL vs)  $$ rightL ".") ---
          (atomL e) 
    and overrideL tmeth     = wordL "with" $$ tmethodL tmeth 
    and iimplL (typ,tmeths) = wordL "impl" $$ aboveListL (typeL typ :: map tmethodL tmeths) 

    let showType x = Layout.showL (typeL x)
    let showExpr x = Layout.showL (exprL x)

end


let vrefL    x = DebugPrint.vrefL x
let ucrefL   x = DebugPrint.ucrefL x
let intL     x = DebugPrint.intL x
let vspecL   x = DebugPrint.vspecL x
let typarL   x = DebugPrint.typarL x
let typarDeclL   x = DebugPrint.typarDeclL x
let typarsL   x = DebugPrint.typarDeclsL x
let typeL    x = DebugPrint.typeL x
let tslotsigL x = DebugPrint.tslotsigL x
let mtypeL   x = DebugPrint.mtypeL x
let mspecL   x = DebugPrint.mspecL x
let vspecTyL x = DebugPrint.vspecTyL x
let vsprL    x = DebugPrint.vsprL x
let bindL    x = DebugPrint. bindL x
let exprL    x = DebugPrint.exprL x
let dtreeL    x = DebugPrint.dtreeL x
let tyconL    x = DebugPrint.tyconL x
let mimplsL  x = DebugPrint.mimplsL x
let vspecAtBindL x = DebugPrint.vspecAtBindL x
let rfrefL x = DebugPrint.rfrefL x
let traitL x = DebugPrint.auxTraitL SimplifyTypes.typeSimplificationInfo0 x

(*--------------------------------------------------------------------------
!* renamings
 *------------------------------------------------------------------------ *)

type vspec_remap = val_ref vspec_map
type expr_remap = { vspec_remap: vspec_remap;
                    tyenv: tyenv }

let empty_vref_remap : vspec_remap = vspec_map_empty()

let empty_expr_remap = { vspec_remap =empty_vref_remap; tyenv = empty_tyenv }

let tmenv_add_tcref_remap tcref1 tcref2 tmenv = 
    let tyenv = tmenv.tyenv in 
    let tyenv = {tyenv with tcref_remap=tcref_map_add tcref1 tcref2 tyenv.tcref_remap} in 
    { tmenv with tyenv = tyenv }
let tmenv_copy_remap_and_bind_typars tmenv tps = 
    let tps',tyenvinner = copy_remap_and_bind_typars tmenv.tyenv tps in
    let tmenvinner = { tmenv with tyenv = tyenvinner } in 
    tps',tmenvinner

let remap_vref tmenv vref = match vspec_map_tryfind (deref_val vref) tmenv.vspec_remap with None -> vref | Some res -> res

(*--------------------------------------------------------------------------
!* 
 *------------------------------------------------------------------------ *)

let mtyp_of_mexpr (TMTyped(mtyp,_,_)) = mtyp

let wrap_modul_as_mtyp_in_namespace x = new_mtype Namespace [ x ] []

let wrap_mtyp_as_mspec id cpath mtyp = 
    new_mspec (Some cpath)  taccessPublic  id  emptyXMLDoc  [] (notlazy mtyp)
let wrap_modul_in_namespace id mspec = 
    wrap_mtyp_as_mspec id (parent_cpath (cpath_of_modul mspec))  (wrap_modul_as_mtyp_in_namespace mspec)
let wrap_mbind_in_namespace (id :ident) (TMBind(tycon,defs))  = 
    let cpath = cpath_of_modul tycon in 
    let tyconP = new_tycon (Some (parent_cpath cpath)) (id.idText,id.idRange) taccessPublic taccessPublic [] emptyXMLDoc false  (notlazy (empty_mtype Namespace)) in
    TMBind(tyconP, TMDefModul (TMBind(tycon,defs)))

let sigTypeOfImplFile (TImplFile(_,mexpr)) = mtyp_of_mexpr mexpr 

(*--------------------------------------------------------------------------
!* Data structures representing what gets hidden and what gets remapped (i.e. renamed or alpha-converted)
 * when a module signature is applied to a module.
 *------------------------------------------------------------------------ *)

type module_repackage_information = 
    { mrpiVals: (val_ref * val_ref) list;
      mrpiTycons: (tycon_ref * tycon_ref) list  }
    
type module_hiding_information = 
    { mhiTycons     : tycon_spec Zset.t; 
      mhiTyconReprs : tycon_spec Zset.t;  
      mhiVals       : val_spec Zset.t; 
      mhiRecdFields : recdfield_ref Zset.t; 
      mhiUnionConstrs : unionconstr_ref Zset.t }

let union_mhi x y = 
    { mhiTycons  = Zset.union x.mhiTycons y.mhiTycons;
      mhiTyconReprs = Zset.union x.mhiTyconReprs y.mhiTyconReprs;
      mhiVals       = Zset.union x.mhiVals y.mhiVals;
      mhiRecdFields = Zset.union x.mhiRecdFields y.mhiRecdFields;
      mhiUnionConstrs = Zset.union x.mhiUnionConstrs y.mhiUnionConstrs; }

let empty_mhi = 
    { mhiTycons = (Zset.empty tycon_spec_order); 
      mhiTyconReprs= (Zset.empty tycon_spec_order);  
      mhiVals = (Zset.empty val_spec_order); 
      mhiRecdFields= (Zset.empty rfref_order); 
      mhiUnionConstrs= (Zset.empty ucref_order) }

let empty_mrpi = { mrpiVals = []; mrpiTycons= [] } 

let mk_repackage_remapping mrpi = 
    { vspec_remap = vspec_map_of_list (map (map1'2 deref_val) mrpi.mrpiVals );
      tyenv = {tpinst = empty_tpinst; tcref_remap = tcref_map_of_list mrpi.mrpiTycons} }

(*--------------------------------------------------------------------------
!* Compute instances of the above for mty -> mty
 *------------------------------------------------------------------------ *)

let acc_tycon_remap msigty tycon (mrpi,mhi) =
    let sigtyconOpt = (Namemap.tryfind (name_of_tycon tycon) msigty.mtyp_tycons) in 
    match sigtyconOpt with 
    | None -> 
        (* The type constructor is not present in the signature. Hence it is hidden. *)
        let mhi = { mhi with mhiTycons = Zset.add tycon mhi.mhiTycons } in 
        (mrpi,mhi) 
    | Some sigtycon  -> 
        (* The type constructor is in the signature. Hence record the repackage entry *)
        let sigtcref = mk_local_tcref sigtycon in 
        let tcref = mk_local_tcref tycon in 
        let mrpi = { mrpi with mrpiTycons = ((tcref, sigtcref) :: mrpi.mrpiTycons) } in            
        (* OK, now look for hidden things *)
        let mhi = 
            if isSome (repr_of_tycon tycon) && isNone (repr_of_tycon sigtycon) then 
                (* The type representation is absent in the signature, hence it is hidden *)
                { mhi with mhiTyconReprs = Zset.add tycon mhi.mhiTyconReprs } 
            else 
                (* The type representation is present in the signature. *)
                (* Find the fields that have been hidden or which were non-public anyway. *)
                mhi 
                |> List.fold_right  (fun rfield mhi ->
                            match any_rfield_of_tycon_by_name sigtycon (name_of_rfield rfield) with 
                            | Some _  -> 
                                (* The field is in the signature. Hence it is not hidden. *)
                                mhi
                            | _ -> 
                                (* The field is not in the signature. Hence it is regarded as hidden. *)
                                let rfref = rfref_of_rfield tcref rfield in 
                                { mhi with mhiRecdFields =  Zset.add rfref mhi.mhiRecdFields })
                        (all_rfields_of_tycon tycon)  
                |> List.fold_right  (fun uconstr mhi ->
                            match uconstr_of_tycon_by_name sigtycon (name_of_uconstr uconstr) with 
                            | Some _  -> 
                                (* The constructor is in the signature. Hence it is not hidden. *)
                                mhi
                            | _ -> 
                                (* The constructor is not in the signature. Hence it is regarded as hidden. *)
                                let ucref = ucref_of_uconstr tcref uconstr in 
                                { mhi with mhiUnionConstrs =  Zset.add ucref mhi.mhiUnionConstrs })
                        (uconstrs_of_tycon tycon)   in 
        (mrpi,mhi) 

let acc_modul_tycon_remap msigty tycon (mrpi,mhi) =
    let sigtyconOpt = (Namemap.tryfind (name_of_tycon tycon) msigty.mtyp_submoduls) in 
    match sigtyconOpt with 
    | None -> 
        (* The type constructor is not present in the signature. Hence it is hidden. *)
        let mhi = { mhi with mhiTycons = Zset.add tycon mhi.mhiTycons } in 
        (mrpi,mhi) 
    | Some sigtycon  -> 
        (* The type constructor is in the signature. Hence record the repackage entry *)
        let sigtcref = mk_local_tcref sigtycon in 
        let tcref = mk_local_tcref tycon in 
        let mrpi = { mrpi with mrpiTycons = ((tcref, sigtcref) :: mrpi.mrpiTycons) } in            
        (mrpi,mhi) 

let acc_val_remap msigty vspec (mrpi,mhi) =
    let sigValOpt = (Namemap.tryfind (name_of_val vspec) msigty.mtyp_vals) in 
    let vref = mk_local_vref vspec in 
    match sigValOpt with 
    | None -> 
        if verbose then dprintf1 "acc_val_remap! v = %s\n" (name_of_val vspec); (* showL(vspecL vspec)); *)
        let mhi = { mhi with mhiVals = Zset.add vspec mhi.mhiVals } in 
        (mrpi,mhi) 
    | Some sigvref  -> 
        (* The value is in the signature. Add the repackage entry. *)
        if !DebugPrint.layout_stamps then dprintf3 "acc_val_remap, value %s: #%d --> #%d\n" (name_of_val vspec) (stamp_of_val vspec) (stamp_of_val sigvref); 
      
        let mrpi = { mrpi with mrpiVals = (vref,mk_local_vref sigvref) :: mrpi.mrpiVals } in 
        (mrpi,mhi) 

let get_submodsigty nm msigty = match Namemap.tryfind nm msigty.mtyp_submoduls with None -> empty_mtype AsNamedType | Some sigsubmodul -> mtyp_of_modul sigsubmodul 

let rec acc_mty_remap mty msigty acc = 
    let acc = Namemap.fold (fun nm submodul acc -> acc_mty_remap (mtyp_of_modul submodul) (get_submodsigty nm msigty) acc) mty.mtyp_submoduls acc in
    let acc = Namemap.fold_range (acc_tycon_remap msigty) mty.mtyp_tycons acc in 
    let acc = Namemap.fold_range (acc_val_remap msigty) mty.mtyp_vals acc in 
    acc 

let mk_mtyp_to_mtyp_remapping mty msigty = 
(*     dprintf2 "mk_mtyp_to_mtyp_remapping,\nmty = %s\nmmsigty=%s\n" (showL(mtypeL mty)) (showL(mtypeL msigty)); *)
    acc_mty_remap mty msigty (empty_mrpi, empty_mhi) 

(*--------------------------------------------------------------------------
!* Compute instances of the above for mexpr -> mty
 *------------------------------------------------------------------------ *)

(* At TMDefRec nodes abstract (virtual) vslots are effectively binders, even though they are tucked away inside the tycon. *)
let vslot_vals_of_tycons tycons =  
  tycons 
  |> map_concat (fun tycon -> if is_fsobjmodel_tycon tycon then (tycon_objmodel_data_of_tycon tycon).fsobjmodel_vslots else []) 
  |> map deref_val

let rec acc_mdef_remap msigty x acc = 
    match x with 
    | TMDefRec(tycons,binds,m) -> 
         (*  Abstract (virtual) vslots in the tycons at TMDefRec nodes are binders. They also need to be added to the remapping. *)
         let vslotvs = vslot_vals_of_tycons tycons in
         fold_right (acc_tycon_remap msigty) tycons 
            (fold_right (acc_val_remap msigty) vslotvs 
                (fold_right (var_of_bind >> acc_val_remap msigty) binds acc))
    | TMDefLet(bind,m)  -> acc_val_remap msigty (var_of_bind bind) acc
    | TMDefs(defs) -> acc_mdefs_remap msigty defs acc
    | TMAbstract(mexpr) -> acc_mty_remap (mtyp_of_mexpr mexpr) msigty acc
    | TMDefModul(TMBind(tycon, def)) -> 
        let id = id_of_modul tycon in 
        acc_modul_tycon_remap msigty tycon (acc_mdef_remap (get_submodsigty id.idText msigty) def acc)
and acc_mdefs_remap msigty mdefs acc = fold_right (acc_mdef_remap msigty) mdefs acc

let mk_mdef_to_mtyp_remapping mdef msigty =  
    if verbose then dprintf2 "mk_mdef_to_mtyp_remapping,\nmdefs = %s\nmsigty=%s\n" (showL(DebugPrint.mdefL mdef)) (showL(mtypeL msigty));
    acc_mdef_remap msigty mdef (empty_mrpi, empty_mhi) 

(*--------------------------------------------------------------------------
!* Compute instances of the above for the assembly boundary
 *------------------------------------------------------------------------ *)

let acc_tycon_assembly_boundary_mhi tycon mhi =
    if access_of_tycon tycon <> taccessPublic then 
        (* The type constructor is not public, hence hidden at the assembly boundary. *)
        { mhi with mhiTycons = Zset.add tycon mhi.mhiTycons } 
    else if repr_access_of_tycon tycon <> taccessPublic then 
        { mhi with mhiTyconReprs = Zset.add tycon mhi.mhiTyconReprs } 
    else 
        mhi 
        |> List.fold_right  
               (fun rfield mhi ->
                   if access_of_rfield rfield <> taccessPublic then 
                       let tcref = mk_local_tcref tycon in 
                       let rfref = rfref_of_rfield tcref rfield in 
                       { mhi with mhiRecdFields = Zset.add rfref mhi.mhiRecdFields } 
                   else mhi)
               (all_rfields_of_tycon tycon)  
        |> List.fold_right  
               (fun uconstr mhi ->
                   if access_of_uconstr uconstr <> taccessPublic then 
                       let tcref = mk_local_tcref tycon in 
                       let uconstr = ucref_of_uconstr tcref uconstr in 
                       { mhi with mhiUnionConstrs = Zset.add uconstr mhi.mhiUnionConstrs } 
                   else mhi)
               (uconstrs_of_tycon tycon)   

let acc_val_assembly_boundary_mhi vspec mhi =
    if access_of_val vspec <> taccessPublic then 
        (* The value is not public, hence hidden at the assembly boundary. *)
        { mhi with mhiVals = Zset.add vspec mhi.mhiVals } 
    else 
        mhi 

let rec acc_mty_assembly_boundary_mhi mty acc = 
    let acc = Namemap.fold_range (fun submodul acc -> acc_mty_assembly_boundary_mhi (mtyp_of_modul submodul) acc) mty.mtyp_submoduls acc in
    let acc = Namemap.fold_range acc_tycon_assembly_boundary_mhi mty.mtyp_tycons acc in 
    let acc = Namemap.fold_range acc_val_assembly_boundary_mhi mty.mtyp_vals acc in 
    acc 

let mk_assembly_boundary_mhi mty = 
(*     dprintf2 "mk_mtyp_to_mtyp_remapping,\nmty = %s\nmmsigty=%s\n" (showL(mtypeL mty)) (showL(mtypeL msigty)); *)
    acc_mty_assembly_boundary_mhi mty empty_mhi

(*--------------------------------------------------------------------------
!* Generic operations on module types
 *------------------------------------------------------------------------ *)

let fold_vals_and_tycons_of_mtyp ft fv = 
    let rec go mty acc = 
        let acc = Namemap.fold_range ft mty.mtyp_submoduls acc in
        let acc = Namemap.fold_range (mtyp_of_modul >> go) mty.mtyp_submoduls acc in
        let acc = Namemap.fold_range ft mty.mtyp_tycons acc in 
        let acc = Namemap.fold_range fv mty.mtyp_vals acc in 
        acc in 
    go 

let all_vals_of_mtyp m = fold_vals_and_tycons_of_mtyp (fun ft acc -> acc) (fun v acc -> v :: acc) m []
let all_tycons_of_mtyp m = fold_vals_and_tycons_of_mtyp (fun ft acc -> ft :: acc) (fun v acc -> acc) m []

(*---------------------------------------------------------------------------
!* Free variables in terms.  All binders are distinct.
 *------------------------------------------------------------------------- *)

let empty_freevars =  
  { uses_method_local_constructs=false;
    free_loctycon_reprs=empty_free_loctycons; 
    free_locvals=empty_free_locvals; 
    free_tyvars=empty_free_tyvars;
    free_rfields = empty_free_rfields;
    free_uconstrs = empty_free_uconstrs}

let union_freevars fvs1 fvs2 = 
  if fvs1 == empty_freevars then fvs2 else 
  if fvs2 == empty_freevars then fvs1 else
  { free_locvals          = union_free_locvals fvs1.free_locvals fvs2.free_locvals;
    free_tyvars           = union_free_tyvars fvs1.free_tyvars fvs2.free_tyvars;
    uses_method_local_constructs= fvs1.uses_method_local_constructs || fvs2.uses_method_local_constructs;
    free_loctycon_reprs   = union_free_loctycons fvs1.free_loctycon_reprs fvs2.free_loctycon_reprs; 
    free_rfields          = union_free_rfields fvs1.free_rfields fvs2.free_rfields; 
    free_uconstrs          = union_free_uconstrs fvs1.free_uconstrs fvs2.free_uconstrs; }

let tyvars f v fvs = 
  let ftyvs = fvs.free_tyvars in 
  let ftyvs' = f v ftyvs in
  if ftyvs == ftyvs' then fvs else { fvs with free_tyvars = ftyvs' } 

let acc_freevars_in_type ty fvs = tyvars acc_free_in_type ty fvs 
let acc_freevars_in_types tys fvs = tyvars acc_free_in_types tys fvs 
let bound_locval v fvs =
  let fvs = tyvars acc_free_in_val v fvs in
  if not (Zset.mem v fvs.free_locvals) then fvs
  else {fvs with free_locvals= Zset.remove v fvs.free_locvals} 

let bound_protect fvs =
  if fvs.uses_method_local_constructs then {fvs with uses_method_local_constructs = false} else fvs
let acc_uses_function_local_constructs flg fvs = 
  if flg && not fvs.uses_method_local_constructs then {fvs with uses_method_local_constructs = true} 
  else fvs 

let bound_locvals vs fvs = fold_right bound_locval vs fvs

let bind_lhs bind fvs = bound_locval (var_of_bind bind) fvs

let removeL l s = fold_right Zset.remove l s

let rec acc_rhs (TBind(_,repr)) acc = acc_free_in_expr repr acc
          
and acc_free_in_switch_cases csl dflt (acc:freevars) =
  Option.fold_right acc_free_in_dtree dflt (fold_right acc_free_in_switch_case csl acc)
and acc_free_in_switch_case (TCase(discrim,dtree)) acc = 
  acc_free_in_dtree dtree (acc_free_in_discrim discrim acc)
and acc_free_in_discrim discrim acc = 
  match discrim with 
  | TTest_unionconstr(ucref,tinst) -> acc_free_ucref ucref (acc_freevars_in_types tinst acc)
  | TTest_array_length(_,ty) -> acc_freevars_in_type ty acc
  | TTest_const _
  | TTest_isnull -> acc
  | TTest_isinst (srcty,tgty) -> acc_freevars_in_type srcty (acc_freevars_in_type tgty acc)
  | TTest_query (exp, tys, vref, idx, apinfo) -> acc_free_in_expr exp (acc_freevars_in_types tys (Option.fold_right acc_free_vref vref acc))
and acc_free_in_dtree x (acc : freevars) =
  match x with 
  | TDSwitch(e1,csl,dflt,_) -> acc_free_in_expr e1 (acc_free_in_switch_cases csl dflt acc)
  | TDSuccess (es,_) -> acc_free_in_exprs es acc
  | TDBind (bind,body) -> union_freevars (bind_lhs bind (acc_rhs bind (free_in_dtree body))) acc
  
and acc_free_locval v fvs =
  let fvs = tyvars acc_free_in_val (deref_local_val v) fvs in 
  if Zset.mem v fvs.free_locvals then fvs 
  else {fvs with free_locvals=Zset.add v fvs.free_locvals}

and acc_loctycon_repr b fvs = 
  if Zset.mem b fvs.free_loctycon_reprs  then fvs
  else { fvs with free_loctycon_reprs = Zset.add b fvs.free_loctycon_reprs } 
and acc_used_tycon_repr tc fvs = 
  if isSome (repr_of_tycon tc)
  then acc_loctycon_repr tc fvs
  else fvs
and acc_free_ucref cr fvs = 
  let fvs = fvs |> acc_used_tycon_repr (deref_tycon (tcref_of_ucref cr)) in
  let fvs = fvs |> tyvars acc_free_tycon (tcref_of_ucref cr) in
  if Zset.mem cr fvs.free_uconstrs then fvs 
  else { fvs with free_uconstrs = Zset.add cr fvs.free_uconstrs } 
  
and acc_free_rfref fr fvs = 
  let fvs = fvs |> acc_used_tycon_repr (deref_tycon (tcref_of_rfref fr)) in 
  let fvs = fvs |> tyvars acc_free_tycon (tcref_of_rfref fr)  in 
  if Zset.mem fr fvs.free_rfields then fvs 
  else { fvs with free_rfields = Zset.add fr fvs.free_rfields } 
  
and acc_free_ecref exnc fvs = fvs
and acc_free_vref vref fvs = 
  match vref with 
  | Ref_private v -> acc_free_locval v fvs
  (* non-local values do not contain free variables *)
  | _ -> fvs

and acc_free_in_method (TMethod(slotsig,tps,tmvs,e,m)) acc =
  acc_free_in_slotsig slotsig
   (union_freevars (tyvars bound_typars tps (bound_locvals tmvs (free_in_expr e)))
    acc)

and acc_free_in_iimpl (ty,overrides) acc = 
  acc_freevars_in_type ty (fold_right acc_free_in_method overrides acc)

and acc_free_in_expr x acc =
  match x with
  (* BINDING CONSTRUCTS *)
  | TExpr_lambda (_,basev,vs,b,_,rty,cache)  -> 
      union_freevars (cached cache (fun () -> Option.fold_right bound_locval basev (fold_right bound_locval vs (acc_freevars_in_type rty (free_in_expr b))))) acc
  | TExpr_tlambda (_,vs,b,_,rty, cache) ->
      union_freevars (cached cache (fun () -> tyvars bound_typars vs (acc_freevars_in_type rty (free_in_expr b)))) acc
  | TExpr_tchoose (vs,b,_) ->
      union_freevars (tyvars bound_typars vs (free_in_expr b)) acc
  | TExpr_letrec (binds,e,_,cache) ->
      union_freevars (cached cache (fun () -> fold_right bind_lhs binds (fold_right acc_rhs binds (free_in_expr e)))) acc
  | TExpr_let (bind,e,_,cache) -> 
      union_freevars (cached cache (fun () -> bind_lhs bind (acc_rhs bind (free_in_expr e)))) acc
  | TExpr_obj (_,typ,basev,basecall,overrides,iimpls,m,cache)   ->  
      union_freevars (cached cache (fun () -> 
          bound_protect
            (Option.fold_right bound_locval basev
              (acc_freevars_in_type typ
                 (acc_free_in_expr basecall
                    (fold_right acc_free_in_method overrides 
                       (fold_right acc_free_in_iimpl iimpls empty_freevars))))))) acc  | TExpr_const _ -> acc
  (* NON-BINDING CONSTRUCTS *)                      
  | TExpr_val (lvr,flags,_) -> 
      acc_uses_function_local_constructs (flags <> NormalValUse) (acc_free_vref lvr acc)
  | TExpr_quote (raw,ast,_,ty) ->  acc_freevars_in_type ty acc (* Note quoted terms are closed *)
  | TExpr_hole (_,ty) ->  acc_freevars_in_type ty acc
  | TExpr_app(f0,f0ty,tyargs,args,_) -> 
      acc_freevars_in_type f0ty
        (acc_free_in_expr f0
           (acc_freevars_in_types tyargs
              (fold_right acc_free_in_expr args acc)))
  | TExpr_link(eref) -> acc_free_in_expr !eref acc
  | TExpr_seq (e1,e2,_,_) -> acc_free_in_expr e1 (acc_free_in_expr e2 acc)

  | TExpr_static_optimization (_,e2,e3,m) -> acc_free_in_expr e2 (acc_free_in_expr e3 acc)
  | TExpr_match (_,dtree,targets,_,_,cache) -> 
      union_freevars 
        (cached cache (fun () -> acc_free_in_targets targets empty_freevars)) 
        (acc_free_in_dtree dtree acc)
  | TExpr_op (op,tinst,args,_) -> acc_free_in_op op (acc_freevars_in_types tinst (acc_free_in_exprs args acc))

and acc_free_in_op op acc =
  match op with
  | TOp_bytes _ | TOp_try_catch | TOp_try_finally | TOp_for _ | TOp_coerce | TOp_get_ref_lval | TOp_tuple | TOp_array | TOp_while | TOp_tuple_field_get _ -> acc
  | TOp_constr_tag_get tr -> acc_used_tycon_repr (deref_tycon tr) acc
  | TOp_uconstr cr | TOp_constr_field_get (cr,_) | TOp_constr_field_set (cr,_) -> acc_free_ucref cr acc
  | TOp_exnconstr ecr | TOp_exnconstr_field_get (ecr,_) | TOp_exnconstr_field_set (ecr,_)  -> acc_free_ecref ecr acc
  | TOp_field_get fr | TOp_field_get_addr fr | TOp_field_set fr -> acc_free_rfref fr acc
  | TOp_recd (kind,tcr) -> 
      let acc = acc_uses_function_local_constructs (kind = RecdExprIsObjInit) acc in
      (acc_used_tycon_repr (deref_tycon tcr) (tyvars acc_free_tycon tcr acc)) 
  | TOp_asm (_,tys) ->  acc_freevars_in_types tys acc
  | TOp_trait_call(TTrait(tys,nm,_,argtys,rty)) -> acc_freevars_in_types tys (acc_freevars_in_types argtys (acc_freevars_in_type rty acc))
  | TOp_lval_op (_,lvr) -> acc_free_vref lvr acc
  | TOp_ilcall ((virt,protect,valu,newobj,superInit,prop,isDllImport,boxthis,mref),enclTypeArgs,methTypeArgs,tys) ->
     acc_freevars_in_types enclTypeArgs 
       (acc_freevars_in_types methTypeArgs  
         (acc_freevars_in_types tys 
           (acc_uses_function_local_constructs protect acc)))

and acc_free_in_targets targets acc = 
  Array.fold_right (fun (TTarget(vs,e)) acc -> fold_right bound_locval vs (acc_free_in_expr e acc)) targets acc

and acc_free_in_exprs es acc = 
  match es with 
  | [] -> acc
  | h::t -> acc_free_in_expr h (acc_free_in_exprs t acc)
and acc_free_in_slotsig (TSlotSig(_,typ,_,_,_,_)) acc = acc_freevars_in_type typ acc

and free_in_dtree e = acc_free_in_dtree e empty_freevars
and free_in_expr e = acc_free_in_expr e empty_freevars

(* Note: these are only an approximation - they are currently used only by the optimizer  *)
let rec acc_free_in_mdef x acc = 
    match x with 
    | TMDefRec(tycons,binds,m) -> fold_right acc_rhs binds  acc
    | TMDefLet(bind,m)  ->acc_rhs bind  acc
    | TMDefs(defs) -> acc_free_in_mdefs defs acc
    | TMAbstract(TMTyped(mtyp,mdef,_)) -> acc_free_in_mdef mdef acc (* not really right, but sufficient for how this is used in optimization *)
    | TMDefModul(TMBind(_, def)) -> acc_free_in_mdef def acc
and acc_free_in_mdefs x acc = 
    fold_right acc_free_in_mdef x acc

(* NOTE: we don't yet need to ask for free variables in module expressions *)

let free_in_rhs bind = acc_rhs bind empty_freevars
let free_in_mdef mdef = acc_free_in_mdef mdef empty_freevars

(*---------------------------------------------------------------------------
!* Destruct - rarely needed
 *------------------------------------------------------------------------- *)

let rec strip_lambda (e,ty) = 
  match e with 
  | TExpr_lambda (_,basevopt,v,b,_,rty,_) -> 
      if isSome basevopt then errorR(InternalError("skipping basevopt", range_of_expr e));
      let (vs',b',rty') = strip_lambda (b,rty) in 
      (v :: vs', b', rty') 
  | _ -> ([],e,ty)

let dest_top_lambda (e,ty) =
  let tps,taue,tauty = match e with TExpr_tlambda (_,tps,b,_,rty,_) -> tps,b,rty | _ -> [],e,ty in 
  let vs,body,rty = strip_lambda (taue,tauty) in
  tps,vs,body,rty

(* This is used to infer arities of expressions *)
(* i.e. base the chosen arity on the syntactic expression shape and type of arguments *)
let infer_arity_of_expr ty partialArgAttribsL retAttribs e = 
    let rec strip_lambda_notypes e = 
        match e with 
        | TExpr_lambda (_,_,vs,b,_,_,_) -> 
            let (vs',b') = strip_lambda_notypes b in 
            (vs :: vs', b') 
        | TExpr_tchoose (tps,b,_) -> strip_lambda_notypes b 
        | _ -> ([],e) in 

    let dest_top_lambda_notypes e =
        let tps,taue = match e with TExpr_tlambda (_,tps,b,_,_,_) -> tps,b | _ -> [],e in 
        let vs,body = strip_lambda_notypes taue in
        tps,vs,body in 

    let tps,vsl,body = dest_top_lambda_notypes e in 
    let fun_arity = length vsl in 
    let dtys,rty =  strip_fun_typ_upto fun_arity (snd (try_dest_forall_typ ty)) in
    let partialArgAttribsL = Array.of_list partialArgAttribsL in 
    TopValInfo (length tps, 
                ((combine vsl dtys) |> list_mapi (fun i (vs,ty) -> 
                    let partialAttribs = if i < Array.length partialArgAttribsL then partialArgAttribsL.(i) else [] in 
                    let tys = try_dest_tuple_typ ty in 
                    let ids = 
                        if length vs = length tys then  vs |> map (fun v -> Some (id_of_val v))
                        else tys |> map (fun _ -> None) in
                    let attribs = 
                        if length partialAttribs = length tys then  partialAttribs 
                        else tys |> map (fun _ -> []) in
                    combine ids attribs |> map (fun (id,attribs) -> TopArgData(attribs,id)))), 
                TopArgData(retAttribs,None))

let infer_arity_of_expr_bind v e = 
    match (arity_of_val v) with
    | Some info -> info
    | None -> infer_arity_of_expr (type_of_val v) [] [] e

(* Use this one after we've recorded a choice in lowertop.ml *)
let chosen_arity_of_bind (TBind(v,repr)) = arity_of_val v

(*-------------------------------------------------------------------------
!* Check if constraints are satisfied that allow us to use more optimized
 * implementations
 *------------------------------------------------------------------------- *)

let underlying_typ_of_enum_typ g typ = 
   assert(is_enum_typ typ);
   let tycon = deref_tycon (tcref_of_stripped_typ typ) in 
   if is_il_enum_tycon tycon then 
       let _,_,tdef = dest_il_tycon tycon in 
       let info = info_for_enum (tdef.tdName,tdef.tdFieldDefs) in
       let il_ty = typ_of_enum_info info in 
       match tname_of_tspec (tspec_of_typ il_ty) with 
       | "System.Byte" -> g.byte_ty
       | "System.SByte" -> g.sbyte_ty
       | "System.Int16" -> g.int16_ty
       | "System.Int32" -> g.int32_ty
       | "System.Int64" -> g.int64_ty
       | "System.UInt16" -> g.uint16_ty
       | "System.UInt32" -> g.uint32_ty
       | "System.UInt64" -> g.uint64_ty
       | "System.Signle" -> g.float32_ty
       | "System.Double" -> g.float_ty
       | "System.Char" -> g.char_ty
       | "System.Boolean" -> g.bool_ty
       | _ -> g.int32_ty
   else 
       match any_rfield_of_tycon_by_name tycon "value__" with 
       | Some rf -> rf.rfield_type
       | None ->  failwith ("no 'value__' field found for enumeration type "^name_of_tycon tycon)


(* CLEANUP NOTE: this absolutely awful. Get rid of this nonsense mutation. *)
let set_val_has_no_arity f = 
  if verbose then  dprintf1 "clearing arity on %s\n" (name_of_val f); 
  (data_of_val f).val_arity <- None; f


(*--------------------------------------------------------------------------
!* Resolve static optimization constraints
 *------------------------------------------------------------------------ *)

let norm_enum_typ g ty = (if is_enum_typ ty then underlying_typ_of_enum_typ g ty else ty) 

let static_optimization_constraint_definitely_satisfied g c =
  match c with
  | TTyconEqualsTycon (a,b) -> 
   let a = norm_enum_typ g a in
   let b = norm_enum_typ g b in
   is_stripped_tyapp_typ a && is_stripped_tyapp_typ b && 
   g.tcref_eq (tcref_of_stripped_typ a) (tcref_of_stripped_typ b)
    
let static_optimization_constraint_definitely_false g c =
  match c with
  | TTyconEqualsTycon (a,b) -> 
   let a = norm_enum_typ g a in
   let b = norm_enum_typ g b in
   is_stripped_tyapp_typ a && is_stripped_tyapp_typ b && 
   not (g.tcref_eq (tcref_of_stripped_typ a) (tcref_of_stripped_typ b))

let mk_static_optimization_expr g (cs,e1,e2,m) = 
  if for_all (static_optimization_constraint_definitely_satisfied g) cs then e1
  else if exists (static_optimization_constraint_definitely_false g) cs then e2
  else TExpr_static_optimization(cs,e1,e2,m)

(*--------------------------------------------------------------------------
!* Copy expressions, including new names for locally bound values.
 * Used to inline expressions.
 *------------------------------------------------------------------------ *)


let mark_as_compgen compgen d = { d with val_flags= ValSpecFlags.encode_compgen_of_vflags (ValSpecFlags.compgen_of_vflags d  || compgen) d.val_flags }
let bind_locval v v' tmenv = 
    if !DebugPrint.layout_stamps then dprintf3 "adding copy remapping for val %s, #%d --> #%d\n" (name_of_val v) (stamp_of_val v) (stamp_of_val v');
    { tmenv with vspec_remap=vspec_map_add v (mk_local_vref v') tmenv.vspec_remap}
let bind_locvals vs vs' tmenv = fold_right2 bind_locval vs vs' tmenv
let bind_tycon tc tc' tyenv = 
    if !DebugPrint.layout_stamps then dprintf3 "adding copy remapping for tycon %s, #%d --> #%d\n" (name_of_tycon tc) (stamp_of_tycon tc) (stamp_of_tycon tc');
    { tyenv with tcref_remap=tcref_map_add (mk_local_ref tc) (mk_local_tcref tc') tyenv.tcref_remap }
let bind_tycons tcs tcs' tyenv = fold_right2 bind_tycon tcs tcs' tyenv

let remap_attrib_kind  tmenv =  function 
  | ILAttrib _ as x -> x
  | FSAttrib vref -> 
      if !DebugPrint.layout_stamps then dprintf1 "remap attrib vref #%d\n" (stamp_of_vref vref);
      FSAttrib(remap_vref tmenv vref)

let rec remap_attrib g tmenv (Attrib (kind, args, props)) = 
    Attrib(remap_attrib_kind tmenv kind, 
           remap_exprs g false tmenv args, 
           props |> map (fun (nm,ty,flg,expr) -> (nm,remap_type tmenv.tyenv ty, flg, remap_expr g false tmenv expr)))

and remap_attribs g tmenv xs =  List.map (remap_attrib g tmenv) xs

and remap_arg_data g tmenv (TopArgData(attribs,nm)) =
    TopArgData(remap_attribs g tmenv attribs,nm)

and remap_top_val_info g tmenv (TopValInfo(n,arginfosl,retInfo)) =
    TopValInfo(n,List.map (List.map (remap_arg_data g tmenv)) arginfosl, remap_arg_data g tmenv retInfo)

and remap_val_data g tmenv d =
    if !DebugPrint.layout_stamps then dprintf1 "remap val data #%d\n" d.val_stamp;
    let ty = d.val_type in 
    let arity = d.val_arity in 
    let ty' = ty |> remap_type tmenv.tyenv in 
    { d with 
        val_type    = ty';
        val_actual_parent = d.val_actual_parent |> remap_parent_ref tmenv.tyenv;
        val_arity         = d.val_arity         |> Option.map (remap_top_val_info g tmenv);
        val_meminfo       = d.val_meminfo       |> Option.map (remap_vspr arity ty ty' tmenv);
        val_attribs       = d.val_attribs       |> remap_attribs g tmenv }

and remap_parent_ref tyenv p =
    match p with 
    | ParentNone -> ParentNone
    | Parent x -> Parent (x |> remap_tcref tyenv.tcref_remap)

and map_immediate_vals_and_tycons_of_modtyp ft fv x = 
    { x with 
          mtyp_submoduls = x.mtyp_submoduls |> Namemap.map ft;
          mtyp_vals      = x.mtyp_vals      |> Namemap.map fv;
          mtyp_tycons    = x.mtyp_tycons    |> Namemap.map ft;} 

and copy_and_remap_val g compgen tmenv v = 
    v |> new_vspec_modified (remap_val_data g tmenv >> mark_as_compgen compgen)

and fixup_val_attribs g tmenv v1 v2 =
    (data_of_val v2).val_attribs <- (attribs_of_val v1) |> remap_attribs g tmenv
    
and copy_and_remap_and_bind_vals g compgen tmenv vs = 
    let vs' = map (copy_and_remap_val g compgen tmenv) vs  in 
    let tmenvinner = bind_locvals vs vs' tmenv in
    (* Fixup attributes now we've built the full map of value renamings (attributes contain value references) *)
    iter2 (fixup_val_attribs g tmenvinner) vs vs';    
    vs', tmenvinner

and copy_and_remap_and_bind_val g compgen tmenv v = 
    let v' = v |> copy_and_remap_val g compgen tmenv in 
    let tmenvinner = bind_locval v v' tmenv in
    (* Fixup attributes now we've built the full map of value renamings (attributes contain value references) *)
    fixup_val_attribs g tmenvinner v v';    
    v', tmenvinner
    
and remap_expr g compgen (tmenv:expr_remap) x =
    match x with
    (* Binding constructs - see also dtrees below *)
    | TExpr_lambda (_,basevopt,vs,b,m,rty,_)  -> 
        let basevopt, tmenv =  map_acc_option (copy_and_remap_and_bind_val g compgen) tmenv basevopt in 
        let vs,tmenv = copy_and_remap_and_bind_vals g compgen tmenv vs in 
        let b = remap_expr g compgen tmenv b in 
        let rty = remap_typeA tmenv.tyenv rty in 
        TExpr_lambda (new_uniq(), basevopt,vs,b,m, rty, new_cache ())
    | TExpr_tlambda (_,tps,b,m,rty,_) ->
        let tps',tmenvinner = tmenv_copy_remap_and_bind_typars tmenv tps in
        mk_tlambda m tps' (remap_expr g compgen tmenvinner b,remap_typeA tmenvinner.tyenv rty)
    | TExpr_tchoose (tps,b,m) ->
        let tps',tmenvinner = tmenv_copy_remap_and_bind_typars tmenv tps in
        TExpr_tchoose(tps',remap_expr g compgen tmenvinner b,m)
    | TExpr_letrec (binds,e,m,_) ->  
        let binds',tmenvinner = copy_and_remap_and_bind_bindings g compgen tmenv binds in 
        TExpr_letrec (binds',remap_expr g compgen tmenvinner e,m,new_cache())
    | TExpr_let _ -> remap_linear_expr g compgen tmenv x (fun x -> x)
    | TExpr_match (exprm,pt,targets,m,ty,_) ->
        prim_mk_match (exprm,remap_dtree g compgen tmenv pt,
                     targets |> Array.map (fun (TTarget(vs,e)) ->
                       let vs',tmenvinner = copy_and_remap_and_bind_vals g compgen tmenv vs in 
                       TTarget(vs', remap_expr g compgen tmenvinner e)),
                     m,remap_typeA tmenv.tyenv ty)
    (* Other constructs *)
    | TExpr_val (vr,isSuperInit,m) -> TExpr_val (remap_vref tmenv vr,isSuperInit,m)
    | TExpr_quote (raw,a,m,ty) ->  TExpr_quote (raw,remap_expr g compgen tmenv a,m,remap_typeA tmenv.tyenv ty)
    | TExpr_hole (m,ty) ->  TExpr_hole (m,remap_typeA tmenv.tyenv ty)
    | TExpr_obj (_,typ,basev,basecall,overrides,iimpls,m,_) -> 
        let basev',tmenvinner = map_acc_option (copy_and_remap_and_bind_val g compgen) tmenv basev in 
        TExpr_obj (new_uniq(),remap_typeA tmenv.tyenv typ,basev',
                     remap_expr g compgen tmenv basecall,
                     List.map (remap_method g compgen tmenvinner) overrides,
                     List.map (remap_iimpl g compgen tmenvinner) iimpls,m,new_cache()) 
    | TExpr_op(op,tinst,args,m) -> TExpr_op (remap_op tmenv op,remap_types tmenv.tyenv tinst,remap_exprs g compgen tmenv args,m)
    | TExpr_app(e1,e1ty,tyargs,args,m) -> TExpr_app(remap_expr g compgen tmenv e1,remap_typeA tmenv.tyenv e1ty,remap_types tmenv.tyenv tyargs,map (remap_expr g compgen tmenv) args,m)
    | TExpr_link(eref) -> remap_expr g compgen tmenv !eref
    | TExpr_seq (e1,e2,dir,m)  -> TExpr_seq (remap_expr g compgen tmenv e1,remap_expr g compgen tmenv e2,dir,m)
    | TExpr_static_optimization (cs,e2,e3,m) -> 
       (* note that type instantiation typically resolve the static constraints here *)
       mk_static_optimization_expr g (map (remap_constraint tmenv.tyenv) cs,
                                      remap_expr g compgen tmenv e2,
                                      remap_expr g compgen tmenv e3,m)

    | TExpr_const (c,m,ty) -> TExpr_const (c,m,remap_typeA tmenv.tyenv ty)


and remap_linear_expr g compgen tmenv e contf =
    match e with 
    | TExpr_let (bind,e,m,_) ->  
      let bind',tmenvinner = copy_and_remap_and_bind_binding g compgen tmenv bind in
      remap_linear_expr g compgen tmenvinner e (contf << mk_let_bind m bind')
    | _ -> contf (remap_expr g compgen tmenv e) 
and remap_constraint tyenv c = 
    match c with 
    | TTyconEqualsTycon(ty1,ty2) -> TTyconEqualsTycon(remap_typeA tyenv ty1, remap_typeA tyenv ty2)

and remap_op tmenv op = 
    match op with 
    | TOp_recd (ctor,tcr)           -> TOp_recd(ctor,remap_tcref tmenv.tyenv.tcref_remap tcr)
    | TOp_constr_tag_get tcr        -> TOp_constr_tag_get(remap_tcref tmenv.tyenv.tcref_remap tcr)
    | TOp_uconstr(ucref)            -> TOp_uconstr(remap_ucref tmenv.tyenv.tcref_remap ucref)
    | TOp_exnconstr(ec)             -> TOp_exnconstr(remap_tcref tmenv.tyenv.tcref_remap ec)
    | TOp_exnconstr_field_get(ec,n) -> TOp_exnconstr_field_get(remap_tcref tmenv.tyenv.tcref_remap ec,n)
    | TOp_exnconstr_field_set(ec,n) -> TOp_exnconstr_field_set(remap_tcref tmenv.tyenv.tcref_remap ec,n)
    | TOp_field_set(rfref)          -> TOp_field_set(remap_rfref tmenv.tyenv.tcref_remap rfref)
    | TOp_field_get(rfref)          -> TOp_field_get(remap_rfref tmenv.tyenv.tcref_remap rfref)
    | TOp_field_get_addr(rfref)     -> TOp_field_get_addr(remap_rfref tmenv.tyenv.tcref_remap rfref)
    | TOp_constr_field_get(ucref,n) -> TOp_constr_field_get(remap_ucref tmenv.tyenv.tcref_remap ucref,n)
    | TOp_constr_field_set(ucref,n) -> TOp_constr_field_set(remap_ucref tmenv.tyenv.tcref_remap ucref,n)
    | TOp_asm (instrs,tys)          -> TOp_asm (instrs,remap_types tmenv.tyenv tys)
    | TOp_trait_call(traitInfo)     -> TOp_trait_call(remap_traitA tmenv.tyenv traitInfo)
    | TOp_lval_op (kind,lvr)        -> TOp_lval_op (kind,remap_vref tmenv lvr)
    | TOp_ilcall ((virt,protect,valu,newobj,superInit,prop,isDllImport,boxthis,mref),enclTypeArgs,methTypeArgs,tys) -> 
       TOp_ilcall ((virt,protect,valu,newobj,superInit,prop,isDllImport,Option.map (fun (a,b) -> (remap_typeA tmenv.tyenv a, remap_typeA tmenv.tyenv b)) boxthis,mref),
                      remap_types tmenv.tyenv enclTypeArgs,remap_types tmenv.tyenv methTypeArgs,remap_types tmenv.tyenv tys)
    | _ ->  op
    

and remap_exprs g compgen tmenv es = list_map (remap_expr g compgen tmenv) es

and remap_dtree g compgen tmenv x =
    match x with 
    | TDSwitch(e1,csl,dflt,m) -> 
        TDSwitch(remap_expr g compgen tmenv e1,
                map (fun (TCase(test,y)) -> 
                  let test' = 
                    match test with 
                    | TTest_unionconstr (uc,tinst)   -> TTest_unionconstr(remap_ucref tmenv.tyenv.tcref_remap uc,remap_types tmenv.tyenv tinst)
                    | TTest_array_length (n,ty)      -> TTest_array_length(n,remap_typeA tmenv.tyenv ty)
                    | TTest_const c                  -> test
                    | TTest_isinst (srcty,tgty)      -> TTest_isinst (remap_typeA tmenv.tyenv srcty,remap_typeA tmenv.tyenv tgty) 
                    | TTest_isnull                   -> TTest_isnull 
                    | TTest_query _ -> failwith "TTest_query should only be used during pattern match compilation" in
                  TCase(test',remap_dtree g compgen tmenv y)) csl, 
                Option.map (remap_dtree g compgen tmenv) dflt,
                m)
    | TDSuccess (es,n) -> 
        TDSuccess (remap_exprs g compgen tmenv es,n)
    | TDBind (bind,rest) -> 
        let bind',tmenvinner = copy_and_remap_and_bind_binding g compgen tmenv bind in
        TDBind (bind',remap_dtree g compgen tmenvinner rest)
        
and copy_and_remap_and_bind_binding g compgen tmenv bind =
    let v = var_of_bind bind in 
    let v', tmenv = copy_and_remap_and_bind_val g compgen tmenv v in
    remap_and_rename_bind g compgen tmenv bind v' , tmenv

and copy_and_remap_and_bind_bindings g compgen tmenv binds = 
    let vs', tmenvinner = copy_and_remap_and_bind_vals g compgen tmenv (map var_of_bind binds) in 
    remap_and_rename_binds g compgen tmenvinner binds vs',tmenvinner

and remap_and_rename_binds g compgen tmenvinner binds vs' = map2 (remap_and_rename_bind g compgen tmenvinner) binds vs'
and remap_and_rename_bind g compgen tmenvinner (TBind(_,repr)) v' = TBind(v', remap_expr g compgen tmenvinner repr)

and remap_method g compgen tmenv (TMethod(slotsig,tps,vs,e,m))  =
    let slotsig' = remap_slotsig tmenv.tyenv slotsig in
    let tps',tmenvinner = tmenv_copy_remap_and_bind_typars tmenv tps in
    let vs', tmenvinner2 = copy_and_remap_and_bind_vals g compgen tmenvinner vs in 
    let e' = remap_expr g compgen tmenvinner2 e in
    TMethod(slotsig',tps',vs',e',m)

and remap_iimpl g compgen tmenv (ty,overrides)  =
    (remap_typeA tmenv.tyenv ty, List.map (remap_method g compgen tmenv) overrides)

and remap_rfield g tmenv x = 
    { x with 
          rfield_type     = x.rfield_type     |> remap_type tmenv.tyenv;
          rfield_pattribs = x.rfield_pattribs |> remap_attribs g tmenv;
          rfield_fattribs = x.rfield_fattribs |> remap_attribs g tmenv; } 
and remap_rfields g tmenv x = x |> all_rfields_of_rfield_tables |> map (remap_rfield g tmenv) |> mk_rfields_table 

and remap_uconstr g tmenv x = 
    { x with 
          uconstr_rfields = x.uconstr_rfields |> remap_rfields g tmenv;
          uconstr_rty     = x.uconstr_rty     |> remap_type tmenv.tyenv;
          uconstr_attribs = x.uconstr_attribs |> remap_attribs g tmenv; } 
and remap_funion g tmenv x = x |> uconstrs_of_funion |> map (remap_uconstr g tmenv)|> mk_funion 

and remap_fsobjmodel g tmenv x = 
    { x with 
          tycon_objmodel_kind = 
             (match x.tycon_objmodel_kind with 
              | TTyconDelegate tslotsig -> TTyconDelegate (remap_slotsig tmenv.tyenv tslotsig)
              | TTyconClass | TTyconInterface | TTyconStruct | TTyconEnum -> x.tycon_objmodel_kind);
          fsobjmodel_vslots  = x.fsobjmodel_vslots  |> map (remap_vref tmenv);
          fsobjmodel_rfields = x.fsobjmodel_rfields |> remap_rfields g tmenv } 


and remap_tycon_repr g tmenv repr = 
    match repr with 
    | TFsObjModelRepr    x -> TFsObjModelRepr (remap_fsobjmodel g tmenv x)
    | TRecdRepr          x -> TRecdRepr (remap_rfields g tmenv x)
    | TFiniteUnionRepr   x -> TFiniteUnionRepr (remap_funion g tmenv x)
    | TIlObjModelRepr    _ -> failwith "cannot remap IL type definitions"
    | TAsmRepr           x -> repr 

and remap_tcaug tmenv x = 
    { x with 
          tcaug_equals        = x.tcaug_equals         |> Option.map (pair_map (remap_vref tmenv) (remap_vref tmenv));
          tcaug_compare        = x.tcaug_compare         |> Option.map (remap_vref tmenv);
          tcaug_structural_hash= x.tcaug_structural_hash |> Option.map (remap_vref tmenv); 
          tcaug_adhoc          = x.tcaug_adhoc           |> Namemap.map (map (remap_vref tmenv));
          tcaug_super   = x.tcaug_super   |> Option.map (remap_type tmenv.tyenv);
          tcaug_implements     = x.tcaug_implements      |> map (map1'3 (remap_type tmenv.tyenv)) } 

and remap_tycon_exnc_info g tmenv inp =
    match inp with 
    | TExnAbbrevRepr x -> TExnAbbrevRepr (remap_tcref tmenv.tyenv.tcref_remap x)
    | TExnFresh      x -> TExnFresh (remap_rfields g tmenv x)
    | TExnAsmRepr  _ | TExnNone -> inp 

and remap_vspr arity ty ty' tmenv x = 
   (* The slotsig in the vspr_implements_slotsig is w.r.t. the type variables in the value's type. *)
   (* REVIEW: this is a bit gross. It would be nice if the slotsig was standalone *)
   assert (isSome arity);
   let tpsorig,_,_,_ = dest_vspr_typ x (the(arity)) ty in 
   let tps,_,_,_ = dest_vspr_typ x (the(arity)) ty' in 
   let renaming,_ = mk_typar_to_typar_renaming tpsorig tps  in 
   let tmenv = { tmenv with tyenv = { tmenv.tyenv with tpinst = tmenv.tyenv.tpinst @ renaming } } in
    { x with 
        vspr_apparent_parent    = x.vspr_apparent_parent    |>  remap_tcref tmenv.tyenv.tcref_remap ;
        vspr_implements_slotsig = x.vspr_implements_slotsig |> Option.map (remap_slotsig tmenv.tyenv); 
    } 

and copy_remap_and_bind_mtyp g compgen tmenv mty = 
    let tycons = all_tycons_of_mtyp mty in 
    let vs = all_vals_of_mtyp mty in 
    let _,_,tmenvinner = copy_and_remap_and_bind_tycons_and_vals g compgen tmenv tycons vs [] in
    let mty' = map_immediate_vals_and_tycons_of_modtyp (rename_tycon tmenvinner.tyenv) (rename_val tmenvinner) mty in
    mty', tmenvinner

and rename_tycon tyenv x = 
    try tcref_map_find (mk_local_ref x) tyenv.tcref_remap |> deref_tycon  
    with Not_found -> error(Error("couldn't remap internal tycon "^showL(tyconL x),range_of_tycon x)); x 

and rename_val tmenv x = 
    match vspec_map_tryfind x tmenv.vspec_remap with 
    | Some v -> deref_val v
    | None -> x
    (* try with Not_found -> error(Error("couldn't remap internal value",range_of_val x)); x  *)

(* Note this operates over a whole nested collection of tycons and vals simultaneously *)
and copy_and_remap_and_bind_tycons_and_vals g compgen tmenv tycons vs vslotvs = 
    let tycons' = tycons |> map copy_tycon in

    let tmenvinner = { tmenv with tyenv = bind_tycons tycons tycons' tmenv.tyenv } in 
    
    (*  Abstract (virtual) vslots in the tycons at TMDefRec nodes are binders. They also need to be copied and renamed. *)
    let vslotvs',tmenvinner = copy_and_remap_and_bind_vals g compgen tmenvinner vslotvs in

    (* Values need to be copied and renamed. *)
    let vs',tmenvinner = copy_and_remap_and_bind_vals g compgen tmenvinner vs in

    (* "if a type constructor is hidden then all its inner values and inner type constructors must also be hidden" *)
    (* Hence we can just lookup the inner tycon/value mappings in the tables. *)

    let lookup_val x = try vspec_map_find x tmenvinner.vspec_remap |> deref_val with Not_found -> error(Error("couldn't remap internal value",range_of_val x)); x in 
    let lookup_tycon x = try tcref_map_find (mk_local_ref x) tmenvinner.tyenv.tcref_remap |> deref_tycon  with Not_found -> error(Error("couldn't remap internal tycon "^showL(tyconL x),range_of_tycon x)); x in 
    iter2 (fun tc tc' -> 
        let tcd = data_of_tycon tc in 
        let tcd' = data_of_tycon tc' in 
        let tps',tmenvinner2 = tmenv_copy_remap_and_bind_typars tmenvinner tcd.tycon_typars in
        tcd'.tycon_typars         <- tps';
        tcd'.tycon_attribs        <- tcd.tycon_attribs |> remap_attribs g tmenvinner2;
        tcd'.tycon_repr           <- tcd.tycon_repr    |> Option.map (remap_tycon_repr g tmenvinner2);
        tcd'.tycon_abbrev         <- tcd.tycon_abbrev  |> Option.map (remap_type tmenvinner2.tyenv) ;
        tcd'.tycon_tcaug          <- tcd.tycon_tcaug   |> remap_tcaug tmenvinner2 ;
        tcd'.tycon_modul_contents <- notlazy (tcd.tycon_modul_contents 
                                              |> Lazy.force 
                                              |> map_immediate_vals_and_tycons_of_modtyp lookup_tycon lookup_val);
        tcd'.tycon_exnc_info      <- tcd.tycon_exnc_info      |> remap_tycon_exnc_info g tmenvinner2) tycons tycons';
    tycons',vs', tmenvinner

and remap_mexpr g compgen tmenv mexpr =
    match mexpr with
    | TMTyped(mty,def,m) -> 
        let def,_ = remap_mdef g compgen tmenv  def in
        let mty,tmenv = copy_remap_and_bind_mtyp g compgen tmenv mty in 
        TMTyped(mty,def,m), tmenv
and remap_mdefs g compgen tmenv x = map_acc_list (remap_mdef g compgen) tmenv x 
and remap_mdef g compgen tmenv mdef =
    match mdef with 
    | TMDefRec(tycons,binds,m) -> 
        (* Abstract (virtual) vslots in the tycons at TMDefRec nodes are binders. They also need to be copied and renamed. *)
        let vslotvs = vslot_vals_of_tycons tycons in
        let tycons,vs,tmenv = copy_and_remap_and_bind_tycons_and_vals g compgen tmenv tycons (map var_of_bind binds) vslotvs in
        let binds = remap_and_rename_binds g compgen tmenv binds vs in 
        TMDefRec(tycons,binds,m),tmenv
    | TMDefLet(bind,m)            ->
        let bind,tmenv = copy_and_remap_and_bind_binding g compgen tmenv bind in
        TMDefLet(bind, m),tmenv
    | TMDefs(defs)      -> 
        let defs,tmenv = remap_mdefs g compgen tmenv defs in 
        TMDefs(defs),tmenv
    | TMAbstract(mexpr) -> 
        let mexpr,tmenv = remap_mexpr g compgen tmenv mexpr in 
        TMAbstract(mexpr),tmenv
    | TMDefModul(TMBind(tycon, def))      -> 
        let [tycon],[],tmenv = copy_and_remap_and_bind_tycons_and_vals g compgen tmenv [tycon] [] [] in
        let def,tmenv = remap_mdef g compgen tmenv def in 
        TMDefModul(TMBind(tycon, def)),tmenv

and remap_assembly g compgen tmenv (TAssembly(mvs)) = 
    let mvs,z = map_acc_list (map_acc_TImplFile (remap_mexpr g compgen)) tmenv mvs in
    TAssembly(mvs),z

let copy_mtyp     g compgen mtyp = copy_remap_and_bind_mtyp g compgen empty_expr_remap mtyp |> fst
let copy_val      g compgen v    = copy_and_remap_and_bind_val g compgen empty_expr_remap v |> fst
let copy_expr     g compgen e    = remap_expr g compgen empty_expr_remap e
let copy_assembly g compgen e    = remap_assembly g compgen empty_expr_remap e |> fst

let inst_expr g tpinst e = remap_expr g false { empty_expr_remap with tyenv = mk_inst_tyenv tpinst } e

(*--------------------------------------------------------------------------
!* Replace Marks - adjust debugging marks when a lambda gets
 * eliminated (i.e. an expression gets inlined)
 *------------------------------------------------------------------------ *)

let rec remark_expr m x =
  match x with
  | TExpr_lambda (uniq,basevopt,vs,b,_,rty,fvs)  -> TExpr_lambda (uniq,basevopt,vs,remark_expr m b,m,rty,fvs)  
  | TExpr_tlambda (uniq,tps,b,_,rty,fvs) -> TExpr_tlambda (uniq,tps,remark_expr m b,m,rty,fvs)
  | TExpr_tchoose (tps,b,_) -> TExpr_tchoose (tps,remark_expr m b,m)
  | TExpr_letrec (binds,e,_,fvs) ->  TExpr_letrec (remark_binds m binds,remark_expr m e,m,fvs)
  | TExpr_let (bind,e,_,fvs) -> TExpr_let (remark_bind m bind,remark_expr m e,m,fvs)
  | TExpr_match (_,pt,targets,_,ty,_) -> prim_mk_match (m,remark_dtree m pt, Array.map (fun (TTarget(vs,e)) ->TTarget(vs, remark_expr m e)) targets,m,ty)
  | TExpr_val (x,isSuperInit,_) -> TExpr_val (x,isSuperInit,m)
  | TExpr_quote (raw,a,_,ty) ->  TExpr_quote (raw,remark_expr m a,m,ty)
  | TExpr_hole (m,ty) ->  TExpr_hole (m,ty)
  | TExpr_obj (n,typ,basev,basecall,overrides,iimpls,m,fvs) -> 
      TExpr_obj (n,typ,basev,remark_expr m basecall,
                   map (remark_method m) overrides,
                   map (remark_iimpl m) iimpls,m,fvs)
  | TExpr_op (op,tinst,args,_) -> TExpr_op (op,tinst,remark_exprs m args,m)
  | TExpr_link (eref) -> remark_expr m !eref
  | TExpr_app(e1,e1ty,tyargs,args,_) -> TExpr_app(remark_expr m e1,e1ty,tyargs,map (remark_expr m) args,m)
  | TExpr_seq (e1,e2,dir,_)  -> TExpr_seq (remark_expr m e1,remark_expr m e2,dir,m)
  | TExpr_static_optimization (eqns,e2,e3,_) -> TExpr_static_optimization (eqns,remark_expr m e2,remark_expr m e3,m)
  | TExpr_const (c,_,ty) -> TExpr_const (c,m,ty)
  
and remark_method m (TMethod(slotsig,tps,vs,e,m)) = 
  TMethod(slotsig,tps,vs,remark_expr m e,m)
and remark_iimpl m (ty,overrides) = 
  (ty, List.map (remark_method m) overrides)
and remark_exprs m es = list_map (remark_expr m) es
and remark_dtree m x =
  match x with 
  | TDSwitch(e1,csl,dflt,_) -> TDSwitch(remark_expr m e1, map (fun (TCase(test,y)) -> TCase(test,remark_dtree m y)) csl, Option.map (remark_dtree m) dflt,m)
  | TDSuccess (es,n) -> TDSuccess (remark_exprs m es,n)
  | TDBind (bind,rest) -> TDBind(remark_bind m bind,remark_dtree m rest)
and remark_binds m binds = map (remark_bind m) binds
and remark_bind m (TBind(v,repr)) = TBind(v, remark_expr m repr)


(*--------------------------------------------------------------------------
!* Reference semantics?
 *------------------------------------------------------------------------ *)

let rfield_alloc_observable f =not (static_of_rfield f) && f.rfield_mutable
let uconstr_alloc_observable uc = uc.uconstr_rfields.rfields_by_index |> array_exists rfield_alloc_observable
let ucref_alloc_observable uc = uc |> uconstr_of_ucref |> uconstr_alloc_observable
  
let tycon_alloc_observable tcr =
  if is_recd_tycon tcr or is_struct_tycon tcr then 
    tcr |> rfields_array_of_tycon |> array_exists rfield_alloc_observable
  else if is_union_tycon tcr then 
    tcr |> uconstrs_array_of_tycon |> array_exists uconstr_alloc_observable
  else false

let tcref_alloc_observable tcr = tycon_alloc_observable (deref_tycon tcr)
  
(* Although from the pure F# perspective exception values cannot be changed, the .NET *)
(* implementation of exception objects attaches a whole bunch of stack information to *)
(* each raised object.  Hence we treat exception objects as if they have identity *)
let ecref_alloc_observable (ecref:tycon_ref) = true 

(* Some of the implementations of library functions on lists use mutation on the tail *)
(* of the cons cell. These cells are always private, i.e. not accessible by any other *)
(* code until the construction of the entire return list has been completed. *)
(* However, within the implementation code reads of the tail cell must in theory be treated *)
(* with caution.  Hence we are conservative and within fslib we don't treat list *)
(* reads as if they were pure. *)
let ucref_rfield_mutable g ucref n = 
  (g.compilingFslib && g.tcref_eq (tcref_of_ucref ucref) g.list_tcr_canon && n = 1) ||
  (rfield_of_ucref_by_idx ucref n).rfield_mutable 
  
let ecref_rfield_mutable ecref n = 
  let ec = strip_eqns_from_ecref ecref in 
  if n < 0 || n >= List.length (instance_rfields_of_tycon ec) then errorR(Error(sprintf "ecref_rfield_mutable, exnc = %s, n = %d" (demangled_name_of_ecref ecref) n,range_of_tcref ecref));
  (rfield_of_ecref_by_idx ecref n).rfield_mutable 

let use_genuine_field tycon f = 
    isSome (literal_value_of_rfield f) || is_enum_tycon tycon || f.rfield_secret

let gen_field_name tycon f = 
    if use_genuine_field tycon f then f.rfield_id.idText
    else "_"^f.rfield_id.idText 


(*-------------------------------------------------------------------------
 * Helpers for building code contained in the initial environment
 *------------------------------------------------------------------------- *)


let lib_Quotations_path = Il.split_namespace lib_Quotations_name
let mk_Quotations_nlpath g = NLPath(g.fslibCcu,lib_Quotations_path)
let mk_Quotations_tcref g n = mk_nonlocal_tcref (mk_Quotations_nlpath g) n 

let lib_Quotations_Typed_path = Il.split_namespace lib_Quotations_Typed_name
let mk_Quotations_Typed_nlpath g = NLPath(g.fslibCcu,lib_Quotations_Typed_path)
let mk_expr_ty g ty =  TType_app(mk_Quotations_tcref g "Expr`1",[ty])
let mk_expr_template_ty g ty1 ty2 ty3 ty4 = TType_app(mk_Quotations_tcref g "Template`4",[ty1;ty2;ty3;ty4])
let lib_Quotations_Raw_name = lib_MF_name ^ ".Quotations.Raw"
let lib_Quotations_Raw_path = Il.split_namespace lib_Quotations_Raw_name
let mk_Quotations_Raw_nlpath g = NLPath(g.fslibCcu,lib_Quotations_Raw_path)
let mk_raw_expr_ty g =  TType_app(mk_Quotations_tcref g "Expr",[])
let mk_raw_expr_template_ty g ty1 ty2  = TType_app(mk_Quotations_tcref g "Template`2",[ty1;ty2])
let mk_bytearray_ty g = mk_il_arr_ty g 1 g.byte_ty

let mk_hash_param_ty g ty = ty --> (mk_byref_typ g g.int_ty  --> g.int_ty)
let mk_compare_ty g ty = ty --> (ty --> g.int_ty)
let mk_equals_obj_ty g ty = ty --> (g.obj_ty --> g.bool_ty)
let mk_rel_ty g ty = ty --> (ty --> g.bool_ty)

(*---------------------------------------------------------------------------
 * Tuples
 *------------------------------------------------------------------------- *)

let mk_tupled_ty g tys = 
    match tys with 
    | [] -> g.unit_ty 
    | [h] -> h
    | _ -> mk_tuple_ty tys

let mk_tupled_vars_ty g vs = 
    mk_tupled_ty g (map type_of_val vs)

let mk_meth_ty g argtys rty = mk_tupled_ty g argtys --> rty 
let mk_nativeptr_ty g ty = TType_app (g.nativeptr_tcr, [ty])
let mk_array_ty g ty = TType_app (g.array_tcr, [ty])

(*----------------------------------------------------------------------------
 * type_of_expr
 *--------------------------------------------------------------------------*)
 
let rec type_of_expr g e = 
  match e with 
  | TExpr_app(f,fty,tyargs,args,m) -> apply_types fty (tyargs,args)
  | TExpr_obj (_,ty,_,_,_,_,_,_)  
  | TExpr_match (_,_,_,_,ty,_) 
  | TExpr_quote(_,_,_,ty) 
  | TExpr_hole(_,ty) 
  | TExpr_const(_,_,ty)              -> (ty)
  | TExpr_val(vref,isSuperInit,_)  -> (type_of_vref vref)
  | TExpr_seq(a,b,k,_) -> type_of_expr g (match k with NormalSeq  -> b | ThenDoSeq -> a)
  | TExpr_lambda(_,basevopt,vs,_,_,rty,_) -> (mk_tupled_vars_ty g vs --> rty)
  | TExpr_tlambda(_,tyvs,_,_,rty,_) -> (tyvs +-> rty)
  | TExpr_let(_,e,_,_) 
  | TExpr_tchoose(_,e,_)
  | TExpr_link { contents=e}
  | TExpr_static_optimization (_,_,e,_) 
  | TExpr_letrec(_,e,_,_) -> type_of_expr g e
  | TExpr_op(op,tinst,args,m) -> 
      match op with 
      | TOp_coerce -> (match tinst with [to_ty;from_ty] -> to_ty | _ -> failwith "bad TOp_coerce node")
      | (TOp_ilcall (_,_,_,rtys) | TOp_asm(_,rtys)) -> (match rtys with [h] -> h | _ -> g.unit_ty)
      | TOp_uconstr uc -> rty_of_uctyp uc tinst
      | TOp_recd (_,tcref) -> mk_tyapp_ty tcref tinst
      | TOp_exnconstr uc -> g.exn_ty
      | TOp_bytes _ -> mk_bytearray_ty g
      | TOp_tuple_field_get(i) -> List.nth tinst i
      | TOp_tuple -> mk_tuple_ty tinst
      | (TOp_for _ | TOp_while) -> g.unit_ty
      | TOp_array -> (match tinst with [ty] -> mk_array_ty g ty | _ -> failwith "bad TOp_array node")
      | (TOp_try_catch | TOp_try_finally) -> (match tinst with [ty] ->  ty | _ -> failwith "bad TOp_try node")
      | TOp_field_get_addr(fref) -> mk_byref_typ g (actual_rtyp_of_rfref fref tinst)
      | TOp_field_get(fref) -> actual_rtyp_of_rfref fref tinst
      | (TOp_field_set _ | TOp_constr_field_set _ | TOp_exnconstr_field_set _ | TOp_lval_op ((LSet | LByrefSet),_)) ->g.unit_ty
      | TOp_constr_tag_get(cref) -> g.int_ty
      | TOp_constr_field_get(cref,j) -> typ_of_ucref_rfield_by_idx cref tinst j
      | TOp_exnconstr_field_get(ecref,j) -> typ_of_ecref_rfield ecref j
      | TOp_lval_op (LByrefGet, v) -> dest_byref_ty g (type_of_vref v)
      | TOp_lval_op (LGetAddr, v) -> mk_byref_typ g (type_of_vref v)
      | TOp_get_ref_lval -> (match tinst with [ty] -> mk_byref_typ g ty | _ -> failwith "bad TOp_get_ref_lval node")
      | TOp_trait_call (TTrait(_,_,_,_,ty)) -> ty

(*--------------------------------------------------------------------------
!* Make applications
 *------------------------------------------------------------------------ *)

let prim_mk_app (f,fty) tyargs argsl m = 
  TExpr_app(f,fty,tyargs,argsl,m)

let rec mk_expr_appl_aux  f fty argsl m =
  if verbose then  dprintf1 "--- mk_expr_appl_aux, fty = %s\n" ((DebugPrint.showType fty));
  match argsl with 
  | arg :: rest ->
      begin match f with 
      (* Try to combine the term application with others *)
      | TExpr_app(f',fty',tyargs,pargs,m2) 
          (* Only do this when the formal return type of the function type is another function type *)
          when is_fun_ty (formal_apply_types fty' (tyargs,pargs)) ->
            if verbose then  dprintf1 "--- mk_expr_appl_aux, combine, fty' = %s\n" ((DebugPrint.showType fty'));
            let pargs' = pargs@[arg] in 
            let f'' = prim_mk_app (f',fty') tyargs pargs' (union_ranges m2 m) in 
            let fty'' = apply_types fty' (tyargs,pargs')  in
            if verbose then  dprintf1 "--- mk_expr_appl_aux, combined, continue, fty'' = %s\n" ((DebugPrint.showType fty''));
            mk_expr_appl_aux f'' fty'' rest m
      | _ -> 
          if not (is_fun_ty fty) then error(InternalError("expected a function type",m));
          let _,rfty = dest_fun_typ fty in 
          mk_expr_appl_aux (prim_mk_app (f,fty) [] [arg] m) rfty rest m
      end

  | [] -> (f,fty)

let rec mk_appl_aux f fty tyargsl argsl m =
  match tyargsl with 
  | tyargs :: rest -> 
      begin match tyargs with 
      | [] -> mk_appl_aux f fty rest argsl m
      | _ -> 
        let arfty = reduce_forall_typ fty tyargs in 
        mk_appl_aux (prim_mk_app (f,fty) tyargs [] m) arfty rest argsl m
      end
  | [] -> mk_expr_appl_aux  f fty argsl m
      
let mk_appl((f,fty),tyargsl,argl,m) = fst (mk_appl_aux f fty tyargsl argl m)
let mk_tyapp m (f,fty) tyargs = match tyargs with [] -> f | _ -> prim_mk_app (f,fty) tyargs [] m 

let mk_val_set m v e = TExpr_op(TOp_lval_op (LSet, v), [], [e], m)             (*   localv <- e      *)
let mk_lval_set m v e = TExpr_op(TOp_lval_op (LByrefSet, v), [], [e], m)       (*  *localv_ptr = e   *)
let mk_lval_get m v = TExpr_op(TOp_lval_op (LByrefGet, v), [], [], m)          (* *localv_ptr        *)
let mk_val_addr m v = TExpr_op(TOp_lval_op (LGetAddr, v), [], [], m)           (* &localv            *)

(*--------------------------------------------------------------------------
!* Decision tree reduction
 *------------------------------------------------------------------------ *)

let rec acc_targets_of_dtree tree acc =
  match tree with 
  | TDSwitch (_,edges,dflt,_) -> fold_right (dest_of_case >> acc_targets_of_dtree) edges (Option.fold_right acc_targets_of_dtree dflt acc)
  | TDSuccess (_,i) -> gen_insert (=) i acc
  | TDBind (_,rest) -> acc_targets_of_dtree rest acc

let rec map_acc_tips_of_dtree f tree =
  match tree with 
  | TDSwitch (e,edges,dflt,m) -> TDSwitch (e,map (map_acc_tips_of_edge f) edges,Option.map (map_acc_tips_of_dtree f) dflt,m)
  | TDSuccess (es,i) -> f es i  
  | TDBind (bind,rest) -> TDBind(bind,map_acc_tips_of_dtree f rest)
and map_acc_tips_of_edge f (TCase(x,t)) = TCase(x,map_acc_tips_of_dtree f t)

let map_targets_of_dtree f tree = map_acc_tips_of_dtree (fun es i -> TDSuccess(es, f i)) tree

(* Dead target elimination *)
let eliminate_dead_targets_from_match tree targets =
    let used = acc_targets_of_dtree tree [] |> Array.of_list in
    if Array.length used < Array.length targets then (
        Array.sort compare used;
        let nused = Array.length used in 
        let ntargets = Array.length targets in
        let tree' = 
            let remap = Array.create ntargets (-1) in
            Array.iteri (fun i tgn -> remap.(tgn) <- i) used;
            map_targets_of_dtree (fun tgn -> if remap.(tgn) = -1 then failwith "eliminate_dead_targets_from_match: failure while eliminating unused targets"; remap.(tgn)) tree in
        let targets' = Array.map (Array.get targets) used in 
        tree',targets'
    ) else (
        tree,targets
    ) 

let rec dtree_has_bindings tree =
    match tree with 
    | TDSwitch (_,edges,dflt,_) -> List.exists (dest_of_case >> dtree_has_bindings) edges || (Option.exists dtree_has_bindings dflt)
    | TDSuccess _ -> false
    | TDBind _ -> true

(* If a target has assignments and can only be reached through one *)
(* branch (i.e. is "linear"), then transfer the assignments to the r.h.s. to be a "let". *)
(* Don't do this when there are any bindings in the tree, since the variables would be extruded from their scope. *)
(* REVIEW: lift bindings in the tree out of the tree to be regular let-bindings shared by the tree and the target *)
(* nodes. *)
let fold_linear_binding_targets_of_match m tree targets =
  if dtree_has_bindings tree then tree,targets else
  let rec acc_tips_of_dtree tree acc =
    match tree with 
    | TDSwitch (_,edges,dflt,_) -> fold_right acc_tips_of_edge edges (Option.fold_right acc_tips_of_dtree dflt acc)
    | TDSuccess (es,i) -> Map.add i (es :: Pmap.tryfind_multi i acc) acc
    | TDBind (bind,rest) -> acc_tips_of_dtree rest acc
  and acc_tips_of_edge (TCase(_,x)) acc = acc_tips_of_dtree x acc in

  let linear_tips = 
     acc_tips_of_dtree tree Map.empty 
     |> Map.filter (fun k v -> match v with [es] -> true | _ -> false) in 
  if not (Map.is_empty linear_tips) then 
      let tree' = 
        tree |> map_acc_tips_of_dtree (fun es i -> 
           match Map.tryfind i linear_tips with 
           | None -> TDSuccess(es,i)
           | Some _ ->  TDSuccess([],i)) in 

      let targets' = 
        targets |> Array.mapi (fun i (TTarget(vs,e) as tg) -> 
            match Map.tryfind i linear_tips with 
            | Some (es :: _) -> TTarget([],mk_lets_bind m (mk_binds vs es) e)
            | _ -> tg ) in
      tree',targets'
  else
    tree,targets

(* Simplify a little as we go, including dead target elimination *)
let rec simplify_trivial_match exprm matchm ty tree targets  = 
  match tree with 
  | TDSuccess(es,n) -> 
      if n >= Array.length targets then failwith "simplify_trivial_match: target out of range";
      let (TTarget(vs,rhs)) = targets.(n) in 
      if List.length vs <> List.length es then failwith ("simplify_trivial_match: invalid argument, n = "^string_of_int n^", length targets = "^string_of_int (Array.length targets));
      mk_lets_bind exprm (mk_binds vs es) rhs
  | _ -> 
      prim_mk_match (exprm,tree,targets,matchm,ty)
 
(* Simplify a little as we go, including dead target elimination *)
let rec mk_and_optimize_match exprm matchm ty tree targets  = 
  let targets = Array.of_list targets in 
  match tree with 
  | TDSuccess _ -> 
      simplify_trivial_match exprm matchm ty tree targets
  | _ -> 
      let tree,targets = eliminate_dead_targets_from_match tree targets in 
      let tree,targets = fold_linear_binding_targets_of_match exprm tree targets in 
      simplify_trivial_match exprm matchm ty tree targets


(*-------------------------------------------------------------------------
!* mk_expra_of_expr
 *------------------------------------------------------------------------- *)

type mutates = DefinitelyMutates | PossiblyMutates | NeverMutates
exception PotentialMutationWarning of string * range 

let rec mk_expra_of_expr g valu mut e m =
  if not valu then (fun x -> x),e else
  match e with 
  (* LVALUE: "x" where "x" is byref *)
  | TExpr_op(TOp_lval_op (LByrefGet, v), _,[], m) 
     -> 
      (fun x -> x), expr_for_vref m v
  (* LVALUE: "x" where "x" is mutable local *)
  | TExpr_val(v, _,m) when mutability_of_vref v = Mutable
     -> 
    (fun x -> x), mk_val_addr m v
  (* LVALUE: "x" where "e.x" is mutable record field. "e" may be an lvalue *)
  | TExpr_op(TOp_field_get rfref, tinst,[expr],m) when (rfield_of_rfref rfref).rfield_mutable
     -> 
      let exprty = type_of_expr g expr in 
      let wrap,expra = mk_expra_of_expr g (is_struct_typ exprty) mut expr m in
      wrap, mk_recd_field_get_addr_via_expra(expra,rfref,tinst,m)

  (* LVALUE: "x" where "e.x" is a .NET static field. "e" may be an lvalue *)
  | TExpr_op(TOp_asm ([Il.I_ldsfld(vol,fspec)],[ty2]), tinst,[],m) (* when ((fspec |> enclosing_typ_of_fspec |> boxity_of_typ) = AsValue)  *)
     -> 
      (fun x -> x),TExpr_op(TOp_asm ([Il.I_ldsflda(fspec)],[mk_byref_typ g ty2]), tinst,[],m)

  (* LVALUE: "x" where "e.x" is a .NET instance field. "e" may be an lvalue *)
  | TExpr_op(TOp_asm ([Il.I_ldfld(align,vol,fspec)],[ty2]), tinst,[e],m) (* when ((fspec |> enclosing_typ_of_fspec |> boxity_of_typ) = AsValue) *)
     -> 
      (fun x -> x),TExpr_op(TOp_asm ([Il.I_ldflda(fspec)],[mk_byref_typ g ty2]), tinst,[e],m)

  (* LVALUE: "x" where "x" is mutable static field. *)
  | TExpr_op(TOp_field_get rfref, tinst,[],m) when (rfield_of_rfref rfref).rfield_mutable
     -> 
      (fun x -> x), mk_static_rfield_get_addr(rfref,tinst,m)

  (* LVALUE: "x" where "e.(n)" e is an array. *)
(*
  | TExpr_app((TExpr_val(vf,_,_) as aexpr),_,[tyarg1; tyarg2; tyarg3],[nexpr],_) 
       when g.vref_eq vf g.idx_lookup_op_vref && is_il_arr_typ g tyarg1 -> 
      (fun x -> x), mk_get_array_elem_addr(aexpr,nexpr,tyarg1,m)

  | TExpr_app((TExpr_val(vf,_,_) as aexpr),_,[tyarg1],[nexpr],_) 
       when g.vref_eq vf g.array_lookup_op_vref -> 
      (fun x -> x), mk_get_array_elem_addr(aexpr,nexpr,tyarg1,m)
*)

  | TExpr_val(v, _,m) when mut = DefinitelyMutates
     -> 
      if is_byref_ty g (type_of_vref v) then error(Error("Unexpected use of a byref-typed variable",m));
      error(Error("A value must be local and mutable in order to mutate the contents of a .NET value type, e.g. 'let mutable x = ...'",m));
(*
  (* TODO : be careful, we may not know it's an array until after the end of type inference *)
  (* TODO : This is a good reason to resolve .[] lookups _early_ *)
  | TExpr_app((TExpr_val(vf,_,_) as aexpr),_,[tyarg1; tyarg2; tyarg3],[nexpr],_) 
       when g.vref_eq vf g.idx_lookup_op_vref -> 
      (fun x -> x), mk_get_array_elem_addr(f,fref,tinst,m)
*)       
       
  | TExpr_val(v, _,m) when mut = DefinitelyMutates
     -> 
      if is_byref_ty g (type_of_vref v) then error(Error("Unexpected use of a byref-typed variable",m));
      error(Error("The value '"^display_name_of_vref v^"' is not mutable. A value must be local and mutable to be used in this expression construct, e.g. 'let mutable x = ...'",m));
  | _ -> 
      begin match mut with 
      | NeverMutates -> ()
      | DefinitelyMutates -> 
        errorR(Error("Invalid mutation of a constant expression. Consider copying the expression to a mutable local, e.g. 'let mutable x = ...'",m));
      | PossiblyMutates -> 
        warning(PotentialMutationWarning("Possible mutation of a constant expression. The expression has been copied to avoid mutating the original copy. Consider explicitly copying the expression to a mutable local, e.g. 'let mutable x = ...'",m));
      end;
      let tmp,tmp_vref,tmpe = mk_mut_local m (nng.nngApply "$struct-addr" m) (type_of_expr g e) in 
      (fun rest -> mk_let m tmp e rest), (mk_val_addr m tmp_vref)

let mk_recd_field_get g (e,fref,tinst,finst,m) = 
  let tycon,fspec = deref_rfield fref in 
  let ftyp = actual_rtyp_of_rfref fref tinst in
  let wrap,e' = mk_expra_of_expr g (is_struct_tycon tycon) NeverMutates e m  in
  wrap (mk_tyapp m (mk_recd_field_get_via_expra(e',fref,tinst,m), ftyp) finst)

let mk_recd_field_set g (e,fref,tinst,e2,m) = 
  let tycon,fspec = deref_rfield fref in 
  let wrap,e' = mk_expra_of_expr g (is_struct_tycon tycon) DefinitelyMutates e m  in
  wrap (mk_recd_field_set_via_expra(e',fref,tinst,e2,m))

(*---------------------------------------------------------------------------
!* Compute fixups for letrec's.
 *
 * Generate an assignment expression that will fixup the recursion 
 * amongst the vals on the r.h.s. of a letrec.  The returned expressions 
 * include disorderly constructs such as expressions/statements 
 * to set closure environments and non-mutable fields. These are only ever 
 * generated by the backend code-generator when processing a "letrec"
 * construct.
 *
 * [self] is the top level value that is being fixed
 * [expr_to_fix] is the r.h.s. expression
 * [rvs] is the set of recursive vals being bound. 
 * [acc] accumulates the expression right-to-left. 
 *
 * Traversal of the r.h.s. term must happen back-to-front to get the
 * uniq's for the lambdas correct in the very rare case where the same lambda
 * somehow appears twice on the right.
 *------------------------------------------------------------------------- *)

let rec iter_letrec_fixups g (selfv : val_spec option) rvs ((access : expr),set) expr_to_fix  = 
  let expr_to_fix =  strip_expr expr_to_fix in 
  match expr_to_fix with 
  | TExpr_const _ -> ()
  | TExpr_op(TOp_tuple,argtys,args,m) ->
      list_iteri
        (fun n -> 
          iter_letrec_fixups g None rvs 
            (mk_tuple_field_get(access,argtys,n,m), 
            (fun e -> 
              (* NICE: it would be better to do this check in the type checker *)
              errorR(Error("Recursively defined values may not appear directly as part of the construction of a tuple value within a recursive binding.",m));
              e)))
        args 
  | TExpr_op(TOp_uconstr (c),tinst,args,m) ->
      list_iteri
        (fun n -> 
          iter_letrec_fixups g None rvs 
            (mk_uconstr_field_get(access,c,tinst,n,m), 
             (fun e -> 
               (* NICE: it would be better to do this check in the type checker *)
               let tycon = tcref_of_ucref c in 
               if not (item_ref_in_this_assembly false tycon) then
                 errorR(Error("Recursive values may not appear directly as a construction of the type '"^(name_of_tcref tycon)^"' within a recursive binding except in the assembly where that type is defined",m));
               mk_uconstr_field_set(access,c,tinst,n,e,m))))
        args 
  | TExpr_op(TOp_recd (_,tcref),tinst,args,m) -> 
      iter2 
        (fun fref arg -> 
          let fspec = rfield_of_rfref fref in
          iter_letrec_fixups g None rvs 
            (mk_recd_field_get_via_expra(access,fref,tinst,m), 
             (fun e -> 
               (* NICE: it would be better to do this check in the type checker *)
               if not fspec.rfield_mutable && not (item_ref_in_this_assembly false tcref) then
                 errorR(Error("Recursive values may not be directly assigned to the non-mutable field '"^fspec.rfield_id.idText^"' of the type '"^(name_of_tcref tcref)^"' within a recursive binding except in the assembly where that type is defined",m));
               mk_recd_field_set g (access,fref,tinst,e,m))) arg )
        (instance_rfrefs_of_tcref tcref)
        args   (* args also!! in arg order *)
  | TExpr_val (_,_,m) 
  | TExpr_lambda (_,_,_,_,m,_,_)  
  | TExpr_obj (_,_,_,_,_,_,m,_)  
  | TExpr_tchoose (_,_,m)  
  | TExpr_tlambda (_,_,_,m,_,_)  -> 
      rvs selfv access set expr_to_fix
  | e -> ()


(*--------------------------------------------------------------------------
!* optimization utilities
 *------------------------------------------------------------------------*)

let const_eq (c1:tconst) (c2:tconst) = (c1 = c2)

let discrim_eq g d1 d2 =
  match d1,d2 with 
  | TTest_unionconstr (c1,_),    TTest_unionconstr(c2,_) -> g.ucref_eq c1 c2
  | TTest_array_length (n1,_),   TTest_array_length(n2,_) -> (n1=n2)
  | TTest_const c1,              TTest_const c2 -> const_eq c1 c2
  | TTest_isnull ,               TTest_isnull -> true
  | TTest_isinst (srcty1,tgty1), TTest_isinst (srcty2,tgty2) -> type_equiv g srcty1 srcty2 && type_equiv g tgty1 tgty2
  | TTest_query (_,_,vrefOpt1,n1,apinfo1),        TTest_query (_,_,vrefOpt2,n2,apinfo2) -> 
      begin match vrefOpt1, vrefOpt2 with 
      | Some vref1, Some vref2 -> g.vref_eq vref1 vref2 && n1 = n2 
      | _ -> false (* for equality purposes these are considered unequal! This is because adhoc computed patterns have no identity. *)
      end
  | _ -> false


(*--------------------------------------------------------------------------
!* computations on constraints
 *------------------------------------------------------------------------*)
 
let joinTyparStaticReq r1 r2 = 
  match r1,r2 with
  | NoStaticReq,r | r,NoStaticReq -> r 
  | HeadTypeStaticReq,r | r,HeadTypeStaticReq -> r
  | CompleteStaticReq,CompleteStaticReq -> CompleteStaticReq
  


(*-------------------------------------------------------------------------
!* exprFolder - fold steps
 *-------------------------------------------------------------------------*)

type 'a exprFolder = {exprIntercept    : ('a -> expr -> 'a) -> 'a -> expr  -> 'a option;   (* intercept? *)
                      (* hook. this bool is 'bound in dtree' *)
                      valDefAcc          : 'a -> bool * val_spec  -> 'a;                     
                      (* hook.  these values are always bound to these expressions. bool is 'recursively' *)
                      valBindAcc         : 'a -> bool * bind list -> 'a;         
                      dtreeAcc         : 'a -> dtree            -> 'a;                     (* hook *)
                      targetIntercept  : ('a -> expr -> 'a) -> 'a -> dtree_target  -> 'a option; (* intercept? *)
                      tmethodIntercept : ('a -> expr -> 'a) -> 'a -> tmethod -> 'a option; (* intercept? *)
                     }

let exprFolder0 =
  { exprIntercept    = (fun exprF z x -> None);
    valDefAcc          = (fun z b  -> z);
    valBindAcc         = (fun z (recusive,bs) -> z);
    dtreeAcc         = (fun z dt -> z);
    targetIntercept  = (fun exprF z x -> None);
    tmethodIntercept = (fun exprF z x -> None);
  }


(*-------------------------------------------------------------------------
!* foldExpr
 *-------------------------------------------------------------------------*)

let mkFolders (folders : _ exprFolder) =
  (******
   * Adapted from usage info folding.
   * Collecting from exprs at moment.
   * To collect ids etc some additional folding needed, over formals etc.
   ******)
  let {exprIntercept    = exprIntercept; 
       valDefAcc          = valDefAcc;
       valBindAcc         = valBindAcc;
       dtreeAcc         = dtreeAcc;
       targetIntercept  = targetIntercept;
       tmethodIntercept = tmethodIntercept} = folders in
  let rec exprsF z xs = fold_left exprF z xs
  and exprF z x =
    match exprIntercept exprF z x with (* fold this node, then recurse *)
      Some z -> z (* intercepted *)
    | None ->     (* structurally recurse *)
        match x with
        | TExpr_const (c,m,ty)                                     -> z
        | TExpr_val (v,isSuperInit,m)                               -> z
        | TExpr_op (c,tyargs,args,m)                               -> exprsF z args
        | TExpr_seq (x0,x1,dir,m)                                  -> exprsF z [x0;x1]
        | TExpr_lambda(lambda_id ,basevopt,argvs,body,m,rty,_)     -> exprF  z body
        | TExpr_tlambda(lambda_id,argtyvs,body,m,rty,_)            -> exprF  z body
        | TExpr_tchoose(_,body,m)                                  -> exprF  z body
        | TExpr_app (f,fty,tys,argtys,m)                           -> let z = exprF z f in
                                                                      let z = exprsF z argtys in
                                                                      z
        | TExpr_letrec (binds,body,m,_)                            -> let z = bindsF false true z binds in
                                                                      let z = exprF z body in
                                                                      z
        | TExpr_let    (bind,body,m,_)                             -> let z = bindsF false false z [bind] in
                                                                      let z = exprF z body in
                                                                      z
        | TExpr_link rX                                            -> exprF z (!rX)
        | TExpr_match (exprm,dtree,targets,m,ty,_)                 -> let z = dtreeF z dtree in
                                                                      let z = Array.fold_left targetF z targets in
                                                                      z
        | TExpr_quote(raw,e,m,_)                                           -> z (* Do not collect at higher levels *)
        | TExpr_hole _                                             -> z
        | TExpr_obj (n,typ,basev,basecall,overrides,iimpls,m,_)      -> let z = exprF z basecall in
                                                                      let z = fold_left tmethodF z overrides in
                                                                      let z = fold_left (foldOn snd (fold_left tmethodF)) z iimpls in
                                                                      z
       | TExpr_static_optimization (tcs,csx,x,m)                   -> exprsF z [csx;x]
  and bindsF dtree isRec z binds =
    let z = valBindAcc z (isRec,binds) in
    fold_left (bindF dtree) z binds 

  and bindF dtree z bind =
    let z = valDefAcc z (dtree,var_of_bind bind) in
    exprF z (rhs_of_bind bind)

  and dtreeF z dtree =
    let z = dtreeAcc z dtree in
    match dtree with
    | TDBind (bind,rest)            -> let z = bindsF true false z [bind] in
                                       dtreeF z rest
    | TDSuccess (args,n)            -> exprsF z args
    | TDSwitch (test,dcases,dflt,r) -> let z = exprF z test in
                                      let z = fold_left dcaseF z dcases in
                                      let z = fold_option dtreeF z dflt in
                                      z

  and dcaseF z = function
      TCase (test,dtree)   -> dtreeF z dtree (* not collecting from test *)

  and targetF z x =
    (match targetIntercept exprF z x with 
         Some z -> z (* intercepted *)
       | None ->     (* structurally recurse *)
           let (TTarget (argvs,body)) = x in
           exprF z body)
            
  and tmethodF z x =
    (match tmethodIntercept exprF z x with 
         Some z -> z (* intercepted *)
       | None ->     (* structurally recurse *)
           let (TMethod(_,_,_,e,_)) = x in
           exprF z e)

  and mexprF z x =
      match x with 
      | TMTyped(mtyp,def,m) -> mdefF z def
  and mdefF z x = 
      match x with
      | TMDefRec(tycons,binds,m) -> 
          (* REVIEW: also iterate the abstract slot vspecs hidden in the _vslots field in the tycons *)
          bindsF false true z binds
      | TMDefLet(bind,m) -> bindsF false false z [bind]
      | TMDefs(defs) -> fold_left mdefF z defs 
      | TMAbstract(x) -> mexprF z x
      | TMDefModul(TMBind(nm, def)) -> mdefF z def

  and implF z x = foldTImplFile mexprF z x
  and implsF z (TAssembly(x)) = fold_left implF z x

  in
  exprF, implsF

let foldExpr     folders = let exprF,implsF = mkFolders folders in exprF
let foldAssembly folders = let exprF,implsF = mkFolders folders in implsF

    
(*-------------------------------------------------------------------------
!* exprStats
 *-------------------------------------------------------------------------*)

let exprStats x =
  let count = ref 0 in
  let folders = {exprFolder0 with exprIntercept = (fun exprF z x -> (count := !count + 1; None))} in
  let () = foldExpr folders () x in
  string_of_int !count ^ " TExpr nodes"

    
(*-------------------------------------------------------------------------
!* 
 *------------------------------------------------------------------------- *)

let mk_string g m n = TExpr_const(TConst_string n,m,g.string_ty)
let mk_int64 g m n = TExpr_const(TConst_int64 n,m,g.int64_ty)
let mk_bool g m b = TExpr_const(TConst_bool b,m,g.bool_ty)
let mk_true g m = mk_bool g m true
let mk_false g m = mk_bool g m false
let mk_unit g m = TExpr_const(TConst_unit,m,g.unit_ty)
let mk_int32 g m n =  TExpr_const(TConst_int32 n,m,g.int32_ty)
let mk_int g m n =  mk_int32 g m (Int32.of_int n)
let mk_zero g m =  mk_int g m 0
let mk_one g m =  mk_int g m 1
let mk_minus_one g  m =  mk_int g m (-1)

let dest_int32 = function TExpr_const(TConst_int32 n,m,ty) -> Some n | _ -> None

let is_fslib_IPrimitiveDelegateEvent_ty g ty     = is_stripped_tyapp_typ ty && g.tcref_eq g.fslib_IPrimitiveDelegateEvent_tcr (tcref_of_stripped_typ ty)
let dest_fslib_IPrimitiveDelegateEvent_ty g ty   = 
  if is_fslib_IPrimitiveDelegateEvent_ty g ty then 
    match tinst_of_stripped_typ ty with 
    | [ty1] -> ty1
    | _ -> failwith "dest_fslib_IPrimitiveDelegateEvent_ty: internal error"
  else failwith "dest_fslib_IPrimitiveDelegateEvent_ty: not an IDelegateEvent type"
let mk_fslib_IDelegateEvent_ty g ty1 ty2 = TType_app (g.fslib_IDelegateEvent_tcr, [ty1;ty2])
let mk_fslib_IEvent_ty g ty = TType_app (g.fslib_IEvent_tcr, [ty])

let mk_refcell_contents_rfref g  = mk_rfref g.refcell_tcr "contents"
let delayed_ucref g = mk_ucref g.lazystatus_tcr "Delayed"
let undelayed_ucref g = mk_ucref g.lazystatus_tcr "Value"
let mk_lazy_error_ucref g = mk_ucref g.lazystatus_tcr "Exception"
let mk_lazy_status_rfref g = mk_rfref g.lazy_tcr_canon "status"

let typed_expr_for_val m v = expr_for_val m v,type_of_val v


(*-------------------------------------------------------------------------
 * Tuples...
 *------------------------------------------------------------------------- *)
 
let mk_tupled g m es tys = 
    match es with 
    | [] -> mk_unit g m 
    | [e] -> e
    | _ -> TExpr_op(TOp_tuple,tys,es,m)

let mk_tupled_vars g m vs = mk_tupled g m (map (expr_for_val m) vs) (map type_of_val vs)

(*--------------------------------------------------------------------------
!* Permutations
 *------------------------------------------------------------------------*)
    
let inverse_perm (sigma:int array) =
  let n = Array.length sigma in
  let inv_sigma = Array.create n (-1) in
  for i = 0 to n-1 do
    let sigma_i = sigma.(i) in
    (* assert( inv_sigma.(sigma_i) = -1 ); *)
    inv_sigma.(sigma_i) <- i
  done;
  inv_sigma
  
let permute (sigma:int array) (data:'a array) = 
  let n = Array.length sigma in
  let inv_sigma = inverse_perm sigma in
  Array.init n (fun i -> data.(inv_sigma.(i)))
  
let permuteList sigma dataL = Array.to_list (permute sigma (Array.of_list dataL))


(*--------------------------------------------------------------------------
!* Permute expressions
 *------------------------------------------------------------------------*)
    
let rec existsR a b pred = if a<=b then pred a || existsR (a+1) b pred else false
let mapi_acc_list f z xs =
  let rec fmapi f i z = function
    | []    -> z,[]
    | x::xs -> let z,x  = f i z x in
               let z,xs = fmapi f (i+1) z xs in
               z,x::xs
  in
  fmapi f 0 z xs

let permuteExpr (sigma:int array) (expr:expr array) (typ:typ array) (names:string array) =
  (* Given expr = xi = [| x0; ... xN |]
   * Given sigma a permutation to apply to the xi.
   * Return (bindings',expr') such that:
   *   (a) xi are permutated under sigma, xi -> position sigma(i).
   *------
   * Motivation:
   *   opt.ml    - put record field assignments in order under known effect information
   *   ilxgen.ml - put record field assignments in order if necessary (no optimisations)
   *               under unknown-effect information.
   *------
   * Method:
   *)
  let inv_sigma = inverse_perm sigma in
  let liftPosition i =
    (* In english, lift out xi if      
         * LC2: xi goes to position that will be preceeded by
         *       (an expr with an effect that originally followed xi).
         *)
    (let i' = sigma.(i) in
         existsR 0 (i' - 1) (fun j' ->
                               let j = inv_sigma.(j') in
                               j > i))
  in
  let rewrite i rbinds xi =
    if liftPosition i then
      let tmpv,tmpe = mk_compgen_local (range_of_expr xi) names.(i) typ.(i) in
      let bind = mk_bind tmpv xi in
      bind :: rbinds,tmpe
    else
      rbinds,xi
  in
  let xis = Array.to_list expr in
  let rbinds,xis = mapi_acc_list rewrite [] xis in
  let binds = List.rev rbinds in
  let expr  = permute sigma (Array.of_list xis) in
  binds,expr
    
let permuteExprList (sigma:int array) (expr:expr list) (typ:typ list)  (names:string list) =
  let binds,expr = permuteExpr sigma (Array.of_list expr) (Array.of_list typ)  (Array.of_list names) in
  binds,Array.to_list expr
  

(*-------------------------------------------------------------------------
 * Build lazy expressions...
 *------------------------------------------------------------------------- *)

let mk_seq  m e1 e2 = TExpr_seq(e1,e2,NormalSeq,m)
let rec mk_seqs g m es = match es with | [e] -> e | e::es -> mk_seq m e (mk_seqs g m es) | [] -> mk_unit g m

let mk_recd(lnk,tcref,tinst,rfrefs,args,m) =  
    (* Remove any abbreviations *)
    let tcref,tinst = dest_stripped_tyapp_typ (mk_tyapp_ty tcref tinst) in 
    
    (* Evaluate the expressions in the original order, but build a record with the results in field order *)
    (* Note some fields may be static. If this were not the case we could just use *)
    (*     let sigma       = Array.map rfref_index ()  *)
    (* However the presence of static fields means rfref_index may index into a non-compact set of instance field indexes. *)
    (* We still need to sort by index. *)
    let rfrefs_array = Array.of_list (list_mapi (fun i x -> (i,x)) rfrefs) in
    Array.sort (fun (_,x) (_,y) -> compare (rfref_index x) (rfref_index y)) rfrefs_array;
    let sigma = Array.create (Array.length rfrefs_array) (-1) in
    Array.iteri (fun j (i,_) -> 
        if sigma.(i) <> -1 then error(InternalError("bad permutation",m));
        sigma.(i) <- j)  rfrefs_array;
    
    let argTyps     = List.map (fun rfref  -> actual_rtyp_of_rfref rfref tinst) rfrefs in
    let names       = List.map name_of_rfref rfrefs in
    let binds,args  = permuteExprList sigma args argTyps names in
    mk_lets_bind m binds (TExpr_op(TOp_recd(lnk,tcref),tinst,args,m))
  
let mk_ldarg0 m ty = mk_asm( [ ldarg_0 ],[],[],[ty],m) 

let mk_refcell     g m ty e = mk_recd(RecdExpr,g.refcell_tcr,[ty],[mk_refcell_contents_rfref g],[e],m)
let mk_refcell_get g m ty e = mk_recd_field_get g (e,mk_refcell_contents_rfref g,[ty],[],m)
let mk_refcell_set g m ty e1 e2 = mk_recd_field_set g (e1,mk_refcell_contents_rfref g,[ty],e2,m)
let mk_lazy_status g m ty e =  mk_recd(RecdExpr,g.lazy_tcr_canon,[ty],[mk_lazy_status_rfref g],[e],m)
let mk_lazy_delayed g m ty e = 
    let delayv,_ = mk_compgen_local m "delay" g.unit_ty in  
    mk_lazy_status g m ty (mk_constr (delayed_ucref g,[ty],[mk_lambda m delayv (e,ty)],m))
let mk_lazystatus_undelayed g m ty e = 
    mk_constr (undelayed_ucref g,[ty],[e],m)
let mk_lazystatus_undefined g m ty = 
    mk_constr (mk_lazy_error_ucref g,[ty],
                  [mk_exnconstr(mk_MFControl_ecref g.fslibCcu "UndefinedException",[],m)],m)

(* throw, rethrow *)
let mk_throw m ty e = mk_asm ([ Il.I_throw ],[], [e],[ty],m)
let mk_rethrow m ty e = mk_asm ([ Il.I_arith Il.AI_pop; Il.I_rethrow ],[], [e],[ty],m)
let dest_throw = function
  | TExpr_op(TOp_asm([Il.I_throw],[ty2]),[],[e],m) -> Some (m,ty2,e)
  | _ -> None
let is_throw x = isSome (dest_throw x)

(* REVIEW: This is a bit over the top. Make this a call to the method in Lazy *)
let mk_lazy_force g m ty lazyv err = 
  let lazye = expr_for_val m lazyv in  
  let statusv,statuse = mk_compgen_local m (name_of_val lazyv^"-status") (mk_lazystatus_ty g ty) in  
  mk_let m statusv (mk_recd_field_get_via_expra(lazye,mk_lazy_status_rfref g,[ty],m)) 
    begin
      let mbuilder = MatchBuilder.create m in 
      let dtree = 
        TDSwitch(statuse,
                [ TCase(TTest_unionconstr(delayed_ucref g,[ty]),
                        (MatchBuilder.add_and_mk_result_target mbuilder 
                           (mk_seq m 
                              (mk_recd_field_set g (lazye,mk_lazy_status_rfref g,[ty],err,m))
                              begin 
                                let resv,rese = mk_compgen_local m "res" ty in  
                                mk_let m resv
                                  (mk_appl((mk_uconstr_field_get(statuse,delayed_ucref g,[ty],0,m), g.unit_ty --> ty), [], [mk_unit g m],m))
                                  (mk_seq m 
                                     (mk_recd_field_set_via_expra(lazye,mk_lazy_status_rfref g,[ty],mk_lazystatus_undelayed g m ty rese,m))
                                     rese)
                              end)));
                  TCase(TTest_unionconstr(undelayed_ucref g,[ty]),
                        (MatchBuilder.add_and_mk_result_target mbuilder 
                           (mk_uconstr_field_get(statuse,undelayed_ucref g,[ty],0,m)))); ],
                Some 
                  (MatchBuilder.add_and_mk_result_target mbuilder 
                      (mk_throw m ty (mk_uconstr_field_get(statuse,mk_lazy_error_ucref g,[ty],0,m)))),
                m) in 
      MatchBuilder.close dtree mbuilder m ty 
    end

(*-------------------------------------------------------------------------
 * List builders
 *------------------------------------------------------------------------- *)
 
let mk_nil g m ty = mk_constr (g.nil_ucref,[ty],[],m)
let mk_cons g ty h t = mk_constr (g.cons_ucref,[ty],[h;t],union_ranges (range_of_expr h) (range_of_expr t))

(*-------------------------------------------------------------------------
 * [gen_range] generates a F# source mark to an IL source mark.
 *------------------------------------------------------------------------- *)

let normalize filename = 
  let n = String.length filename in 
  let res = Buffer.create n in 
  for i = 0 to n-1 do 
    let c = String.get filename i in 
    Buffer.add_char res (match c with '/' -> '\\' | _ -> c);
  done;
  Buffer.contents res
let mk_doc filename = 
 { Il.sourceLanguage=None;
   Il.sourceVendor=None;
   Il.sourceDocType=None;
   Il.sourceFile=filename }
let memoize_file = memoize (file_of_file_idx >> fullpath >> normalize >> mk_doc) 
let gen_range m = 
  Some { Il.sourceDocument=memoize_file (file_idx_of_range m);
         Il.sourceLine=start_line_of_range m;
         Il.sourceColumn= (start_col_of_range m)+1 ; (* .NET && VS  measure first column as column 1 *)
         Il.sourceEndLine= (end_line_of_range m);
         Il.sourceEndColumn=(end_col_of_range m)+1 } 



let mk_compgen_bind g nng nm m e = 
    let locv,loce = mk_compgen_local m (nng.nngApply nm m) (type_of_expr g e) in
    locv,loce,mk_bind locv e 

(*----------------------------------------------------------------------------
 * Make some fragments of code
 *--------------------------------------------------------------------------*)

let box = Il.I_box (Il.mk_tyvar_ty u16_zero)
let isinst = Il.I_isinst (Il.mk_tyvar_ty u16_zero)
let unbox = Il.I_unbox_any (Il.mk_tyvar_ty u16_zero)
let mk_unbox ty e m = mk_asm ([ unbox ], [ty],[e], [ ty ], m)
let mk_isinst ty e m = mk_asm ([ isinst ], [ty],[e], [ ty ], m)

let mspec_Object_GetHashCode     ilg = Il.mk_nongeneric_instance_mspec_in_nongeneric_boxed_tref(ilg.tref_Object,"GetHashCode",[],ilg.typ_int32)
let mspec_Type_GetTypeFromHandle ilg = Il.mk_static_nongeneric_mspec_in_nongeneric_boxed_tref(ilg.tref_Type,"GetTypeFromHandle",[ilg.typ_RuntimeTypeHandle],ilg.typ_Type)
let fspec_Missing_Value  ilg = Il.mk_fspec_in_nongeneric_boxed_tref(ilg.tref_Missing,"Value",ilg.typ_Missing)

let array_get_info           g = Intrinsic(mk_MFLanguagePrimitivesIntrinsicFunctions_nlpath g.fslibCcu, "ArrayGet"       ,([vara]                +-> (mk_array_ty g vara_ty --> (g.int_ty --> vara_ty))))
let poly_hash_info           g = Intrinsic(mk_MFLanguagePrimitivesIntrinsicFunctions_nlpath g.fslibCcu, "StructuralHash" ,([vara]                +-> (vara_ty             --> g.int_ty))  )
let unpickle_quoted_expr     g = Intrinsic(mk_Quotations_Typed_nlpath g,       "Unpickle"      ,([vara;varb;varc;vard] +-> ( g.system_Type_typ  --> (mk_bytearray_ty g --> mk_expr_template_ty g vara_ty varb_ty varc_ty vard_ty))))
let unpickle_quoted_raw_expr g = Intrinsic(mk_Quotations_Raw_nlpath g,         "Unpickle"      ,([vara;varb]           +-> ( g.system_Type_typ  --> (mk_bytearray_ty g --> mk_raw_expr_template_ty g vara_ty varb_ty))))

let typed_expr_for_val_info m (Intrinsic(mvr,nm,ty) as i) =
  let e = vref_for_val_info i in 
  expr_for_vref m e,ty

let mk_call_unbox                g m ty e1    = mk_appl(typed_expr_for_val_info m g.unbox_info,       [[ty]], [ e1 ],  m)
let mk_call_unbox_fast           g m ty e1    = mk_appl(typed_expr_for_val_info m g.unbox_fast_info,  [[ty]], [ e1 ],  m)
let mk_call_istype               g m ty e1    = mk_appl(typed_expr_for_val_info m g.istype_info,      [[ty]], [ e1 ],  m)
let mk_call_istype_fast          g m ty e1    = mk_appl(typed_expr_for_val_info m g.istype_fast_info, [[ty]], [ e1 ],  m)
let mk_call_typeof               g m ty       = mk_appl(typed_expr_for_val_info m g.typeof_info,      [[ty]], [ ],  m)

     
let mk_call_create_instance       g m ty       = mk_appl(typed_expr_for_val_info m g.create_instance_info, [[ty]], [  (mk_unit g m) ],  m)
let mk_call_poly_compare_outer    g m ty e1 e2 = mk_appl(typed_expr_for_val_info m g.poly_compare_outer_info, [[ty]], [  e1;e2 ],  m)
let mk_call_poly_equals_outer     g m ty e1 e2 = mk_appl(typed_expr_for_val_info m g.poly_equals_outer_info, [[ty]], [  e1;e2 ],  m)
let mk_call_poly_hash_param_outer g m ty e1 e2 = mk_appl(typed_expr_for_val_info m g.poly_hash_param_outer_info, [[ty]], [ e1; e2 ], m)
let mk_call_array_get             g m ty e1 e2 = mk_appl(typed_expr_for_val_info m (array_get_info g), [[ty]], [ e1 ; e2 ],  m)
let mk_call_string_to_bigint_vref g m e1       = mk_appl(typed_expr_for_val_info m g.string_to_bigint_info, [], [ e1 ],  m)
let mk_call_string_to_decimal_vref g m e1       = mk_appl(typed_expr_for_val_info m g.string_to_decimal_info, [], [ e1 ],  m)
let mk_call_string_to_bignum_vref g m e1       = mk_appl(typed_expr_for_val_info m g.string_to_bignum_info, [], [ e1 ],  m)

let mk_call_string_compare g m e1 e2 = mk_call_poly_compare_outer g m g.string_ty e1 e2 

let mk_call_new_format                g m aty bty cty dty ety e1    = mk_appl(typed_expr_for_val_info m g.new_format_info, [[aty;bty;cty;dty;ety]], [ e1 ],  m)

let try_elim_bigint_bignum_constants g m c = 
    match c with 
    | TConst_bigint s -> 
        Some(mk_call_string_to_bigint_vref g m (mk_string g m s))

    | TConst_decimal s -> 
        Some(mk_call_string_to_decimal_vref g m (mk_string g m s))
        
    | TConst_bignum s -> 
        Some(mk_call_string_to_bignum_vref g m (mk_string g m s))
    | _ -> 
        None

let mk_seq_ty g ty = mk_tyapp_ty g.seq_tcr [ty] 

let mk_call_seq_map_concat g m alphaTy betaTy arg1 arg2 = 
    (* We're intantiating val map_concat<'a,'sb,'b,'sa> when 'sb :> seq<'b> and 'sa :> seq<'a> : ('a -> 'sb) -> 'sa -> seq<'b> *)
    (* We set 'sa -> typeof(arg2) *)
    (* We set 'sb -> range(typeof(arg1)) *)
    let enumty1 = type_of_expr g arg2 in 
    let enumty2 = try range_of_fun_typ (type_of_expr g arg1) with _ -> (* defensive programming *) (mk_seq_ty g betaTy) in 
    mk_appl(typed_expr_for_val_info m g.seq_map_concat_info, [[alphaTy;enumty2;betaTy;enumty1]], [ arg1; arg2 ],  m) 
                  
let mk_call_seq_using g m ty1 elemTy arg1 arg2 = 
    (* We're intantiating val using : 'a -> ('a -> 'sb) -> seq<'b> when 'sb :> seq<'b> and 'a :> IDisposable *)
    (* We set 'sb -> range(typeof(arg2)) *)
    let enumty = try range_of_fun_typ (type_of_expr g arg2) with _ -> (* defensive programming *) (mk_seq_ty g elemTy) in 
    mk_appl(typed_expr_for_val_info m g.seq_using_info, [[ty1;enumty;elemTy]], [ arg1; arg2 ],  m) 
                  
let mk_call_seq_delay g m elemTy arg1 = 
    (* We're intantiating val using : (unit -> 'sa) -> seq<'a> when 'sa :> seq<'a> *)
    (* We set 'sa -> range(typeof(arg1)) *)
    let enumty = try range_of_fun_typ (type_of_expr g arg1) with _ -> (* defensive programming *) (mk_seq_ty g elemTy) in 
    mk_appl(typed_expr_for_val_info m g.seq_delay_info, [[enumty;elemTy]], [ arg1 ],  m) 
                  
let mk_call_seq_append g m elemTy arg1 arg2 = 
    let ty1 = type_of_expr g arg1 in
    let ty2 = type_of_expr g arg2 in
    mk_appl(typed_expr_for_val_info m g.seq_append_info, [[ty1;elemTy;ty2]], [ arg1; arg2 ],  m) 

let mk_call_seq_generated g m elemTy arg1 arg2 = 
    let enumty = type_of_expr g arg2 in 
    mk_appl(typed_expr_for_val_info m g.seq_generated_info, [[enumty;elemTy]], [ arg1; arg2 ],  m) 
                       
let mk_call_seq_finally g m elemTy arg1 arg2 = 
    let enumty = type_of_expr g arg1 in 
    mk_appl(typed_expr_for_val_info m g.seq_finally_info, [[enumty;elemTy]], [ arg1; arg2 ],  m) 
                       
let mk_call_seq_of_functions g m ty1 ty2 arg1 arg2 arg3 = 
    mk_appl(typed_expr_for_val_info m g.seq_of_functions_info, [[ty1;ty2]], [ arg1; arg2; arg3  ],  m) 
                  
let mk_call_seq_to_array g m elemTy arg1 =  
    let enumty = type_of_expr g arg1 in
    mk_appl(typed_expr_for_val_info m g.seq_to_array_info, [[enumty;elemTy]], [ arg1 ],  m) 
                  
let mk_call_seq_to_list g m elemTy arg1 = 
    let enumty = type_of_expr g arg1 in
    mk_appl(typed_expr_for_val_info m g.seq_to_list_info, [[enumty;elemTy]], [ arg1 ],  m) 
                  
let mk_call_seq_map g m ty1 ty2 arg1 arg2 = 
    let enumty = type_of_expr g arg2 in
    mk_appl(typed_expr_for_val_info m g.seq_map_info, [[ty1;ty2;enumty]], [ arg1; arg2 ],  m) 
                  
let mk_call_seq_singleton g m ty1 arg1 = 
    mk_appl(typed_expr_for_val_info m g.seq_singleton_info, [[ty1]], [ arg1 ],  m) 
                  
let mk_call_seq_empty g m ty1 = 
    mk_appl(typed_expr_for_val_info m g.seq_empty_info, [[ty1]], [ ],  m) 
                 
let mk_call_unpickle_quotation_vref g m tyargs e1 e2 = mk_appl(typed_expr_for_val_info m (unpickle_quoted_expr g), [tyargs], [ e1; e2 ],  m)
let mk_call_unpickle_raw_quotation_vref g m tyargs e1 e2 = mk_appl(typed_expr_for_val_info m (unpickle_quoted_raw_expr g), [tyargs], [ e1; e2 ],  m)


let query_asm e = 
    match strip_expr e with 
    |  TExpr_op(TOp_asm (instrs,_),[],args,_) ->Some(instrs,args)
    | _ -> None 

let dest_incr e = 
    match query_asm e with 
    | Some([ Il.I_arith Il.AI_add ],[TExpr_const(TConst_int32 1l,_,_) ;arg2]) -> Some(arg2)
    | _ -> None 
     
let mk_decr g m e = mk_asm([ Il.I_arith Il.AI_sub  ],[],[e; mk_one g m],[g.int_ty],m)
let mk_incr g m e = mk_asm([ Il.I_arith Il.AI_add  ],[],[mk_one g m; e],[g.int_ty],m)
let mk_ldlen g m arre = mk_asm ([ Il.I_ldlen; Il.I_arith (Il.AI_conv Il.DT_I4) ],[],[ arre ], [ g.int_ty ], m)
  (* The conv.i4 assumes that int_ty is int32. Note: ldlen returns native UNSIGNED int *)
let mk_ceq g m e1 e2 = mk_asm ([ Il.I_arith Il.AI_ceq  ],[],  [e1; e2],[g.bool_ty],m)

let mk_null m ty = TExpr_const(TConst_default, m,ty)



(*----------------------------------------------------------------------------
 * CompilationMappingAttribute, SourceLevelConstruct
 *--------------------------------------------------------------------------*)

let tref_CompilationMappingAttr() = mk_tref(Msilxlib.scoref (), lib_MFCore_name ^ ".CompilationMappingAttribute")
let tref_SourceLevelConstruct ()=  mk_tref(Msilxlib.scoref (), lib_MFCore_name ^ ".SourceLevelConstruct")
let is_CompilationMappingAttr ca = is_il_attrib (tref_CompilationMappingAttr ()) ca
let mk_CompilationMappingAttrPrim g k nums = 
    mk_custom_attribute g.ilg (tref_CompilationMappingAttr(), 
                               ((mk_nongeneric_value_typ (tref_SourceLevelConstruct())) :: (nums |> List.map (fun _ -> g.ilg.typ_Int32))),
                               ((k :: nums) |> List.map (fun n -> CustomElem_int32(Nums.int_to_i32 n))),
                               [])
let mk_CompilationMappingAttr g kind = mk_CompilationMappingAttrPrim g kind []
let mk_CompilationMappingAttrWithSeqNum g kind seqNum = mk_CompilationMappingAttrPrim g kind [seqNum]
let mk_CompilationMappingAttrWithVariantNumAndSeqNum g kind varNum seqNum = mk_CompilationMappingAttrPrim g kind [varNum;seqNum]

(*----------------------------------------------------------------------------
 * FSharpInterfaceDataVersionAttribute
 *--------------------------------------------------------------------------*)

let tref_IntfDataVersionAttr () = mk_tref(Msilxlib.scoref (), lib_MFCore_name ^ ".FSharpInterfaceDataVersionAttribute")
let mk_IntfDataVersionAttr g ((v1,v2,v3,v4) : Il.version_info)  = 
  mk_custom_attribute g.ilg
      (tref_IntfDataVersionAttr(), 
       [g.ilg.typ_Int32;g.ilg.typ_Int32;g.ilg.typ_Int32],
       [CustomElem_int32 (Nums.u16_to_i32 v1);
        CustomElem_int32 (Nums.u16_to_i32 v2) ; 
        CustomElem_int32 (Nums.u16_to_i32 v3)],[])

let is_IntfDataVersionAttr ca = is_il_attrib (tref_IntfDataVersionAttr ()) ca

let is_matching_IntfDataVersionAttr  ((v1,v2,v3,v4) : Il.version_info)  cattr = 
  is_IntfDataVersionAttr cattr &&
  (* ok to use ecma_mscorlib_refs here since we're querying metadata, not making it *)
  begin match decode_cattr_data Il.ecma_mscorlib_refs cattr with 
    [CustomElem_int32 u1; 
      CustomElem_int32 u2;
      CustomElem_int32 u3 ],_ -> v1 = Nums.i32_to_u16 u1 && v2 = Nums.i32_to_u16 u2 && v3 = Nums.i32_to_u16 u3
  | _ -> warning(Failure("unexpected decode of InterfaceDataVersionAttribute")); false
  end

let mk_CompilerGeneratedAttr g n = mk_custom_attribute g.ilg (tref_CompilationMappingAttr(), [mk_nongeneric_value_typ (tref_SourceLevelConstruct())],[CustomElem_int32(Nums.int_to_i32 n)],[])

(* match inp with :? ty as v -> e2[v] | _ -> e3 *)
let mk_isinst_cond g m tgty (vinp,vinpe) v e2 e3 = 
  let mbuilder = MatchBuilder.create m in
  (* REVIEW: save the result here *)
  let tg2 = TDSuccess([mk_unbox tgty vinpe m], MatchBuilder.add_target mbuilder (TTarget([v],e2))) in
  let tg3 = MatchBuilder.add_and_mk_result_target mbuilder e3 in
  let dtree = TDSwitch(vinpe,[TCase(TTest_isinst(type_of_val vinp,tgty),tg2)],Some tg3,m) in
  MatchBuilder.close dtree mbuilder m (type_of_expr g e2)


(*--------------------------------------------------------------------------
!* tupled lambda --> method/function with a given arity specification.
 *
 * adjust_arity_of_lambda_body: "(vs,body)" represents a lambda "fun (vs) ->  body".  The
 * aim is to produce a "static method" represented by a pair
 * "(mvs, body)" where mvs has the length "arity".
 *------------------------------------------------------------------------ *)


let untupled_to_tupled niceNameGen vs =
  let untupled_tys = map type_of_val vs in 
  let m = range_of_val (hd vs) in 
  let tupledv,tuplede = mk_compgen_local m (niceNameGen.nngApply "arg" m) (mk_tuple_ty untupled_tys) in
  let untupling_es =  list_mapi (fun i ty ->  mk_tuple_field_get(tuplede,untupled_tys,i,m)) untupled_tys in 
  tupledv, mk_lets m vs untupling_es 
    
(* The required tupled-arity (arity) can either be 1 *)
(* or N, and likewise for the tuple-arity of the input lambda, i.e. either 1 or N *)
(* where the N's will be identical. *)
let adjust_arity_of_lambda_body g niceNameGen arity vs body = 
  let nvs = length vs in 
  if not (nvs = arity || nvs = 1 || arity = 1) then failwith ("lengths don't add up");
  if length vs = arity then 
    vs,body
  else  if nvs = 1 then
    let v = hd vs in
    let untupled_tys = dest_tuple_typ (type_of_val v) in 
    if  (length untupled_tys <> arity) then failwith "length untupled_tys <> arity";
    let m = range_of_val v in 
    let dummyvs,dummyes = 
      split 
        (list_mapi 
           (fun i ty -> mk_compgen_local m (name_of_val v ^"_"^string_of_int i) ty) 
           untupled_tys) in 
    let body = mk_let m v (mk_tupled g m dummyes untupled_tys) body in 
    dummyvs,body
  else 
    let tupledv, untupler =  untupled_to_tupled niceNameGen vs in 
    [tupledv],untupler body

let multi_lambda_to_tupled_lambda niceNameGen vs body = 
   match vs with 
  | [] -> failwith "multi_lambda_to_tupled_lambda: expected some argments"
  | [v] -> v,body 
  | vs -> 
      let tupledv, untupler =  untupled_to_tupled niceNameGen vs in 
      tupledv, untupler body 
      

(*--------------------------------------------------------------------------
!* Beta reduction via let-bindings. Reduce immediate apps. of lambdas to let bindings. 
 * Includes binding the immediate application of polymorphic 
 * functions. Input type is the type of the function.  Makes use of the invariant
 * that any two expressions have distinct local variables (because we explicitly copy
 * expressions).
 *------------------------------------------------------------------------ *)

let rec beta_mk_appl_aux g niceNameGen (f,fty,tyargsl,argsl,m) =
  (* let verbose = true in *)
  match f with 
  | TExpr_let(bind,body,mlet,_) ->
    (* Lift bindings out, i.e. (let x = e in f) y --> let x = e in f y *)
    (* This increases the scope of 'x', which I don't like as it mucks with debugging *)
    (* scopes of variables, but this is an important optimization, especially when the '|>' *)
    (* notation is used a lot. *)
    (* REVIEW: only apply this when beta-reduction really occurs *)
    if verbose then dprintf0 "--- beta_mk_appl_aux, reducing under let\n";
    mk_let_bind mlet bind (beta_mk_appl_aux g niceNameGen (body,fty,tyargsl,argsl,m))
  | _ -> 
  match tyargsl,argsl with 
  | [] :: rest,_ -> beta_mk_appl_aux g niceNameGen (f,fty,rest,argsl,m)

  | tyargs :: rest,_ -> 
      (* Bind type parameters by immediate substitution *)
      begin match f with 
      | TExpr_tlambda(_, tyvs,body,_,bodyty,_) when length tyvs = length tyargs -> 
          if verbose then dprintf0 "--- beta_mk_appl_aux, binding type parameters\n";
          let tpenv = bind_typars tyvs tyargs empty_tpenv in
          let body' = remark_expr m (inst_expr g tpenv body) in 
          let bodyty' = inst_type tpenv bodyty in 
          if verbose then dprintf0 "--- beta_mk_appl_aux, continuing\n";
          beta_mk_appl_aux g niceNameGen (body',bodyty', rest,argsl,m) 

      | _ -> 
          let f,fty = mk_appl_aux f fty [tyargs] [] m in 
          beta_mk_appl_aux g niceNameGen (f,fty, rest,argsl,m)
      end

  | [], arg :: rest ->
      (* Bind term parameters by "let" explicit substitutions *)
      begin match f with 
      | TExpr_lambda(_,None,argvs,body,_,bodyty,_) -> 
          if verbose then dprintf0 "--- beta_mk_appl_aux, make lambda of right shape\n";
          let argv,body = multi_lambda_to_tupled_lambda niceNameGen argvs body in 
          if verbose then dprintf0 "--- beta_mk_appl_aux, reducing and recursing\n";
          mk_let m argv arg (beta_mk_appl_aux g niceNameGen (remark_expr m body, bodyty, [],rest,m))
      | _ -> 
          if verbose then dprintf2 "--- beta_mk_appl_aux, no reduction, make appl, fty = %s, ty f = %s\n" ((DebugPrint.showType fty)) ((DebugPrint.showType (type_of_expr g f)));
          let f,fty = mk_expr_appl_aux f fty [arg] m in 
          if verbose then dprintf1 "--- beta_mk_appl_aux, no reduction, made appl, continue, fty = %s\n" ((DebugPrint.showType fty));
          beta_mk_appl_aux g niceNameGen (f,fty, [], rest,m)
      end

  | [],[] -> 
      if verbose then dprintf0 "--- beta_mk_appl_aux, done\n";
      f
      
let beta_mk_appl g niceNameGen (f,fty,tyargsl,argl,m) = 
  beta_mk_appl_aux g niceNameGen (f,fty,tyargsl,argl,m)

(*---------------------------------------------------------------------------
 * Adjust for expected usage
 * Convert a use of a value to saturate to the given arity.
 *------------------------------------------------------------------------- *)
 
let mth i xs = if i<length xs then Some (List.nth xs i) else None
let adjust_val_for_expected_arity g m vref flags arity_info =
    
    (* REVIEW: use the arity info here for the names *)
    let tps,argtysl,rty,_ = dest_top_type arity_info (type_of_vref vref) in 
    
    (* Build a lambda expression for the saturated use of the toplevel value... *)
    let expr,exprty = 
      let tps' = copy_typars tps in 
      let tyargs' = map mk_typar_ty tps' in 
      let tpenv = bind_typars tps tyargs' empty_tpenv in
      let rty' = inst_type tpenv rty in 
      let vsl= 
        argtysl |> list_map (fun argtys -> 
          let n = length argtys in
          argtys |> list_map (fun (argty,TopArgData(_,nm)) -> 
            let nm = match nm with None -> nng.nngApply "arg" m | Some id -> nng.nngApply id.idText id.idRange in
            let ty = inst_type tpenv argty in
            fst (mk_local m nm ty))) in 
      let call = beta_mk_appl g nng (TExpr_val(vref,flags,m),type_of_vref vref,[tyargs'],(map (mk_tupled_vars g m) vsl),m) in
      let tauexpr,tauty = 
        fold_right 
          (fun vs (e,ty) -> mk_multi_lambda m vs (e, ty), (mk_tupled_vars_ty g vs --> ty))
          vsl
          (call, rty') in 
      (* Build a type-lambda expression for the toplevel value if needed... *)
      mk_tlambda m tps' (tauexpr,tauty),tps' +-> tauty in 
    expr,exprty
  
(*---------------------------------------------------------------------------
 * linearise_top_match - when only one non-failing target, make linear.  The full
 * complexity of this is only used for spectacularly rare bindings such as 
 *    type ('a,'b) either = This of 'a | That of 'b
 *    let this_f1 = This (fun x -> x)
 *    let This fA | That fA = this_f1
 * 
 * Here a polymorphic top level binding "fA" is _computed_ by a pattern match!!!
 * The TAST coming out of type checking must, however, define fA as a type function,
 * since it is marked with an arity that indicates it's r.h.s. is a type function]
 * without side effects and so can be compiled as a generic method (for example).
 *------------------------------------------------------------------------- *)

(* polymorphic things bound in complex matches at top level require eta expansion of the *)
(* type function to ensure the r.h.s. of the binding is indeed a type function *)
let tlambda_eta m tps (tm,ty) = 
  if isNil tps then tm else mk_tlambda m tps (mk_appl ((tm,ty),[(map mk_typar_ty tps)],[],m),ty)
  
(* For match with only one non-failing target T0, the other targets, T1... failing (say, raise exception).
 *   tree, T0(v0,..,vN) => rhs ; T1() => fail ; ...
 * Convert it to bind T0's variables, then continue with T0's rhs:
 *   let tmp = switch tree, TO(fv0,...,fvN) => Tup (fv0,...,fvN) ; T1() => fail; ... in
 *   let v1  = #1 tmp in ...
 *   and vN  = #N tmp in
 *   rhs
 * Motivation:
 * - For top-level let bindings with possibly failing matches,
 *   this makes clear that subsequent bindings (if reached) are top-level ones.
 *)
let linearise_top_match_aux g (m,tree,targets,m2,ty) =
  let targetsL = Array.to_list targets in
  (* items* package up 0,1,more items *)
  let itemsProj tys i x = 
    match tys with []  -> failwith "itemsProj: no items?"
    | [t] -> x (* no projection needed *)
    | tys -> TExpr_op(TOp_tuple_field_get(i),tys,[x],m) in
  let isThrowingTarget = function TTarget(_,x) -> is_throw x in
  if 1 + count isThrowingTarget targetsL = length targetsL then
    (* Have failing targets and ONE successful one, so linearize *)
    let (TTarget (vs,rhs)) = the (tryfind (isThrowingTarget >> not) targetsL) in
    (* note - old code here used copy value to generate locals - this was not right *)
    let fvs      = map (fun v -> fst(mk_local (range_of_val v) (name_of_val v) (type_of_val v))) vs in (* fresh *)
    let vtys     = map type_of_val vs in
    let tmpTy    = mk_tupled_vars_ty g vs in
    let tmp,tmpe = mk_compgen_local m (nng.nngApply "matchvars" m) tmpTy in
    (* This forces representation as top value, to maintain the invariant from the *)
    (* type checker that anything related to binding top-level values is marked with an *)
    (* arity.  *)
    (data_of_val tmp).val_arity <- Some TopValData.emptyTopValData;  
    let newTg    = TTarget (fvs,mk_tupled_vars g m fvs) in
    let fixup (TTarget (tvs,tx)) = 
       match dest_throw tx with
       | Some (m,ty,e) -> let tx = mk_throw m tmpTy e in
                          TTarget(tvs,tx) (* Throwing targets, recast it's "return type" *)
       | None          -> newTg       (* Non-throwing target,  replaced [new/old] *)
    in
    let targets  = Array.map fixup targets in
    let binds    = 
        vs |> list_mapi (fun i v -> 
            let ty = (type_of_val v) in
            let rhs = 
              tlambda_eta m 
                (fst (try_dest_forall_typ ty)) 
                (itemsProj vtys i tmpe, ty) in
            (* update the arity of the value *)
            (data_of_val v).val_arity <- Some (infer_arity_of_expr ty [] [] rhs);  
            mk_bind v rhs)  in (* vi = proj tmp *)
    mk_let m
      tmp (prim_mk_match (m,tree,targets,m2,tmpTy)) (* note, probably retyped match, but note, result still has same type *)
      (mk_lets_bind m binds rhs)                             
  else
    (* no change *)
    prim_mk_match (m,tree,targets,m2,ty)

let linearise_top_match g = function
  | TExpr_match (m,tree,targets,m2,ty,cache) -> linearise_top_match_aux g (m,tree,targets,m2,ty)
  | x -> x


(*---------------------------------------------------------------------------
 * XMLDoc signatures
 *------------------------------------------------------------------------- *)


let commaEncs strs  = String.concat "," strs
let angleEnc  str   = "{" ^ str ^ "}" 
let tcrefEnc  g tcref = 
     let nm = name_of_tcref tcref in 
     let nm = 
       if not g.typeCheckerConfiguredToAssumeErasureOfGenerics then nm 
       else demangle_dotnet_generic_overloading nm in 
     text_of_path (full_path_to_tcref tcref @ [nm]) 
let lengthEnc xs = string_of_int (length xs) 
let typarEnc g gtps typar = 
    if g.typeCheckerConfiguredToAssumeErasureOfGenerics 
    then "System.Object"
    else
      let idx = 
        try gen_index typar_ref_eq  typar gtps
        with Not_found -> warning(Error("Typar not found during XMLDoc generation", range_of_typar typar)); 0 in 
       "``"^string_of_int idx 

let rec typeEnc  g gtps ty = 
  if verbose then  dprintf0 "--> typeEnc";
  match (strip_tpeqns_and_tcabbrevs ty) with 
  | TType_forall (typars,typ) -> 
      "Microsoft.FSharp.Core.TypeFunc"
  | _ when is_compat_array_typ g ty    -> 
      typeEnc g gtps (List.hd (tinst_of_stripped_typ ty))^ "[]"
  | _ when is_il_arr_typ g ty   -> 
      typeEnc g gtps (List.hd (tinst_of_stripped_typ ty))^ name_of_tcref (tcref_of_stripped_typ ty)
  | TType_app (tcref,tinst)   -> 
      tcrefEnc g tcref ^ tyargsEnc g gtps tinst
  | TType_tuple typs          -> 
      if g.typeCheckerConfiguredToAssumeErasureOfGenerics 
      then "Microsoft.FSharp.Core.Tuple" ^ lengthEnc typs 
      else "Microsoft.FSharp.Core.Tuple`" ^ lengthEnc typs ^ tyargsEnc g gtps typs
  | TType_fun (f,x)           -> 
      "Microsoft.FSharp.Core.FastFunc" ^ (if g.typeCheckerConfiguredToAssumeErasureOfGenerics then "" else "`2")^ tyargsEnc g gtps [f;x]
  | TType_var typar           -> 
      typarEnc g gtps (deref_local typar)
  | TType_unknown | TType_modul_bindings  -> 
      "System.Object"
and tyargsEnc  g gtps args = 
     if g.typeCheckerConfiguredToAssumeErasureOfGenerics || isNil args then ""
     else angleEnc (commaEncs (map (typeEnc g gtps) args)) 

let argsEnc g gtps argTs =
  if isNil argTs then "" 
  else "(" ^ String.concat "," (List.map (typeEnc g gtps) argTs) ^ ")"

let xmlDocSigOfVal g path v =
  let tps,methTypars,argTs,prefix,path,name = 
    match member_info_of_val v with 
    | None -> 
        (* Regular F# values *)
        let w = arity2_of_val v in 
        let tps,argTs,_,_ = dest_top_type w (type_of_val v) in
        let name = compiled_name_of_val v in 
        let prefix =
          if  TopValData.numCurriedArgs w = 0 && isNil tps then "P:"
          else "M:" in
        tps,tps,argTs,prefix,path,name

    | Some membInfo -> 
        (* Methods, Properties etc. *)
        let tps,argTs,rtnT,_ = dest_vspr_typ membInfo (the (arity_of_val v)) (type_of_val v) in 
        let prefix,name = 
          match membInfo.vspr_flags.memFlagsKind with 
          | MemberKindClassConstructor 
          | MemberKindConstructor 
          | MemberKindMember -> "M:", compiled_name_of_val v
          | MemberKindPropertyGetSet 
          | MemberKindPropertySet
          | MemberKindPropertyGet -> "P:",get_property_name membInfo in 
        let path = 
          path^"."^ name_of_tcref (actual_parent_of_vspr_val v) in
        let methTypars = 
          match partition_val_typars v with
          | Some(_,_,memberMethodTypars,_,_) ->  memberMethodTypars
          | None -> tps in 
        tps,methTypars,[argTs],prefix,path,name in 
  let args = argsEnc g tps (argTs |> List.concat |> map fst |> filter (is_unit_typ g >> not)) in
  let arity = List.length methTypars in (* C# XML doc adds ``<arity> to *generic* member names *)
  let genArity = if g.typeCheckerConfiguredToAssumeErasureOfGenerics || arity=0 then "" else Printf.sprintf "``%d" arity in
  prefix ^ path ^ "." ^ name ^ genArity ^ args

let xmlDocSigOfTycon (g:tcGlobals) path tc = "T:" ^ path ^ "." ^ name_of_tycon tc 
let xmlDocSigOfSubModul (g:tcGlobals) path = "T:" ^ path 

(*---------------------------------------------------------------------------
 * From TAST tycon_ref to IL type_ref
 *------------------------------------------------------------------------- *)

let boxity_of_tycon tycon = if is_struct_tycon tycon then AsValue else AsObject

let il_tref_for_cpath (CompPath(sref,p)) item = 
  let rec top racc  p = 
    match p with 
    | [] -> Il.mk_tref (sref,text_of_path  (List.rev (item::racc)))
    | (h,istype)::t -> 
      match istype with 
      | AsMangledNamedType _ | AsNamedType -> 
        let outerTypeName = (text_of_path (List.rev ((adjust_module_name istype h)::racc))) in 
        Il.mk_nested_tref (sref, (outerTypeName :: List.map (fun (nm,istype) -> adjust_module_name istype nm) t),item)
      | _ -> 
        top (h::racc) t in 
  top [] p 


let gen_il_type_repr cpath n boxity = 
  TyrepNamed (il_tref_for_cpath cpath n, boxity)

let rec il_repr_of_tcref tycon =
   let tycon = deref_tycon tycon in 
   assert(abbrev_of_tycon tycon = None);
   cached (il_repr_cache_of_tycon tycon) (fun () -> 
      match exn_repr_of_tycon tycon with 
      | TExnAbbrevRepr ecref2 -> il_repr_of_tcref ecref2
      | TExnAsmRepr tref -> TyrepNamed(tref,AsObject)
      | _ -> 
      match repr_of_tycon tycon with 
      | Some (TAsmRepr typ) -> TyrepOpen typ
      | _ -> gen_il_type_repr (cpath_of_tycon tycon)  (name_of_tycon tycon) (boxity_of_tycon tycon))


(*--------------------------------------------------------------------------
!* Some unions have null as representations 
 *------------------------------------------------------------------------*)


let enum_CompilationRepresentationAttribute_Static = 1l
let enum_CompilationRepresentationAttribute_Instance = 2l
let enum_CompilationRepresentationAttribute_StaticInstanceMask = 3l
let enum_CompilationRepresentationAttribute_ModuleSuffix = 4l
let enum_CompilationRepresentationAttribute_PermitNull = 8l

let tycon_permitNull g tycon =
     match fsthing_int32_attrib  g g.attrib_CompilationRepresentationAttribute (attribs_of_tycon tycon) with
     | Some(flags) -> ((flags &&& enum_CompilationRepresentationAttribute_PermitNull) <> 0l)
     | _ -> false 

(* WARNING: this must match optimizeAlternativeToNull in ilx/cu_erase.ml *) 
(* REVIEW: make this fully attribute controlled *)
let isUnionThatUsesNullAsRepresentation (g:tcGlobals) tycon =
  (is_union_tycon tycon && 
   let ucs = uconstrs_array_of_tycon tycon in 
   (Array.length ucs = 0 ||
     (tycon_permitNull g tycon &&
      array_exists_one is_nullary_of_uconstr ucs &&
      array_exists (is_nullary_of_uconstr >> not) ucs)))

let tyconCompilesInstanceMembersAsStatic g tycon = isUnionThatUsesNullAsRepresentation g tycon
let tcrefCompilesInstanceMembersAsStatic g tcref = tyconCompilesInstanceMembersAsStatic g (deref_tycon tcref)

let typNullIsExtraValue g ty = 
        is_il_ref_typ g ty 

let typNullIsTrueValue g ty = 
       (is_stripped_tyapp_typ ty && isUnionThatUsesNullAsRepresentation g (deref_tycon (tcref_of_stripped_typ ty)))  
    || (is_unit_typ g ty)

let typNullNever g ty = 
       (is_struct_typ ty)
    || (is_byref_ty g ty)

let typNullNotLiked g ty = 
       not (typNullIsExtraValue g ty) 
    && not (typNullIsTrueValue g ty) 
    && not (typNullNever g ty) 

let typSatisfiesNullConstraint g ty = 
       typNullIsExtraValue g ty  
    || typNullIsTrueValue g ty   

(* Can we use the fast helper for the 'LanguagePrimitives.IntrinsicFunctions.TypeTestGeneric'? *)
let can_use_istype_fast g ty = 
     not (is_typar_ty ty) && 
     not (typNullIsTrueValue g ty) && 
     not (typNullNever g ty)

(* Can we use the fast helper for the 'LanguagePrimitives.IntrinsicFunctions.UnboxGeneric'? *)
let can_use_unbox_fast g ty = 
     not (is_typar_ty ty) && 
     not (typNullNotLiked g ty)
     
     
(*--------------------------------------------------------------------------
!* Nullness tests and pokes 
 *------------------------------------------------------------------------*)

let mk_nonnull_test g m e = mk_asm ([ Il.I_arith Il.AI_ldnull ; Il.I_arith Il.AI_cgt_un  ],[],  [e],[g.bool_ty],m)
let mk_nonnull_cond g m ty e1 e2 e3 = mk_cond m ty (mk_nonnull_test g m e1) e2 e3
let mk_nonnull_poke g m e = mk_asm ([ Il.I_arith Il.AI_dup ; Il.I_ldvirtftn (mspec_Object_GetHashCode g.ilg); Il.I_arith Il.AI_pop  ],[],  [e],[type_of_expr g e],m)


let moduleNameIsMangled g attrs =
     match fsthing_int32_attrib g g.attrib_CompilationRepresentationAttribute attrs with
     | Some(flags) -> ((flags &&& enum_CompilationRepresentationAttribute_ModuleSuffix) <> 0l)
     | _ -> false 


let vsprCompiledAsInstance g parent isExtensionMember membInfo attrs =
    let explicitInstance,explicitStatic = 
       match fsthing_int32_attrib g g.attrib_CompilationRepresentationAttribute attrs with
       | Some(flags) -> 
         ((flags &&& enum_CompilationRepresentationAttribute_Instance) <> 0l),
         ((flags &&& enum_CompilationRepresentationAttribute_Static) <> 0l)
       | _ -> false,false in 

    not isExtensionMember &&
    (explicitInstance ||
     isSome membInfo.vspr_implements_slotsig ||
     (membInfo.vspr_flags.memFlagsInstance &&
      not explicitStatic &&
      not (tcrefCompilesInstanceMembersAsStatic g parent)))


let is_sealed_typ g ty =
  not (is_ref_typ g ty) ||
  is_any_array_typ g ty || 
  (if is_il_named_typ ty then 
    let tcr,tinst = dest_stripped_tyapp_typ ty in 
    let _,_,tdef = dest_il_tcref tcr in 
    tdef.tdSealed
  else if (is_fsobjmodel_interface_typ ty || is_fsobjmodel_class_typ ty) then 
    let tcr,tinst = dest_stripped_tyapp_typ ty in 
    (fsthing_bool_attrib g g.attrib_SealedAttribute (attribs_of_tcref tcr) = Some(true))
  else true)
  
let valCompiledAsInstance g v =
    match member_info_of_val v with 
    | Some(membInfo) -> 
        (* Note it doesn't matter if we pass 'actual_parent_of_vspr_val' or 'apparent_parent_of_vspr_val' here. *)
        (* These only differ if the value is an extension member, and in that case vsprCompiledAsInstance always returns *)
        (* false anyway *)
        vsprCompiledAsInstance g (apparent_parent_of_vspr_val v)  (isext_of_val v) membInfo (attribs_of_val v)  
    |  _ -> false

let vrefCompiledAsInstance g vref = valCompiledAsInstance g (deref_val vref)


(*---------------------------------------------------------------------------
 * Crack information about an F# object model call
 *------------------------------------------------------------------------- *)

let get_member_call_info g (vref,vFlags) = 
    match member_info_of_vref vref with 
    | Some(membInfo) when not (isext_of_vref vref) -> 
      let numEnclTypeArgs = length(typars_of_tcref (apparent_parent_of_vspr_vref vref)) in
      let virtualCall = 
        (membInfo.vspr_flags.memFlagsVirtual || 
         membInfo.vspr_flags.memFlagsOverride || 
         membInfo.vspr_flags.memFlagsAbstract) && 
         not membInfo.vspr_flags.memFlagsFinal && 
         not (vFlags = VSlotDirectCall) in 
      let newobj    = (membInfo.vspr_flags.memFlagsKind = MemberKindConstructor) && (vFlags = NormalValUse) in
      let superInit = (membInfo.vspr_flags.memFlagsKind = MemberKindConstructor) && (vFlags = CtorValUsedAsSuperInit) in
      let selfInit  = (membInfo.vspr_flags.memFlagsKind = MemberKindConstructor) && (vFlags = CtorValUsedAsSelfInit) in
      let instance = vrefCompiledAsInstance g vref in 
      let isPropGet = (membInfo.vspr_flags.memFlagsKind = MemberKindPropertyGet) in
      let propset = (membInfo.vspr_flags.memFlagsKind = MemberKindPropertyGet) in
      numEnclTypeArgs, virtualCall,newobj,superInit,selfInit ,instance,isPropGet,propset
    | _ -> 0,false,false,false,false,false,false,false

(*---------------------------------------------------------------------------
 * Active pattern name helpers
 *------------------------------------------------------------------------- *)

let core_display_name_of_vref vref = core_display_name_of_val (deref_val vref)

let apinfo_of_vname nm = 
    let rec loop nm = 
        if String.contains nm '|' then 
           let n = String.index nm '|' in
           String.sub nm 0 n :: loop (String.sub nm (n+1) (String.length nm - n - 1))
        else
           [nm] in
    let len = String.length nm in 
    let nm = decompileOpName nm in
    if String.contains nm '|' &&
       (String.index nm '|' = 0) &&
       len >= 3 &&
       (String.rindex nm '|' = len - 1)  then (
        let res = loop (String.sub nm 1 (len - 2)) in
        let resH,resT = frontAndBack res in 
        Some(if resT = "_" then APInfo(false,resH) else APInfo(true,res))
        (* dprintf2 "apinfo_of_vname %s, res = %s\n" nm (String.concat ";" res);*)
    ) else None

let apinfo_of_vref vref = apinfo_of_vname (core_display_name_of_vref vref)
let name_of_apref (APElemRef(_,vref,n)) =
    match apinfo_of_vref vref with
    | None -> error(InternalError("name_of_apref: not an active pattern name", range_of_vref vref))
    | Some (APInfo(total,nms)) -> 
        if n < 0 || n >= List.length nms  then error(InternalError("name_of_apref: index out of range for active pattern refernce", range_of_vref vref));
        List.nth nms n

let mk_choices_tcref g n = 
     match n with 
     | 0 | 1 -> failwith "mk_choices_tcref"
     | 2 -> g.choice2_tcr
     | 3 -> g.choice3_tcr
     | 4 -> g.choice4_tcr
     | 5 -> g.choice5_tcr
     | 6 -> g.choice6_tcr
     | 7 -> g.choice7_tcr
     | _ -> failwith "active patterns returning more than 7 possibilities are not yet implemented"
let mk_choices_typ g tinst = 
     match length tinst with 
     | 0 -> g.unit_ty
     | 1 -> List.hd tinst
     | _ -> mk_tyapp_ty (mk_choices_tcref g (length tinst)) tinst

let mk_choices_ucref g n i = 
     mk_ucref (mk_choices_tcref g n) ("Choice"^string_of_int n^"_"^string_of_int (i+1))

let names_of_apinfo (APInfo(_,nms)) = nms
let total_of_apinfo (APInfo(total,_)) = total

let mk_apinfo_result_typ g apinfo rtys = 
    let choicety = mk_choices_typ g rtys in 
    if total_of_apinfo apinfo then choicety else mk_option_ty g choicety
    
let mk_apinfo_typ g apinfo dty rtys = mk_fun_ty dty (mk_apinfo_result_typ g apinfo rtys)
    

(*---------------------------------------------------------------------------
!* rewrite_expr: rewrite bottom up with interceptors 
 *-------------------------------------------------------------------------*)

type rwEnv = {pre_intercept: ((expr -> expr) -> expr -> expr option) option;
              post_transform: expr -> expr option;
              underQuotations: bool }    

let rec rewrite_bind env (TBind(v,e) as bind) = TBind(v,rewrite_expr env e) 

and rewrite_binds env binds = map (rewrite_bind env) binds

and rewrite_expr env expr =
  match expr with 
  | TExpr_let _ ->
      rewrite_linear_expr env expr (fun e -> e)
  | _ -> 
      let expr = 
         match pre_rewrite_expr env expr with 
         | Some expr -> expr
         | None -> rewrite_expr_structure env expr in
      post_rewrite_expr env expr 

and pre_rewrite_expr env expr = 
     match env.pre_intercept  with 
     | Some f -> f (rewrite_expr env) expr
     | None -> None 
and post_rewrite_expr env expr = 
     match env.post_transform expr with 
     | None -> expr 
     | Some expr -> expr 

and rewrite_expr_structure env expr =  
  match expr with
  | TExpr_hole _ 
  | TExpr_const _ 
  | TExpr_val _ -> expr
  | TExpr_app(f0,f0ty,tyargsl,argsl,m) -> 
      let f0'   = rewrite_expr env f0 in
      let args' = rewrite_exprs env argsl in
      TExpr_app(f0',f0ty,tyargsl,args',m)
  | TExpr_quote(raw,ast,m,ty) -> 
      TExpr_quote(raw,(if env.underQuotations then rewrite_expr env ast else ast),m,ty)
  | TExpr_obj (_,ty,basev,basecall,overrides,iimpls,m,_) -> 
      TExpr_obj(new_uniq(),ty,basev,rewrite_expr env basecall,map (rewrite_override env) overrides,
                map (rewrite_iimpl env) iimpls,m,new_cache())
  | TExpr_link eref -> 
      rewrite_expr env !eref
  | TExpr_op (c,tyargs,args,m) -> 
      TExpr_op(c,tyargs,rewrite_exprs env args,m)
  | TExpr_lambda(lambda_id,basevopt,argvs,body,m,rty,_) -> 
      let body' = rewrite_expr env body in 
      mk_basev_multi_lambda m basevopt argvs (body',rty)
  | TExpr_tlambda(lambda_id,argtyvs,body,m,rty,_) -> 
      let body' = rewrite_expr env body in 
      mk_tlambda m argtyvs (body',rty)
  | TExpr_match(exprm,dtree,targets,m,ty,_) -> 
      let dtree' = rewrite_dtree env dtree in
      let targets' = rewrite_targets env targets in
      mk_and_optimize_match exprm m ty dtree' targets'
  | TExpr_letrec (binds,e,m,_) ->
      let binds = rewrite_binds env binds in
      let e' = rewrite_expr env e in 
      TExpr_letrec(binds,e',m,new_cache())
  | TExpr_let _ -> failwith "unreachable - linear"
  | TExpr_seq (e1,e2,dir,m) -> 
      let e1' = rewrite_expr env e1 in      
      let e2' = rewrite_expr env e2 in      
      TExpr_seq(e1',e2',dir,m)
  | TExpr_static_optimization (constraints,e2,e3,m) ->
      let e2' = rewrite_expr env e2 in      
      let e3' = rewrite_expr env e3 in      
      TExpr_static_optimization(constraints,e2',e3',m)
  | TExpr_tchoose (a,b,m) -> TExpr_tchoose(a,rewrite_expr env b,m)
and rewrite_linear_expr env expr contf =
    (* schedule a rewrite on the way back up by adding to the continuation *)
    let contf = contf << post_rewrite_expr env in 
    match pre_rewrite_expr env expr with 
    | Some expr -> contf expr  (* done - intercepted! *)
    | None -> 
        match expr with 
        | TExpr_let (bind,body,m,_) ->  
            let bind = rewrite_bind env bind in 
            rewrite_linear_expr env body (contf << mk_let_bind m bind) 
        | _ -> 
            (* no longer linear *)
            contf (rewrite_expr env expr) 

and rewrite_exprs env exprs = list_map (rewrite_expr env) exprs

and rewrite_dtree env x =
  match x with 
  | TDSuccess (es,n) -> TDSuccess(rewrite_exprs env es,n)
  | TDSwitch (e,cases,dflt,m) ->
      let e' = rewrite_expr env e in 
      let cases' = map (fun (TCase(discrim,e)) -> TCase(discrim,rewrite_dtree env e)) cases in 
      let dflt' = Option.map (rewrite_dtree env) dflt in 
      TDSwitch (e',cases',dflt',m)
  | TDBind (bind,body) ->
      let bind' = rewrite_bind env bind in 
      let body' = rewrite_dtree env body in 
      TDBind (bind',body')
and rewrite_target env (TTarget(vs,e)) = TTarget(vs,rewrite_expr env e)
and rewrite_targets env targets = map (rewrite_target env) (Array.to_list targets)

and rewrite_override env (TMethod(slotsig,tps,vs,e,m)) =
  TMethod(slotsig,tps,vs,rewrite_expr env e,m)
and rewrite_iimpl env (ty,overrides) = 
  (ty, map (rewrite_override env) overrides)
    
and rewrite_mexpr env x = 
    match x with  
    (* | TMTyped(mty,e,m) -> TMTyped(mty,rewrite_mexpr env e,m) *)
    | TMTyped(mty,def,m) ->  TMTyped(mty,rewrite_mdef env def,m)
and rewrite_mdefs env x = map (rewrite_mdef env) x
    
and rewrite_mdef env x = 
    match x with 
    | TMDefRec(tycons,binds,m) -> TMDefRec(tycons,rewrite_binds env binds,m)
    | TMDefLet(bind,m)         -> TMDefLet(rewrite_bind env bind,m)
    | TMDefs(defs)             -> TMDefs(rewrite_mdefs env defs)
    | TMAbstract(mexpr)        -> TMAbstract(rewrite_mexpr env mexpr)
    | TMDefModul(TMBind(nm, rhs)) -> TMDefModul(TMBind(nm,rewrite_mdef env rhs))

and rewrite_assembly env (TAssembly(mvs)) = TAssembly(map (mapTImplFile (rewrite_mexpr env)) mvs)


let is_flag_enum_typ (g:tcGlobals) typ = 
   (is_enum_typ typ (* && tcref_has_attrib g g.attrib_FlagsAttribute (tcref_of_stripped_typ typ) *) ) 

(*--------------------------------------------------------------------------
!* Build a mrpi that converts all "local" references to "public" things 
 * to be non local references.
 *------------------------------------------------------------------------ *)

let mk_export_remapping viewedCcu = 
    let acc_tycon_remap tycon tmenv = 
        match pubpath_of_tycon tycon with 
        | Some pubpath -> 
            if !DebugPrint.layout_stamps then dprintf2 "adding export remapping for tycon '%s#%d'\n" (name_of_tycon tycon) (stamp_of_tycon tycon);
                let tcref = 
                    if tycon_is_modul tycon then 
                         rescope_module_pubpath viewedCcu pubpath tycon 
                     else 
                         rescope_tycon_pubpath viewedCcu pubpath tycon in
            tmenv_add_tcref_remap (mk_local_tcref tycon) tcref tmenv
        | None -> error(Error("unexpected tycon without a pubpath when remapping assembly data",range_of_tycon tycon)) in 
    let acc_val_remap vspec tmenv = 
        match pubpath_of_val vspec with 
        | Some pubpath -> 
            if !DebugPrint.layout_stamps then dprintf1 "adding export remapping for value #%d\n" (stamp_of_val vspec);
            {tmenv with vspec_remap=vspec_map_add vspec (rescope_val_pubpath viewedCcu pubpath vspec) tmenv.vspec_remap}
        | None -> error(Error("unexpected value without a pubpath when remapping assembly data",range_of_val vspec)) in 
    fun mspec -> fold_vals_and_tycons_of_mtyp acc_tycon_remap acc_val_remap (mtyp_of_modul mspec) empty_expr_remap

(*--------------------------------------------------------------------------
!* Apply a "local to nonlocal" renaming to a module type.  This can't use
 * remap_mspec since the remapping we want isn't to newly created nodes
 * but rather to remap to the nonlocal references. This is deliberately 
 * "breaking" the binding structure implicit in the module type, which is
 * the whole point - one things are rewritten to use non local references then
 * the elements can be copied at will, e.g. when inlining during optimization.
 *------------------------------------------------------------------------ *)


let rec remap_tycon_data_to_nonlocal g tmenv d = 
    let tps',tmenvinner = tmenv_copy_remap_and_bind_typars tmenv d.tycon_typars in

    { d with 
          tycon_typars         = tps';
          tycon_attribs        = d.tycon_attribs        |> remap_attribs g tmenvinner;
          tycon_repr           = d.tycon_repr           |> Option.map (remap_tycon_repr g tmenvinner);
          tycon_abbrev         = d.tycon_abbrev         |> Option.map (remap_type tmenvinner.tyenv) ;
          tycon_tcaug          = d.tycon_tcaug          |> remap_tcaug tmenvinner ;
          tycon_modul_contents = 
              notlazy (d.tycon_modul_contents 
                       |> Lazy.force 
                       |> map_immediate_vals_and_tycons_of_modtyp (remap_tycon_to_nonlocal g tmenv) 
                                                                  (remap_val_to_nonlocal g tmenv));
          tycon_exnc_info      = d.tycon_exnc_info      |> remap_tycon_exnc_info g tmenvinner}

and remap_tycon_to_nonlocal g tmenv x = 
    x |> new_tycon_modified (remap_tycon_data_to_nonlocal g tmenv)  

and remap_val_to_nonlocal g  tmenv inp = 
    inp |> new_vspec_modified (remap_val_data g tmenv)

let apply_export_remapping_to_mspec g tmenv x = remap_tycon_to_nonlocal g tmenv x

(* Which constraints actually get compiled to .NET constraints? *)
let is_compiled_constraint cx = 
    match cx with 
      | TTyparIsNotNullableValueType _ 
      | TTyparIsReferenceType _
      | TTyparRequiresDefaultConstructor _
      | TTyparCoercesToType _ -> true
      | _ -> false
    
(* Is a value a first-class polymorphic value with .NET constraints? *)
let is_poly_constrained_val v = 
    let ty = (type_of_val v) in 
    is_forall_ty ty && 
    ty |> dest_forall_typ |> fst |> List.exists (constraints_of_typar >> List.exists is_compiled_constraint)

(* Does a type support a given interface? *)
let tcaug_has_interface g tcaug ty = 
    gen_exists (fun (x,_,_) -> type_equiv g ty x)  tcaug.tcaug_implements

(* Does a type have an override matching the given name and argument types? *)
(* Used to detet the presence of 'Equals' and 'GetHashCode' in type checking *)
let tcaug_has_override g tcaug nm argtys = 
  tcaug.tcaug_adhoc 
  |> Namemap.find_multi nm
  |> gen_exists (fun vref -> 
                    match member_info_of_vref vref with 
                    | None -> false 
                    | Some membInfo -> let ptys = map fst (arginfos_of_member_vref g vref) in 
                                   length ptys = length argtys &&
                                   List.for_all2 (type_equiv g) ptys argtys  &&  
                                   membInfo.vspr_flags.memFlagsOverride) 

let mk_fast_for_loop g m idv start' dir finish' body' =
    let mk_bump = if dir then mk_incr else mk_decr in
    let startv,starte   = mk_compgen_local (range_of_expr start')  (nng.nngApply (name_of_val idv ^ "_start") m) g.int_ty in  
    let finishv,finishe = mk_compgen_local (range_of_expr finish') (nng.nngApply (name_of_val idv ^ "_end")   m) g.int_ty in
    (* use "finish+1" as upper limit - store in the loop expression so expression quotation can reverse-this mapping *)
    let finishe = mk_bump g (range_of_expr finishe) finishe in 
    mk_let (range_of_expr start')  startv  start' (
    mk_let (range_of_expr finish') finishv finish' (  
    mk_for g (idv,starte,dir,finishe,body',m)))
