package org.springframework.uaa.client;

import java.io.IOException;
import java.net.Authenticator;
import java.net.HttpURLConnection;
import java.net.Proxy;
import java.net.URL;

/**
 * Allows customization of HTTP proxy configuration that may be used by {@link TransmissionService}
 * and other similar classes that require proxy services.
 * 
 * @author Ben Alex
 * @author Christian Dupuis
 * @since 1.0.1
 *
 */
public interface ProxyService {

	/**
	 * Prepares a {@link HttpURLConnection} for the given URL, including sensible settings for timeouts and
	 * the proxy configuration correctly. This method is the most important offered by {@link ProxyService},
	 * with the others primarily for diagnostic purposes or very specialised requirements.
	 * 
	 * @param url to connect (required)
	 * @return a proxy and timeout-aware connection object (never null)
	 * @throws IOException if there were setup difficulties
	 */
	HttpURLConnection prepareHttpUrlConnection(URL url) throws IOException;

	/**
	 * Setup HTTP proxy for the given {@link URL}.
	 *  
	 * @param uaaUrl the {@link URL} to access to through the proxy
	 * @return proxy instance to use for the given {@link URL}
	 */
	Proxy setupProxy(URL url);
	
	/**
	 * Optionally provides an {@link Authenticator} for the given URL and proxy.
	 * 
	 * @param url the {@link URL} to access to through the proxy
	 * @param proxy the proxy that will be used to connect to <code>url</code> (can be <code>null</code>)
	 * @return authenticator to use to connect to the proxy, or null if no authenticator is needed
	 */
	Authenticator setupProxyAuthentication(URL url, Proxy proxy);

}
