package org.springframework.uaa.client.internal;

import java.io.IOException;
import java.net.Authenticator;
import java.net.HttpURLConnection;
import java.net.Proxy;
import java.net.ProxySelector;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.List;

import org.springframework.uaa.client.ProxyService;

/**
 * Basic implementation of {@link ProxyService} that relies on the JDK proxy
 * selector, in particular the <code>http.proxyHost</code> and
 * <code>http.proxyPort</code> system properties.
 * 
 * <p>
 * Note it is also possible to set these in /etc/java-6-openjdk/net.properties
 * (or equivalent), including <code>java.net.useSystemProxies=true</code> to
 * delegate to the settings defined by the host operating system. Learn more
 * about JDK proxy configuration in the <a href=
 * "http://download.oracle.com/javase/6/docs/technotes/guides/net/proxies.html"
 * >proxy configuration article</a>.
 * 
 * @author Ben Alex
 * @author Christian Dupuis
 * @since 1.0.1
 * 
 */
public class BasicProxyService implements ProxyService {

	public Proxy setupProxy(URL url) {
		List<Proxy> proxies = null;
		try {
			proxies = ProxySelector.getDefault().select(url.toURI());
		} catch (URISyntaxException e) {
			// this can't happen
		}

		if (proxies != null) {
			for (Proxy proxy : proxies) {
				if (proxy.type().equals(Proxy.Type.HTTP)) {
					return proxy;
				}
			}
		}
		return null;
	}

	public Authenticator setupProxyAuthentication(URL url, Proxy proxy) {
		return null;
	}

	public HttpURLConnection prepareHttpUrlConnection(URL url) throws IOException {
		// Prepare proxy and proxy authentication
		Proxy proxy = setupProxy(url);
		Authenticator proxyAuthentication = setupProxyAuthentication(url, proxy);
		if (proxyAuthentication != null) {
			Authenticator.setDefault(proxyAuthentication);
		}
		
		HttpURLConnection connection = (HttpURLConnection) (proxy != null ?  url.openConnection(proxy) : url.openConnection());
		connection.setConnectTimeout(5000);
		connection.setReadTimeout(5000);
		connection.setUseCaches(false);
		return connection;
	}

}
