/*****************************************************************************
 * Copyright (c) 2014-2020 OpenRCT2 developers
 *
 * For a complete list of all authors, please refer to contributors.md
 * Interested in contributing? Visit https://github.com/OpenRCT2/OpenRCT2
 *
 * OpenRCT2 is licensed under the GNU General Public License version 3.
 *****************************************************************************/

#include "SmallSceneryRemoveAction.h"

#include "../Cheats.h"
#include "../OpenRCT2.h"
#include "../common.h"
#include "../core/MemoryStream.h"
#include "../interface/Window.h"
#include "../localisation/Localisation.h"
#include "../localisation/StringIds.h"
#include "../management/Finance.h"
#include "../ride/Ride.h"
#include "../world/Park.h"
#include "../world/SmallScenery.h"
#include "../world/TileElementsView.h"
#include "GameAction.h"
#include "SmallSceneryPlaceAction.h"

using namespace OpenRCT2;

SmallSceneryRemoveAction::SmallSceneryRemoveAction(const CoordsXYZ& location, uint8_t quadrant, ObjectEntryIndex sceneryType)
    : _loc(location)
    , _quadrant(quadrant)
    , _sceneryType(sceneryType)
{
}

void SmallSceneryRemoveAction::AcceptParameters(GameActionParameterVisitor& visitor)
{
    visitor.Visit(_loc);
    visitor.Visit("object", _sceneryType);
    visitor.Visit("quadrant", _quadrant);
}

uint16_t SmallSceneryRemoveAction::GetActionFlags() const
{
    return GameAction::GetActionFlags();
}

void SmallSceneryRemoveAction::Serialise(DataSerialiser& stream)
{
    GameAction::Serialise(stream);

    stream << DS_TAG(_loc) << DS_TAG(_quadrant) << DS_TAG(_sceneryType);
}

GameActions::Result::Ptr SmallSceneryRemoveAction::Query() const
{
    GameActions::Result::Ptr res = std::make_unique<GameActions::Result>();

    if (!LocationValid(_loc))
    {
        return MakeResult(GameActions::Status::InvalidParameters, STR_CANT_REMOVE_THIS, STR_LAND_NOT_OWNED_BY_PARK);
    }

    auto* entry = get_small_scenery_entry(_sceneryType);
    if (entry == nullptr)
    {
        return MakeResult(GameActions::Status::InvalidParameters, STR_CANT_REMOVE_THIS, STR_INVALID_SELECTION_OF_OBJECTS);
    }

    res->Cost = entry->removal_price * 10;
    res->Expenditure = ExpenditureType::Landscaping;
    res->Position = _loc;

    if (!(gScreenFlags & SCREEN_FLAGS_SCENARIO_EDITOR) && !(GetFlags() & GAME_COMMAND_FLAG_GHOST) && !gCheatsSandboxMode)
    {
        // Check if allowed to remove item
        if (gParkFlags & PARK_FLAGS_FORBID_TREE_REMOVAL)
        {
            if (scenery_small_entry_has_flag(entry, SMALL_SCENERY_FLAG_IS_TREE))
            {
                res->Error = GameActions::Status::NoClearance;
                res->ErrorTitle = STR_CANT_REMOVE_THIS;
                res->ErrorMessage = STR_FORBIDDEN_BY_THE_LOCAL_AUTHORITY;
                return res;
            }
        }

        // Check if the land is owned
        if (!map_is_location_owned(_loc))
        {
            res->Error = GameActions::Status::NoClearance;
            res->ErrorTitle = STR_CANT_REMOVE_THIS;
            res->ErrorMessage = STR_LAND_NOT_OWNED_BY_PARK;
            return res;
        }
    }

    TileElement* tileElement = FindSceneryElement();
    if (tileElement == nullptr)
    {
        return MakeResult(GameActions::Status::InvalidParameters, STR_CANT_REMOVE_THIS, STR_INVALID_SELECTION_OF_OBJECTS);
    }

    return res;
}

GameActions::Result::Ptr SmallSceneryRemoveAction::Execute() const
{
    GameActions::Result::Ptr res = std::make_unique<GameActions::Result>();

    auto* entry = get_small_scenery_entry(_sceneryType);
    if (entry == nullptr)
    {
        return MakeResult(GameActions::Status::InvalidParameters, STR_INVALID_SELECTION_OF_OBJECTS);
    }

    res->Cost = entry->removal_price * 10;
    res->Expenditure = ExpenditureType::Landscaping;
    res->Position = _loc;

    TileElement* tileElement = FindSceneryElement();
    if (tileElement == nullptr)
    {
        return MakeResult(GameActions::Status::InvalidParameters, STR_CANT_REMOVE_THIS, STR_INVALID_SELECTION_OF_OBJECTS);
    }

    res->Position.z = tile_element_height(res->Position);

    map_invalidate_tile_full(_loc);
    tile_element_remove(tileElement);

    return res;
}

TileElement* SmallSceneryRemoveAction::FindSceneryElement() const
{
    const bool isGhost = GetFlags() & GAME_COMMAND_FLAG_GHOST;
    for (auto* sceneryElement : TileElementsView<SmallSceneryElement>(_loc))
    {
        // If we are removing ghost elements
        if (isGhost && sceneryElement->IsGhost() == false)
            continue;

        if (sceneryElement->GetSceneryQuadrant() != _quadrant)
            continue;

        if (sceneryElement->GetBaseZ() != _loc.z)
            continue;

        if (sceneryElement->GetEntryIndex() != _sceneryType)
            continue;

        return sceneryElement->as<TileElement>();
    }
    return nullptr;
}
