/*
 * Copyright 2002-2007 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.web.struts;

import java.io.IOException;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.struts.action.Action;
import org.apache.struts.action.ActionMapping;
import org.apache.struts.action.ActionServlet;
import org.apache.struts.action.RequestProcessor;
import org.apache.struts.config.ModuleConfig;

import org.springframework.beans.BeansException;
import org.springframework.web.context.WebApplicationContext;

/**
 * Subclass of Struts's default {@link RequestProcessor} that looks up
 * Spring-managed Struts {@link Action Actions} defined in
 * {@link ContextLoaderPlugIn ContextLoaderPlugIn's} {@link WebApplicationContext}
 * (or, as a fallback, in the root <code>WebApplicationContext</code>).
 *
 * <p>In the Struts config file, you can either specify the original
 * <code>Action</code> class (as when generated by XDoclet), or no
 * <code>Action</code> class at all. In any case, Struts will delegate to an
 * <code>Action</code> bean in the <code>ContextLoaderPlugIn</code> context.
 *
 * <pre class="code">&lt;action path="/login" type="myapp.MyAction"/&gt;</pre>
 *
 * or
 *
 * <pre class="code">&lt;action path="/login"/&gt;</pre>
 *
 * The name of the <code>Action</code> bean in the
 * <code>WebApplicationContext</code> will be determined from the mapping path
 * and module prefix. This can be customized by overriding the
 * {@link #determineActionBeanName} method.
 *
 * <p>Example:
 * <ul>
 * <li>mapping path "/login" -> bean name "/login"<br>
 * <li>mapping path "/login", module prefix "/mymodule" ->
 * bean name "/mymodule/login"
 * </ul>
 *
 * <p>A corresponding bean definition in the <code>ContextLoaderPlugin</code>
 * context would look as follows; notice that the <code>Action</code> is now
 * able to leverage fully Spring's configuration facilities:
 *
 * <pre class="code">
 * &lt;bean name="/login" class="myapp.MyAction"&gt;
 *   &lt;property name="..."&gt;...&lt;/property&gt;
 * &lt;/bean&gt;</pre>
 *
 * Note that you can use a single <code>ContextLoaderPlugIn</code> for all
 * Struts modules. That context can in turn be loaded from multiple XML files,
 * for example split according to Struts modules. Alternatively, define one
 * <code>ContextLoaderPlugIn</code> per Struts module, specifying appropriate
 * "contextConfigLocation" parameters. In both cases, the Spring bean name has
 * to include the module prefix.
 *
 * <p>If you also need the Tiles setup functionality of the original
 * <code>TilesRequestProcessor</code>, use
 * <code>DelegatingTilesRequestProcessor</code>. As there is just a
 * single central class to customize in Struts, we have to provide another
 * subclass here, covering both the Tiles and the Spring delegation aspect.
 *
 * <p>If this <code>RequestProcessor</code> conflicts with the need for a
 * different <code>RequestProcessor</code> subclass (other than
 * <code>TilesRequestProcessor</code>), consider using
 * {@link DelegatingActionProxy} as the Struts <code>Action</code> type in
 * your struts-config file.
 *
 * <p>The default implementation delegates to the
 * <code>DelegatingActionUtils</code> class as much as possible, to reuse as
 * much code as possible despite the need to provide two
 * <code>RequestProcessor</code> subclasses. If you need to subclass yet
 * another <code>RequestProcessor</code>, take this class as a template,
 * delegating to <code>DelegatingActionUtils</code> just like it.
 *
 * @author Juergen Hoeller
 * @since 1.0.2
 * @see #determineActionBeanName
 * @see DelegatingTilesRequestProcessor
 * @see DelegatingActionProxy
 * @see DelegatingActionUtils
 * @see ContextLoaderPlugIn
 * @deprecated as of Spring 3.0
 */
@Deprecated
public class DelegatingRequestProcessor extends RequestProcessor {

	private WebApplicationContext webApplicationContext;
	

	@Override
	public void init(ActionServlet actionServlet, ModuleConfig moduleConfig) throws ServletException {
		super.init(actionServlet, moduleConfig);
		if (actionServlet != null) {
			this.webApplicationContext = initWebApplicationContext(actionServlet, moduleConfig);
		}
	}

	/**
	 * Fetch ContextLoaderPlugIn's {@link WebApplicationContext} from the
	 * <code>ServletContext</code>, falling back to the root
	 * <code>WebApplicationContext</code>.
	 * <p>This context is supposed to contain the Struts <code>Action</code>
	 * beans to delegate to.
	 * @param actionServlet the associated <code>ActionServlet</code>
	 * @param moduleConfig the associated <code>ModuleConfig</code>
	 * @return the <code>WebApplicationContext</code>
	 * @throws IllegalStateException if no <code>WebApplicationContext</code> could be found
	 * @see DelegatingActionUtils#findRequiredWebApplicationContext
	 * @see ContextLoaderPlugIn#SERVLET_CONTEXT_PREFIX
	 */
	protected WebApplicationContext initWebApplicationContext(
			ActionServlet actionServlet, ModuleConfig moduleConfig) throws IllegalStateException {

		return DelegatingActionUtils.findRequiredWebApplicationContext(actionServlet, moduleConfig);
	}

	/**
	 * Return the <code>WebApplicationContext</code> that this processor
	 * delegates to.
	 */
	protected final WebApplicationContext getWebApplicationContext() {
		return this.webApplicationContext;
	}


	/**
	 * Override the base class method to return the delegate action.
	 * @see #getDelegateAction
	 */
	@Override
	protected Action processActionCreate(
			HttpServletRequest request, HttpServletResponse response, ActionMapping mapping)
			throws IOException {

		Action action = getDelegateAction(mapping);
		if (action != null) {
			return action;
		}
		return super.processActionCreate(request, response, mapping);
	}

	/**
	 * Return the delegate <code>Action</code> for the given mapping.
	 * <p>The default implementation determines a bean name from the
	 * given <code>ActionMapping</code> and looks up the corresponding
	 * bean in the <code>WebApplicationContext</code>.
	 * @param mapping the Struts <code>ActionMapping</code>
	 * @return the delegate <code>Action</code>, or <code>null</code> if none found
	 * @throws BeansException if thrown by <code>WebApplicationContext</code> methods
	 * @see #determineActionBeanName
	 */
	protected Action getDelegateAction(ActionMapping mapping) throws BeansException {
		String beanName = determineActionBeanName(mapping);
		if (!getWebApplicationContext().containsBean(beanName)) {
			return null;
		}
		return (Action) getWebApplicationContext().getBean(beanName, Action.class);
	}

	/**
	 * Determine the name of the <code>Action</code> bean, to be looked up in
	 * the <code>WebApplicationContext</code>.
	 * <p>The default implementation takes the
	 * {@link org.apache.struts.action.ActionMapping#getPath mapping path} and
	 * prepends the
	 * {@link org.apache.struts.config.ModuleConfig#getPrefix module prefix},
	 * if any.
	 * @param mapping the Struts <code>ActionMapping</code>
	 * @return the name of the Action bean
	 * @see DelegatingActionUtils#determineActionBeanName
	 * @see org.apache.struts.action.ActionMapping#getPath
	 * @see org.apache.struts.config.ModuleConfig#getPrefix
	 */
	protected String determineActionBeanName(ActionMapping mapping) {
		return DelegatingActionUtils.determineActionBeanName(mapping);
	}

}
