/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.config.java.annotation.valuesource;

import static java.lang.String.format;
import static org.hamcrest.CoreMatchers.*;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;

import org.apache.commons.lang.exception.ExceptionUtils;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.springframework.beans.factory.BeanCreationException;
import org.springframework.config.java.annotation.Bean;
import org.springframework.config.java.annotation.Configuration;
import org.springframework.config.java.annotation.ExternalValue;
import org.springframework.config.java.context.JavaConfigApplicationContext;
import org.springframework.config.java.valuesource.SystemPropertiesValueResolver;
import org.springframework.config.java.valuesource.ValueResolutionException;


/**
 * System tests for {@link SystemPropertiesValueSource}.
 */
public class SystemPropertiesValueSourceTests {

    private static final String NONEXISTENT_PROPERTY_NAME = "invalid.property";
    private static final String EXISTENT_PROPERTY_NAME = "valid.property";
    private static final String EXISTENT_PROPERTY_VALUE = "value";

    private JavaConfigApplicationContext ctx;

    static class ValueHolder {
        public String value;
    }
    
    @BeforeClass
    public static void initSystemProperties() {
        System.setProperty(EXISTENT_PROPERTY_NAME, EXISTENT_PROPERTY_VALUE);
    }

    @Before
    public void initContext() {
        ctx = null;
    }

    // -------------------------------------------------------------------------

    @Test
    public void testNonExistentSystemProperty() {
        try {
            ctx = new JavaConfigApplicationContext(ConfigWithNonExistentSystemProperty.class);
            fail("should have thrown " + ValueResolutionException.class.getName());
        } catch (BeanCreationException ex) {
            Throwable rootCause = ExceptionUtils.getRootCause(ex);
            assertThat(rootCause, instanceOf(ValueResolutionException.class));
            assertThat(rootCause.getMessage(),
                    equalTo(format("Cannot resolve property for name '%s' against resolvers [%s]",
                            NONEXISTENT_PROPERTY_NAME, SystemPropertiesValueResolver.class.getSimpleName())));
        }

    }

    @Configuration
    @SystemPropertiesValueSource
    static abstract class ConfigWithNonExistentSystemProperty {
        abstract @ExternalValue(NONEXISTENT_PROPERTY_NAME) String value();

        public @Bean ValueHolder valueHolder() {
            ValueHolder valueHolder = new ValueHolder();
            valueHolder.value = value();
            return valueHolder;
        }
    }

    // -------------------------------------------------------------------------

    @Test
    public void testExistentSystemProperty() {
        ctx = new JavaConfigApplicationContext(ConfigWithExistentSystemProperty.class);
        assertThat(ctx.getBean(ValueHolder.class).value, not(nullValue()));
        assertThat(ctx.getBean(ValueHolder.class).value, equalTo(EXISTENT_PROPERTY_VALUE));
    }

    @Configuration
    @SystemPropertiesValueSource
    static abstract class ConfigWithExistentSystemProperty {
        abstract @ExternalValue(EXISTENT_PROPERTY_NAME) String value();

        public @Bean ValueHolder valueHolder() {
            ValueHolder valueHolder = new ValueHolder();
            valueHolder.value = value();
            return valueHolder;
        }
    }

}
