/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package test.feature.lifecycle.bfpp;

import static org.springframework.config.java.util.BeanMetadata.COPY_HIDDEN;

import org.junit.Assert;
import org.junit.Test;

import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanFactoryPostProcessor;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;

import org.springframework.config.java.annotation.Bean;
import org.springframework.config.java.annotation.Meta;
import org.springframework.config.java.context.JavaConfigApplicationContext;

import org.springframework.context.support.ClassPathXmlApplicationContext;

import test.feature.lifecycle.bpp.HiddenBeanPostProcessorTests.HasHiddenBeansNeedingPostProcessing;

import java.util.HashSet;
import java.util.Set;


/**
 * System tests covering {@link BeanFactoryPostProcessor} lifecycle support.
 *
 * @author  Chris Beams
 */
public class BeanFactoryPostProcessorPropagationTests {

    @Test
    public void testProgrammaticallyRegisteredBeanFactoryPostProcessorAppliesToHiddenBeans() {
        RememberingBeanFactoryPostProcessor rememberingFactoryPostProcessor = new RememberingBeanFactoryPostProcessor();

        JavaConfigApplicationContext ctx = new JavaConfigApplicationContext();
        ctx.addBeanFactoryPostProcessor(rememberingFactoryPostProcessor);
        ctx.addConfigClass(HasHiddenBeansNeedingPostProcessing.class);
        ctx.refresh();

        validateBeanFactoryPostProcessor(rememberingFactoryPostProcessor);
    }

    @Test
    public void testJavaBeanDefinitionRegisteredBeanFactoryPostProcessorAppliesToHiddenBeans() {
        JavaConfigApplicationContext ctx =
            new JavaConfigApplicationContext(HasHiddenBeansNeedingPostProcessing_WithDeclarativeBFPP.class);

        RememberingBeanFactoryPostProcessor rememberingFactoryPostProcessor =
            (RememberingBeanFactoryPostProcessor) ctx.getBean("bfpp1");
        validateBeanFactoryPostProcessor(rememberingFactoryPostProcessor);

        rememberingFactoryPostProcessor = (RememberingBeanFactoryPostProcessor) ctx.getBean("bfpp2");
        validateBeanFactoryPostProcessor(rememberingFactoryPostProcessor);
    }

    @Test
    public void testXmlBeanDefinitionRegisteredBeanFactoryPostProcessorAppliesToHiddenBeans() {
        ClassPathXmlApplicationContext cac =
            new ClassPathXmlApplicationContext("BeanFactoryPostProcessorPropagationTests.xml", this.getClass());

        RememberingBeanFactoryPostProcessor rememberingFactoryPostProcessor =
            (RememberingBeanFactoryPostProcessor) cac.getBean("bfpp1");
        validateBeanFactoryPostProcessor(rememberingFactoryPostProcessor);

        rememberingFactoryPostProcessor = (RememberingBeanFactoryPostProcessor) cac.getBean("bfpp2");
        validateBeanFactoryPostProcessor(rememberingFactoryPostProcessor);
    }

    private void validateBeanFactoryPostProcessor(RememberingBeanFactoryPostProcessor rememberingFactoryPostProcessor) {
        Assert.assertTrue("Must have recorded public bean method",
                          rememberingFactoryPostProcessor.beansSeen.contains("publicPoint"));
        Assert.assertTrue("Must have recorded package bean method",
                          rememberingFactoryPostProcessor.beansSeen.contains("packagePoint"));
        Assert.assertTrue("Must have recorded protected bean method",
                          rememberingFactoryPostProcessor.beansSeen.contains("protectedPoint"));
    }

    static class HasHiddenBeansNeedingPostProcessing_WithDeclarativeBFPP extends HasHiddenBeansNeedingPostProcessing {
        @Bean(meta = @Meta(key = COPY_HIDDEN, value = "true"))
        public RememberingBeanFactoryPostProcessor bfpp1() { return new RememberingBeanFactoryPostProcessor(); }

        @Bean(meta = @Meta(key = COPY_HIDDEN, value = "true"))
        public RememberingBeanFactoryPostProcessor bfpp2() { return new RememberingBeanFactoryPostProcessor(); }
    }

    static class RememberingBeanFactoryPostProcessor implements BeanFactoryPostProcessor {
        private Set<String> beansSeen = new HashSet<String>();

        public void postProcessBeanFactory(ConfigurableListableBeanFactory clbf) throws BeansException {
            for (String beanName : clbf.getBeanDefinitionNames()) {
                if (beansSeen.contains(beanName)) {
                    throw new IllegalStateException("Cannot process a bean twice");
                }

                beansSeen.add(beanName);
            }
        }
    }

}
