/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package test.feature.xmlbootstrap;

import static org.junit.Assert.*;

import java.awt.Point;

import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.junit.Test;
import org.springframework.aop.Advisor;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.config.java.annotation.Bean;
import org.springframework.config.java.annotation.Configuration;
import org.springframework.config.java.annotation.ExternalBean;
import org.springframework.config.java.context.JavaConfigApplicationContext;
import org.springframework.config.java.plugin.aop.AspectJAutoProxy;
import org.springframework.config.java.process.ConfigurationPostProcessor;
import org.springframework.context.support.ClassPathXmlApplicationContext;

import test.common.beans.DependsOnTestBean;
import test.common.beans.IOther;
import test.common.beans.ITestBean;
import test.common.beans.TestBean;
import test.common.config.BaseConfiguration;


/**
 * Various system tests relating to bootstrapping JavaConfig via XML (using
 * {@link ConfigurationPostProcessor}). Tests a broad set of functionality, including aspects,
 * factory beans, etc.
 *
 * @author  Rod Johnson
 * @author  Chris Beams
 * @see     BootstrapJavaConfigViaXmlTests
 */
public class ConfigurationPostProcessorTests {

    @Test
    public void testPriorityOrdering() {
        ConfigurationPostProcessor cpp = new ConfigurationPostProcessor();
        assertEquals(Integer.MIN_VALUE, cpp.getOrder());
    }

    @Test
    public void testSingleton() {
        //ClassPathXmlApplicationContext bf = new ClassPathXmlApplicationContext("test.xml", getClass());
        JavaConfigApplicationContext bf = new JavaConfigApplicationContext(AdvisedBaseConfiguration.class);

        ITestBean tb = (ITestBean) bf.getBean("tom");
        assertEquals("tom", tb.getName());
        assertEquals("becky", tb.getSpouse().getName());

        ITestBean tb2 = (ITestBean) bf.getBean("tom");
        assertSame(tb, tb2);
    }

    @Test
    public void testInjectedConfig() {
        ClassPathXmlApplicationContext bf = new ClassPathXmlApplicationContext("injectedTest.xml", getClass());

        ITestBean tb = (ITestBean) bf.getBean("testBean");
        assertEquals("testBean was not injected", 33, tb.getAge());
    }

    /** Test that a bean defined in XML can be injected with a bean from a config. */
    @Test
    public void testReverseInjectionWithAutowire() {
        ClassPathXmlApplicationContext bf = new ClassPathXmlApplicationContext("reverseInjection.xml", getClass());

        DependsOnTestBean tb = (DependsOnTestBean) bf.getBean("dependsOnTestBean");
        assertNotNull(tb.tb);
        assertEquals(33, tb.tb.getAge());
    }

    @Test
    public void testReverseInjectionExplicit() {
        ClassPathXmlApplicationContext bf = new ClassPathXmlApplicationContext("reverseInjection.xml", getClass());

        DependsOnTestBean tb = (DependsOnTestBean) bf.getBean("dependsOnTestBeanExplicit");
        assertNotNull(tb.tb);
        assertEquals(33, tb.tb.getAge());
    }

    @Test
    public void testProxiedPrototype() {
        ClassPathXmlApplicationContext bf = new ClassPathXmlApplicationContext("test.xml", getClass());

        ITestBean tb = (ITestBean) bf.getBean("prototype");
        assertEquals("prototype", tb.getName());
        assertEquals("becky", tb.getSpouse().getName());

        ITestBean tb2 = (ITestBean) bf.getBean("prototype");
        assertEquals("prototype", tb.getName());
        assertEquals("becky", tb.getSpouse().getName());

        assertNotSame(tb, tb2);
    }

    @Test
    public void testNonProxiedPrototype() {
        ClassPathXmlApplicationContext bf = new ClassPathXmlApplicationContext("test.xml", getClass());

        Point tb = (Point) bf.getBean("prototypePoint");
        // assertEquals(3, tb.); assertEquals("becky", tb.getSpouse().getName());

        Point tb2 = (Point) bf.getBean("prototypePoint");
        // assertEquals("prototype", tb.getName()); assertEquals("becky", tb.getSpouse().getName());

        assertNotSame(tb, tb2);

        // ITestBean tomsBecky = tb.getSpouse();
        // ITestBean factorysBecky = (ITestBean) bf.getBean("becky");
        // assertSame(tomsBecky, factorysBecky);
    }

    @Test
    public void testPointcut() {
        ClassPathXmlApplicationContext bf = new ClassPathXmlApplicationContext("test.xml", getClass());

        IOther becky = (IOther) bf.getBean("becky");

        assertEquals("No advisors put in factory", 0, bf.getBeanNamesForType(Advisor.class).length);

        assertTrue("Becky bean should be proxied", AopUtils.isAopProxy(becky));

        try {
            // Should fire pointcut
            becky.absquatulate();
            fail();
        } catch (UnsupportedOperationException ex) {
            // OK
        }

        assertFalse("No pointcut in factory: method was protected (hidden)", bf.containsBean("debugAdvice"));
    }

    @Test
    public void testAbstractBeanDefinition() throws Exception {
        ClassPathXmlApplicationContext bf = new ClassPathXmlApplicationContext("abstractDef.xml", getClass());

        assertFalse("AbstractConfig @Beans should not have been processed", AbstractConfig.testBeanCreated);
        assertEquals("AbstractConfig should not be registrered as bean", 0,
                     bf.getBeanNamesForType(AbstractConfig.class).length);
    }

    @Test
    public void testFactoryBean() {
        ClassPathXmlApplicationContext bf = new ClassPathXmlApplicationContext("myfactory.xml", getClass());

        ITestBean tb = (ITestBean) bf.getBean("factoryCreatedTestBean");
        assertEquals("jenny", tb.getName());
    }

    @Test
    public void testExternalBean() {
        ClassPathXmlApplicationContext bf = new ClassPathXmlApplicationContext("externalBean.xml", getClass());
        TestBean bob = (TestBean) bf.getBean("bob");
        assertTrue(bf.containsBean("ann"));
        assertEquals("External bean must have been satisfied", "Ann", bob.getSpouse().getName());
    }

    @Test
    public void testExternalBeanNonAbstract() {
        ClassPathXmlApplicationContext bf = new ClassPathXmlApplicationContext("externalBean.xml", getClass());
        TestBean bob = (TestBean) bf.getBean("bob");
        assertTrue(bf.containsBean("ann"));
        assertEquals("External bean must have been satisfied", "Ann", bob.getSpouse().getName());
    }

    @Configuration
    static class AbstractConfig {

        public static boolean testBeanCreated = false;

        @Bean
        public TestBean testBean() {
            // this should never occur
            testBeanCreated = true;
            return new TestBean();
        }
    }

    @Configuration
    static class PlaceholderConfig {
        @Bean
        public TestBean placeholder() { return new TestBean(); }
    }

    @Configuration
    static class InjectedConfig {
        private int age;

        public void setAge(int age) { this.age = age; }

        @Bean
        public TestBean testBean() {
            TestBean tb = new TestBean();
            tb.setAge(age);
            return tb;
        }
    }
    
    @AspectJAutoProxy(proxyTargetClass=true)
    @Aspect
    @Configuration
    static class AdvisedBaseConfiguration extends BaseConfiguration {
        @Before("execution(* absquatulate())")
        protected void debugAdvice() { throw new UnsupportedOperationException(); }
    }

    @Configuration
    static class FactoryBeanConfig {
        @Bean
        public MyFactory factoryCreatedTestBean() {
            String myString = "jenny";
            MyFactory f = new MyFactory();
            f.setMyString(myString);
            return f;
        }
    }

    static class MyFactory implements FactoryBean {

        private String myString;

        public void setMyString(String myString) { this.myString = myString; }

        public String getMyString() { return myString; }

        /**
         * @see  org.springframework.config.java.context.java.beans.factory.FactoryBean#getObject()
         */
        public Object getObject() throws Exception {
            TestBean tb = new TestBean();
            tb.setName(myString);
            return tb;
        }

        /**
         * @see  org.springframework.config.java.context.java.beans.factory.FactoryBean#getObjectType()
         */
        public Class<?> getObjectType() { return TestBean.class; }

        /**
         * @see  org.springframework.config.java.context.java.beans.factory.FactoryBean#isSingleton()
         */
        public boolean isSingleton() { return true; }
    }

    @Configuration @Aspect
    static class Advised1 {
        @Bean
        public TestBean oldJohannes() { return new TestBean("johannes", 29); }

        @Around("execution(int *.getAge())")
        public Object age(ProceedingJoinPoint pjp) throws Throwable {
            int realAge = (Integer) pjp.proceed();
            return realAge * 2;
        }
    }

    @Configuration @Aspect
    static class Advised2 {
        @Bean
        public TestBean youngJohannes() { return new TestBean("johannes", 29); }

        @Around("execution(int *.getAge())")
        public Object age() throws Throwable { return 21; }
    }

    @Configuration @Aspect
    static class PostProcessedConfig {
        private int counter;

        @Bean
        public ITestBean adrian() {
            TestBean adrian = new TestBean();
            adrian.setName("adrian");
            return adrian;
        }

        @Bean
        public ITestBean testEligibleForAutoproxying() { return new TestBean(); }

        @Bean
        public AdvisedByConfig advisedByConfig() { return new AdvisedByConfig(); }

        @SuppressWarnings("unused")
        @Around("execution(int *.intValue())")
        private int returnCount(ProceedingJoinPoint pjp) { return counter++; }
    }

    @Aspect
    static class CountAspect {
        public static int counter;

        @SuppressWarnings("unused")
        @Around("execution(int *.intValue())")
        private int countIntValueInvocation(ProceedingJoinPoint pjp) throws Throwable {
            ++counter;
            return (Integer) pjp.proceed();
        }
    }

    static class AdvisedByConfig {
        public int intValue() { return 0; }
    }

    @Configuration
    public abstract static class ExternalBeanConfiguration {
        @Bean
        public TestBean bob() {
            TestBean bob = new TestBean();
            bob.setSpouse(ann());
            return bob;
        }

        // Will be taken from XML
        @ExternalBean
        public abstract TestBean ann();
    }

    @Configuration
    public static class ExternalBeanProvidingConfiguration {
        @Bean
        public TestBean ann() { return new TestBean(); }
    }

    @Configuration
    static class ExternalBeanConfigurationNonAbstract extends ExternalBeanConfiguration {

        @Override @ExternalBean
        public TestBean ann() { throw new UnsupportedOperationException("should not be called"); }
    }

}
