/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.config.java.aspect;

import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;

import org.springframework.beans.factory.annotation.Required;


/**
 * Aspect that weaves tracking logic into methods annotated with Spring's {@link Required @Required}
 * annotation. Requires AspectJ bytecode (load-time or build-time) weaving to function.
 *
 * @author  Chris Beams
 * @see     org.springframework.config.java.annotation.Configuration#checkRequired()
 * @see     META-INF/aop.xml
 */
@Aspect
public class RequiredMethodInvocationTracker {

    private static final RequiredMethodInvocationRegistry invocationRegistry = new RequiredMethodInvocationRegistry();

    @Before("execution(@org.springframework.beans.factory.annotation.Required * *(..))")
    public void logRequiredMethodInvocation(JoinPoint jp) {
        Signature signature = jp.getSignature();
        getInvocationRegistry().registerMethodInvocation(jp.getTarget(), signature.getDeclaringTypeName(),
                                                         signature.getName());
    }

    public static RequiredMethodInvocationRegistry getInvocationRegistry() {
        return invocationRegistry;
    }

}
