/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.myfaces.orchestra;

import org.apache.myfaces.orchestra.conversation.ConversationContextFactory;
import org.apache.myfaces.orchestra.conversation.ConversationManagerFactory;
import org.apache.myfaces.orchestra.conversation.ConversationManagerFactoryImpl;
import org.apache.myfaces.orchestra.frameworkAdapter.FrameworkAdapter;

import org.apache.myfaces.orchestra.conversation.ConversationContextFactoryImpl;

/**
 * A factory of factory classes, just like the JSF javax.faces.FactoryFinder class.
 * <p>
 * This class ensures that the user can configure their own instances of the critical
 * orchestra factory classes by defining beans with appropriate names in the dependency
 * injection framework. Orchestra code should always use the factory-finder methods
 * here when creating object instances, instead of the "new" operator.
 * <p>
 * The factory classes returned here should all be thread-safe; they are effectively
 * "application scoped" objects.
 */
public class FactoryFinder
{
    // The key of an optional bean that the user can define in the DI framework to override
    // the settings of the ConversationManagerFactory used by Orchestra.
    public static final String CONVERSATION_MANAGER_FACTORY_KEY
        = ConversationManagerFactory.class.getName();

    // The key of an optional bean that the user can define in the DI framework to override
    // the settings of the ConversationContextFactory used by Orchestra.
    public static final String CONVERSATION_CONTEXT_FACTORY_KEY
        = ConversationContextFactory.class.getName();

    private static final ConversationManagerFactory CONVERSATION_MANAGER_FACTORY_DFLT
        = new ConversationManagerFactoryImpl();

    private static final ConversationContextFactory CONVERSATION_CONTEXT_FACTORY_DFLT
        = new ConversationContextFactoryImpl();

    /**
     * Return an object that implements interface ConversationManagerFactory.
     * <p>
     * Normally, the returned object will be an instance of ConversationManagerFactoryImpl.
     */
    public static ConversationManagerFactory getConversationManagerFactory()
    {
        FrameworkAdapter fa = FrameworkAdapter.getCurrentInstance();
        ConversationManagerFactory factory = (ConversationManagerFactory) fa.getBean(
                CONVERSATION_MANAGER_FACTORY_KEY);

        if (factory == null)
        {
            factory = CONVERSATION_MANAGER_FACTORY_DFLT;
        }

        return factory;
    }

    /**
     * Return an object that implements interface ConversationContextFactory.
     * <p>
     * Normally, the returned object will be an instance of ConversationContextFactoryImpl.
     * However users can configure a bean in the DI framework to specify a different class - or to
     * return an instance of ConversationContextFactoryImpl with a custom timeout value configured.
     */
    public static ConversationContextFactory getConversationContextFactory()
    {
        FrameworkAdapter fa = FrameworkAdapter.getCurrentInstance();
        ConversationContextFactory factory = (ConversationContextFactory) fa.getBean(
                CONVERSATION_CONTEXT_FACTORY_KEY);

        if (factory == null)
        {
            factory = CONVERSATION_CONTEXT_FACTORY_DFLT;
        }

        return factory;
    }
}
