/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.myfaces.orchestra.viewController;

import org.apache.myfaces.orchestra.lib.OrchestraException;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

/**
 * Invokes ViewController events using reflection.
 * <p>
 * If the target bean has a method with any of the following signatures then it
 * receives the appropriate callback:
 * <ul>
 * <li>public void initView()</li>
 * <li>public void preProcess()</li>
 * <li>public void preRenderView()</li>
 * </ul>
 * <p>
 * Note that each method here returns false if the target bean does not
 * have a method with an appropriate signature; this allows this executor to 
 * be "chained" with others.
 */
public class ReflectiveViewControllerExecutor extends AbstractViewControllerExecutor
{
    /**
     * Helper method to find the method which should get invoked.
     */
    protected boolean invokeOnViewController(Object bean, String methodName)
    {
        try
        {
            Method method = bean.getClass().getMethod(methodName, (Class[]) null); // NON-NLS
            method.invoke(bean, (Object[]) null);
        }
        catch (NoSuchMethodException e)
        {
            return false;
        }
        catch (IllegalAccessException e)
        {
            throw new OrchestraException(e);
        }
        catch (InvocationTargetException e)
        {
            throw new OrchestraException(e);
        }

        return true;
    }

    /* the ViewControllerExecutor interface */ 
    public boolean invokeInitView(String beanName, Object bean)
    {
        return invokeOnViewController(bean, "initView");
    }

    public boolean invokePreRenderView(String beanName, Object bean)
    {
        return invokeOnViewController(bean, "preRenderView");
    }

    public boolean invokePreProcess(String beanName, Object bean)
    {
        return invokeOnViewController(bean, "preProcess");
    }
}
